<?php
/**
 * RdbObject and its Manager
 * class for manupulating data stored in relational database server.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * @author Haruki Setoyama <haruki@planewave.org> 
 * @copyright Copyright &copy; 2003, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @package rdbObject
 * @since 2003.06
 * @version 0.2 $Id: rdbobject.php,v 1.10 2003/11/20 05:54:27 haruki Exp $
 * @link http://www.planewave.org/rdbobject/
 */
 
/**
 * define
 */
if (! defined('RDBOBJECT_DIR'))
    define('RDBOBJECT_DIR', dirname(__FILE__));
define('RDBOBJECT_VALUE_VALID', true);
define('RDBOBJECT_VALUE_BELOW_MIN', -1);
define('RDBOBJECT_VALUE_BEYOND_MAX', -2);
define('RDBOBJECT_VALUE_NULL_NOT_ALLOWED', -5);
define('RDBOBJECT_VALUE_INVALID_DATATYPE', -3);

define('RDBOBJECT_WARNING', 1);
define('RDBOBJECT_ERROR_INTERNAL', 2);
define('RDBOBJECT_ERROR_CONFIG', 4);
define('RDBOBJECT_ERROR_ARGUMENT', 8);
/**
 * define type code
 */
define('RDBOBJECT_ELEMENT_NULL', 0);
define('RDBOBJECT_ELEMENT_VIRTUAL', 1);
define('RDBOBJECT_ELEMENT_STRING', 2);
define('RDBOBJECT_ELEMENT_NUMERIC', 4);
define('RDBOBJECT_ELEMENT_BINARY', 8);

/**
 * require
 */
require_once 'PEAR.php';
require_once RDBOBJECT_DIR.'/rdbobjectmanager.php';
require_once 'singleton.php';

/**
 * rdbObject
 * 
 * @abstract 
 */
class rdbObject extends PEAR {
    /**
     * array data form databese with *_fetch_assoc()
     * 
     * @access private only manager, element and $this can access
     */
    var $_dbRawData = array();

    /**
     * name of the manager class that manages this object.
     * 
     * @access private only manager and $this can access 
     */
    var $_manager;

    /**
     * column name
     * 
     * @access private only manager and $this can access
     */
    var $_column = array();

    /**
     * type of varibales
     * 
     * @access private only manager and $this can access
     */
	var $_type = array();

    /**
     * true when the element has been changed.
     * 
     * @access private only manager and $this can access
     */
    var $_change = array();

    /**
     * public data handle object for each data type.
     * 
     * @access private only manager and $this can access
     */
    var $_element;

    /**
     * element name of primary key.
     * 
     * @access private only manager and $this can access
     */
    var $_pkey;

    /**
     * initialize variables of this rdbObject
     * This function must be called in constructor.
     * 
     * @access private 
     * @return void 
     */
    function _init($configs)
    {
        foreach ($configs as $name => $config) {
            $classname = 'rdbObjectElement_' . $config['type'];
            if (!class_exists($classname)) {
                $this->raiseError("Element type '${config['type']}' configured in ".get_class($this).'is invalid.' , RDBOBJECT_ERROR_CONFIG);
                continue;
            } 

            $this->_element[$name] = new $classname;
            $this->_element[$name]->name = $name;
			$this->_type[$name] = $this->_element[$name]->_type();
            if (! ($this->_type[$name] & RDBOBJECT_ELEMENT_VIRTUAL)) {
                $this->_column[$name] = $name;
            } 
			
            if (isset($config['option'])) {
                $this->_element[$name]->option = $config['option'];
			}
            if ($config['type'] == 'pkey') {
                $this->_pkey = $name;
			}
        } 
		
		if (! isset($this->_pkey)) {
		    $this->raiseError("Primary key is not defined in ".get_class($this) , RDBOBJECT_ERROR_CONFIG);
		}
		
		$this->_blank();
    } 

    /**
     * make this rdbObject blank with default-option
     * 
     * @access private only manager can access
     * @return void 
     */
    function _blank()
    {
        foreach($this->_column as $name => $column) {
            if (isset($this->_element[$name]->option['default'])) {
                $this->_dbRawData[$column] = $this->_element[$name]->option['default'];
            } elseif (! empty($this->_element[$name]->option['null'])) {
                $this->_dbRawData[$column] = null;
				$this->_element[$name]->option['default'] = null;
            } elseif ($this->_type[$name] & RDBOBJECT_ELEMENT_NUMERIC) {
                $this->_dbRawData[$column] = 0;
				$this->_element[$name]->option['default'] = 0;
            } else {
                $this->_dbRawData[$column] = '';
				$this->_element[$name]->option['default'] = '';
            } 
        } 
    } 

    /**
     * _pre_save()
     * this method is called before saving
     * 
     * @access private only manager can access
     * @return void 
     */
    function _pre_save($varnames)
    {
        $err = array();
        foreach ($varnames as $varname) {
            $this->_element[$varname]->preSave($this);
        } 
    } 

	function _to_query_string($varname, $value)
	{
		if ($this->_element[$varname]->isValidType($value) !== true) {
			return false;		
		}	
		return $this->_element[$varname]->toString($value);	
	} 
	
    /**
     * clone myself.
     * But pkey will be zero, that means 'not saved' state.
     * 
     * @access public 
     * @return rdbObject 
     */
    function &clone()
    {
        $cloned = $this;
        $cloned->set($this->_pkey, 0);
        return $cloned;
    } 

    /**
     * get value of data
     * 
     * @param string $varname 
     * @access public 
     * @return mixed 
     */
    function get($varname)
    {
		if (! @isset($this->_column[$varname])) {
            $this->_error_variable_not_defined($varname, 'get');
			return null;
		}

        return $this->_element[$varname]->toMixed($this->_dbRawData[$this->_column[$varname]]); 
    } 

    /**
     * get options of the variable
     * 
     * @param string $varname 
     * @access public 
     * @return array 
     */
    function getOption($varname)
    {
		if (! @isset($this->_element[$varname])) {
            $this->_error_variable_not_defined($varname, 'getOption');
			return array();
		}
        
        return $this->_element[$varname]->option;
    } 

    /**
     * set value of data
     * 
     * @param string $varname 
     * @param mixed $value 
     * @access public 
     * @return true |int|PEARError  int is error code
     */
    function set($varname, $value)
    {
		if (! @isset($this->_column[$varname])) {
            return $this->_error_variable_not_defined($varname, 'set');
		}
		$ret = $this->_element[$varname]->isValid($value);
        if ($ret !== true) return $ret;
        $this->_dbRawData[$this->_column[$varname]] = $this->_element[$varname]->toString($value);
        $this->change[$varname] = true;
        return true;
    }

    /**
     * get primary key
     * 
     * @access public 
     * @return int 
     */
    function pkey()
    {
		if (isset($this->_pkey)) {
        	return intval($this->get($this->_pkey));
		} else {
			return 0;
		}
    } 

    /**
     * get related object
     * 
     * @param string $varname 
     * @access public 
     * @return array array of rdbObject
     */
    function &getRelated($varname)
    {
        if (! @isset($this->_element[$varname])) {
            $this->_error_variable_not_defined($varname, 'getRelated');
			return array();
		}

        return $this->_element[$varname]->getRelated($this);
    } 

    /**
     * set related object
     * 
     * @param string $varname 
     * @param rdbObject $rdbObject 
     * @access public 
     * @return true |PEARError
     */
    function setRelated($varname, &$rdbObject)
    {
		if (! @isset($this->_element[$varname])) {
            return $this->_error_variable_not_defined($varname, 'setRelated');
		}
        
        $ret = $this->_element[$varname]->setRelated($rdbObject, $this);
        if ($ret !== true) return $ret;
        return true; 
    } 

    /**
     * create related object
     * 
     * @param string $varname 
     * @access public 
     * @return rdbObject|null
     */
    function &createRelated($varname)
    {
		if (! @isset($this->_element[$varname])) {
            $this->_error_variable_not_defined($varname, 'createRelated');
			return null;
		}
        
        return $this->_element[$varname]->createRelated($this);
    } 

    /**
     * delete all the related objects
     * 
     * @param string $varname 
     * @access public 
     * @return true |PEARError
     */
    function deleteRelated($varname)
    {
		if (! @isset($this->_element[$varname])) {
            return $this->_error_variable_not_defined($varname, 'deleteRelated');
		}
		
        return $this->_element[$varname]->deleteRelated($this);
    } 

    /**
     * save object to databse
     * 
     * @access public 
     * @return true |PEARError
     */
    function save()
    {
        $manager = &singleton::getInstance($this->_manager);
        return $manager->save($this);
    } 

    /**
     * delete object from databse
     * 
     * @access public 
     * @return true |PEARError
     */
    function delete()
    {
        $manager = &singleton::getInstance($this->_manager);
        return $manager->delete($this);
    } 

    /**
     * PEARError for 'Variable not exists'
     * 
     * @access private 
     * @return PEARError 
     */
    function _error_variable_not_defined($varname, $func)
    {
		$msg = '<b>Invalid Argument</b>: ';
		if (! is_string($varname)) {
		    $msg .= 'Variable name nust be string';
		} else {
        	$msg .= "Variable '$varname' not defined";
    	}
		$msg .= ' at <b>'.$func.'()</b>';
		if (function_exists('debug_backtrace')) {
			$bt = debug_backtrace();
		    $msg .= ' in <b>'.$bt[1]['file'].'</b>';
			$msg .= ' on line <b>'.$bt[1]['line'].'</b>'."\n";
		}
		return $this->raiseError($msg , RDBOBJECT_ERROR_ARGUMENT);
 
	}
} 

/**
 * rdbObjectElement
 * Base class to manipulate data with type specific way
 * 
 * @abstract 
 */
class rdbObjectElement {
    /**
     * array of options
     * 
     * @access public 
     */
    var $option = array();

    /**
     * variable name which indicates this element
     * 
     * @access public 
     */
    var $name;

    /**
     * changes string value to original mixed value 
     * 
     * @param string $value
	 * @return mixed 
     * @access public 
     */
    function toMixed($value)
    {
        return $value;
    } 

	/**
     * changes original mixed value to string for storing in Database  
     * 
     * @param mixed $value
	 * @return string 
     * @access public 
     */
	function toString($value)
	{
		return $value;
	}

	/**
     * checks if the value is valid to set
     * 
     * @param mixed $value 
     * @return bool
     * @access public 
     */
	function isValidType($value)
	{
		return true;
	}
	
    /**
     * checks if the value is valid to set
     * 
     * @param mixed $value 
     * @return true|int  see error code
     * @access public 
     */
    function isValid($value)
    {
        if ($value === null && empty($this->option['null'])) {
            return RDBOBJECT_VALUE_NULL_NOT_ALLOWED;
        }
		if ($this->isValidType($value) !== true) {
		    return RDBOBJECT_VALUE_INVALID_DATATYPE;
		} 
        if ( ($ret = $this->_check($value)) !== true) {
            return $ret;
        }
		return true;
		
    } 

    /**
     * check value
     * OVERRIDE this function in child classes
     * 
     * @param mixed $value 
     * @access private 
     * @return true |int_error_code
	 * @abstract
     */
    function _check($value)
    {
        return true;
    } 	

    /**
     * get related object
     * 
     * @param rdbObject $self 
     * @access public 
     * @return array array of rdbObject
     */
    function &getRelated(&$self)
    {
        return array();
    } 

    /**
     * set related object
     * 
     * @param rdbObject $self 
     * @access public 
     * @return true |int  int is error_code
     */
    function setRelated(&$self)
    {
        return RDBOBJECT_VALUE_INVALID_DATATYPE;
    } 

    /**
     * create related object
     * 
     * @param rdbObject $self 
     * @access public 
     * @return rdbObject 
     */
    function &createRelated(&$self)
    {
        return null;
    } 

    /**
     * delete related object
     * 
     * @param rdbObject $self 
     * @access public 
     * @return 
     */
    function deleteRelated(&$self)
    {
    } 

    /**
     * this function is called before save()
     * 
     * @param rdbObject $self 
     * @access public 
     * @return void 
     */
    function preSave(&$self)
    {
        return;
    } 
	
	/**
	 * returns type of element
	 * 
	 * @return int
	 * @access private
	 **/
	function _type()
	{
		return RDBOBJECT_ELEMENT_NULL;
	}
} 

/**
 * rdbObjectElement_text.
 * for normal text.
 * 
 * @access public 
 */
class rdbObjectElement_text extends rdbObjectElement {
	function isValidType($value)
	{
		if (is_string($value) || is_int($value) || is_float($value)) {
			return true;
		}
		return false;	
	}

    function _check($value)
    {
        $len = strlen((string) $value);
        if (isset($this->option['min']) && $len < $this->option['min']) {
            return RDBOBJECT_VALUE_BELOW_MIN;
        }
		if (isset($this->option['max']) && $len > $this->option['max']) {
            return RDBOBJECT_VALUE_BEYOND_MAX;
        } 
        return true; 
    }
	
	function _type()
	{
		return RDBOBJECT_ELEMENT_STRING;
	} 
} 

/**
 * rdbObjectElement_password.
 * the password will be convertd by MD5
 * 
 * @access public 
 */
class rdbObjectElement_password extends rdbObjectElement_text {	
	function toString($value)
	{
		return md5($value);
	}
} 

/**
 * rdbObjectElement_array.
 * this uses serialize() functon to save.
 * 
 * @access public 
 */
class rdbObjectElement_array extends rdbObjectElement {
	function isValidType($value)
	{
		return (is_array($value)) ? true : false;
	}
	
    function _check($value)
    {
        if (isset($this->option['max'])) {
			$serialized = serialize($value);
            if (strlen($serialized) > $this->option['max']) {
                return RDBOBJECT_VALUE_BEYOND_MAX;
            } 
        } 
        return true;
    } 

    function toMixed($value)
    {
        return ($value !== null) ? unserialize($value) : null;
    } 
	
	function toString($value)
	{
		return serialize($value);
	}
} 

/**
 * rdbObjectNumericElement.
 * child class of this handles only numeric data.
 * 
 * @abstract 
 */
class rdbObjectNumericElement extends rdbObjectElement {
	function _type()
	{
		return RDBOBJECT_ELEMENT_NUMERIC;
	}
} 

/**
 * rdbObjectElement_int.
 * integer.
 * 
 * @access public 
 */
class rdbObjectElement_int extends rdbObjectNumericElement {
	function isValidType($value)
	{
		if (!is_numeric($value)) {
		    return false;
		}
		if (intval($value) != $value) {
		    return false;
		}
		return true;
	}
	
    function _check($value)
    { 
        if (isset($this->option['min']) && $value < $this->option['min']) {
            return RDBOBJECT_VALUE_BELOW_MIN;
        } 
        if (isset($this->option['max']) && $value > $this->option['max']) {
            return RDBOBJECT_VALUE_BEYOND_MAX;
        } 
        return true;
    } 
} 

/**
 * rdbObjectElement_numeric.
 * general numeric type.
 * 
 * @access public 
 */
class rdbObjectElement_numeric extends rdbObjectNumericElement {
	function isValidType($value)
	{
		return (is_numeric($value)) ? true : false;
	}
	
    function _check($value)
    {
        if (isset($this->option['min']) && $value < $this->option['min']) {
            return RDBOBJECT_VALUE_BELOW_MIN;
        } 
        if (isset($this->option['max']) && $value > $this->option['max']) {
            return RDBOBJECT_VALUE_BEYOND_MAX;
        } 
        return true;
    }
	
	function toString($value)
	{
		if (isset($this->option['decimal'])) {
            $value = floatval(sprintf('%.' . intval($this->option['decimal']) . 'f', $value));
        }
		return $value;
	} 
} 

/**
 * rdbObjectElement_bool.
 * bool. 1 means true, 0 means false.
 * 
 * @access public 
 */
class rdbObjectElement_bool extends rdbObjectNumericElement {
    function toString($value)
    {
        return (bool)$value ? 1: 0;
    } 
	
	function toMixed($value)
	{
		return (bool)$value;
	}
} 

/**
 * rdbObjectElement_bool.
 * bool. 1 means true, 0 means false.
 * 
 * @access public 
 */
class rdbObjectElement_updatetime extends rdbObjectElement_int {
    function _check($value)
    {
        return RDBOBJECT_VALUE_INVALID_DATATYPE;
    } 

    function preSave(&$self)
    {
        $self->_dbRawData[$self->_column[$this->name]] = time();
        $self->change[$this->name] = true;
    } 
} 

/**
 * rdbObjectElement_pkey.
 * primary key. positive integer. 0 means 'not saved'.
 * 
 * @access public 
 */
class rdbObjectElement_pkey extends rdbObjectElement_int {
    function _check($value)
    {
        if ($value < 0) {
            return RDBOBJECT_VALUE_BELOW_MIN;
        } 
        return true;
    } 
} 

/**
 * rdbObjectElement_fkey.
 * foreign key.
 * option['related'] is the manager class name of a rdbobject
 * which this foreign key indicates.
 * 
 * @access public 
 */
class rdbObjectElement_fkey extends rdbObjectElement_int {
    function _check($value)
    { 
        if ($value < 0 || ($value == 0 && empty($this->option['zero']))) {
            return RDBOBJECT_VALUE_BELOW_MIN;
        } 
        return true;
    } 

    function getRelated(&$self)
    {
        $manager_related = &singleton::getInstance($this->option['related']);
        $key = $self->_dbRawData[$self->_column[$this->name]];
        $ret = $manager_related->open($key);
        if (is_a($ret, 'rdbObject') || $ret == null) {
            return array($ret);
        } else {
            return $ret;
		}
    } 

    function setRelated(&$obj, &$self)
    {
        if (!is_a($obj, 'rdbObject') || $obj->_manager != strtolower($this->option['related']))
            return RDBOBJECT_VALUE_INVALID_DATATYPE;

        return $self->set($this->name, $obj->pkey());
    } 

    function &createRelated(&$self)
    {
        $manager_related = &singleton::getInstance($this->option['related']);
        return $manager_related->create();
    } 
} 

/**
 * rdbObjectVirtualElement.
 * child class of this has no column in the database.
 * 
 * @abstract 
 */
class rdbObjectVirtualElement extends rdbObjectElement {
    function _check($value)
    {
        return RDBOBJECT_VALUE_INVALID_DATATYPE;
    } 
	
	function _type()
	{
		return RDBOBJECT_ELEMENT_VIRTUAL;
	}
} 

/**
 * rdbObjectElement_sub
 * 
 * @access public 
 */
class rdbObjectElement_1m extends rdbObjectVirtualElement {
    function setRelated(&$obj, &$self)
    {
        if (! is_a($obj, 'rdbObject') || $obj->_manager != strtolower($this->option['related'])) {
            return RDBOBJECT_VALUE_INVALID_DATATYPE;
        } 
        $ret = $obj->set($this->option['fkey'], $self->pkey());
        if ($ret !== true) return $ret;
        return $value->save();
    } 

    function getRelated(&$self)
    {
        $m_related = &singleton::getInstance($this->option['related']);
        return $m_related->openWith($this->option['fkey'], $self->pkey());
    } 

    function &createRelated(&$self)
    {
        $manager_related = &singleton::getInstance($this->option['related']);
        $obj = $manager_related->create();
        $obj->set($this->option['fkey'], $self->pkey());
        return $obj;
    } 

    function deleteRelated(&$self)
    {
        if ($this->option['delete'] === false) return true;

        $h_related = &singleton::getInstance($this->option['related']);
        $relateds = $h_related->openWith($this->option['fkey'], $self->pkey());
        if (empty($relateds)) return true;

        if (! empty($this->option['zero'])) {
            foreach ($relateds as $related) {
                $related->set($this->option['fkey'], 0);
                $related->save();
            } 
        } else {
            foreach ($relateds as $related) {
                $related->delete();
            } 
        } 
    } 
} 

/**
 * rdbObjectElement_assoc
 * 
 * @access public 
 */
class rdbObjectElement_mm extends rdbObjectVirtualElement {
    function setRelated(&$obj, &$self)
    {
        if (!is_a($obj, 'rdbObject') || $obj->_manager != strtolower($this->option['related']))
            return RDBOBJECT_VALUE_INVALID_DATATYPE;

        $m_relation = &singleton::getInstance($this->option['relation']);
        $comp = new sqlComparisons;
        $comp->add($this->option['fkey_self'], $self->pkey());
        $comp->add($this->option['fkey_related'], $obj->pkey());
        $ret = &$m_relation->openConditional($comp);
        if (!empty($ret)) return true;

        $relation = &$m_relation->create();
        $relation->set($this->option['fkey_self'], $self->pkey());
        $relation->set($this->option['fkey_related'], $obj->pkey());
        return $relation->save();
    } 

    function &createRelated(&$self)
    {
        $manager_related = &singleton::getInstance($this->option['related']);
        return $manager_related->create();
    } 

    function &getRelated(&$self)
    {
        $m_relation = &singleton::getInstance($this->option['relation']);
        $m_related = &singleton::getInstance($this->option['related']);

        return $m_related->openJoin($m_relation, $this->option['fkey_related'] , $this->option['fkey_self'], $self->pkey());
    } 

    function deleteRelated(&$self)
    {
        if ($this->option['delete'] === false) return true;

        $m_relation = &singleton::getInstance($this->option['relation']);
        $relations = &$m_relation->openWith($this->option['fkey_self'], $self->pkey());
        foreach ($relations as $relation) {
            $relation->delete();
        } 
    } 
} 

?>