#include <Python.h>
#include <senna.h>


static PyObject *SennaError;


static PyObject *
senna_sen_init(PyObject *self, PyObject *args)
{
  sen_rc rc;
  if (!PyArg_ParseTuple(args, "")) {
    return NULL;
  }
  rc = sen_init();
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_fin(PyObject *self, PyObject *args)
{
  sen_rc rc;
  if (!PyArg_ParseTuple(args, "")) {
    return NULL;
  }
  rc = sen_fin();
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_index_create(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_index *index;
  PyObject *index_obj;
  char *path = "";
  int key_size = 0;
  int flags = SEN_INDEX_NGRAM|SEN_INDEX_NORMALIZE;
  sen_encoding encoding = sen_enc_utf8;
  int initial_n_segments = 0;
  static char *kwlist[] = {"path", "key_size", "flags", "initial_n_segments", "encoding", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "s|iiii", kwlist,
                                   &path, &key_size, &flags, &initial_n_segments, &encoding)) {
    return NULL;
  }
  index = sen_index_create(path, key_size, flags, initial_n_segments, encoding);
  if (!index) {
    return Py_BuildValue("");
  }
  index_obj = PyCObject_FromVoidPtr((void*)index, NULL);
  return Py_BuildValue("O", index_obj);
}


static PyObject *
senna_sen_index_open(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_index *index;
  PyObject *index_obj;
  char *path = "";
  static char *kwlist[] = {"path", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "s", kwlist, &path)) {
    return NULL;
  }
  index = sen_index_open(path);
  if (!index) {
    return Py_BuildValue("");
  }
  index_obj = PyCObject_FromVoidPtr((void*)index, NULL);
  return Py_BuildValue("O", index_obj);
}


static PyObject *
senna_sen_index_close(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_index *index;
  PyObject *index_obj;
  static char *kwlist[] = {"index", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &index_obj)) {
    return NULL;
  }
  index = (sen_index*)PyCObject_AsVoidPtr(index_obj);
  rc = sen_index_close(index);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_index_remove(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  char *path = "";
  static char *kwlist[] = {"path", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "s", kwlist, &path)) {
    return NULL;
  }
  rc = sen_index_remove(path);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_index_rename(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  char *old_name = "";
  char *new_name = "";
  static char *kwlist[] = {"old_name", "new_name", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "ss", kwlist, &old_name, &new_name)) {
    return NULL;
  }
  rc = sen_index_rename(old_name, new_name);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_index_upd(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_index *index;
  PyObject *index_obj;
  char *key = "";
  char *oldvalue = "";
  char *newvalue = "";

  static char *kwlist[] = {"index", "key", "oldvalue", "newvalue", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "Osss", kwlist,
                                   &index_obj, &key, &oldvalue, &newvalue)) {
    return NULL;
  }
  index = (sen_index*)PyCObject_AsVoidPtr(index_obj);
  rc = sen_index_upd(index, key, oldvalue, newvalue);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_index_sel(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_index *index;
  PyObject *index_obj;
  sen_records *records;
  PyObject *records_obj;
  char *string = "";

  static char *kwlist[] = {"index", "string", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "Os", kwlist, &index_obj, &string)) {
    return NULL;
  }
  index = (sen_index*)PyCObject_AsVoidPtr(index_obj);
  records = sen_index_sel(index, string);
  if (!records) {
    return Py_BuildValue("");
  }
  records_obj = PyCObject_FromVoidPtr((void*)records, NULL);
  return Py_BuildValue("O", records_obj);
}


static PyObject *
senna_sen_records_next(PyObject *self, PyObject *args, PyObject *keywds)
{
  int keysize = 0;
  sen_records *records;
  PyObject *records_obj;
  static char *kwlist[] = {"records", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &records_obj)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  keysize = sen_records_next(records, NULL, 0, NULL);
  return Py_BuildValue("i", keysize);
}


static PyObject *
senna_sen_records_rewind(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_records *records;
  PyObject *records_obj;
  static char *kwlist[] = {"records", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &records_obj)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  rc = sen_records_rewind(records);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_records_curr_score(PyObject *self, PyObject *args, PyObject *keywds)
{
  int score;
  sen_records *records;
  PyObject *records_obj;
  static char *kwlist[] = {"records", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &records_obj)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  score = sen_records_curr_score(records);
  return Py_BuildValue("i", score);
}


static PyObject *
senna_sen_records_curr_key(PyObject *self, PyObject *args, PyObject *keywds)
{
  int keysize = 0;
  sen_records *records;
  PyObject *records_obj;
  char *keybuf;
  PyObject *keybuf_obj;
  int bufsize = SEN_SYM_MAX_KEY_SIZE;
  static char *kwlist[] = {"records", "bufsize", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|i", kwlist,
                                   &records_obj, &bufsize)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  keybuf = PyMem_Malloc(bufsize);
  keysize = sen_records_curr_key(records, keybuf, bufsize);
  if (keysize == 0) {
    keybuf_obj = Py_BuildValue("");
  } else {
    keybuf_obj = Py_BuildValue("s", keybuf);
  }
  PyMem_Free(keybuf);
  return keybuf_obj;
}


static PyObject *
senna_sen_records_nhits(PyObject *self, PyObject *args, PyObject *keywds)
{
  int nhits;
  sen_records *records;
  PyObject *records_obj;
  static char *kwlist[] = {"records", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &records_obj)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  nhits = sen_records_nhits(records);
  return Py_BuildValue("i", nhits);
}


static PyObject *
senna_sen_records_find(PyObject *self, PyObject *args, PyObject *keywds)
{
  int score;
  sen_records *records;
  char *key = "";
  PyObject *records_obj;
  static char *kwlist[] = {"records", "key", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "Os", kwlist, &records_obj, &key)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  score = sen_records_find(records, key);
  return Py_BuildValue("i", score);
}


static PyObject *
senna_sen_records_close(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_records *records;
  PyObject *records_obj;
  static char *kwlist[] = {"records", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &records_obj)) {
    return NULL;
  }
  records = (sen_records*)PyCObject_AsVoidPtr(records_obj);
  rc = sen_records_close(records);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_sym_open(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_sym *sym;
  PyObject *sym_obj;
  char *path = "";
  static char *kwlist[] = {"path", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "s", kwlist, &path)) {
    return NULL;
  }
  sym = sen_sym_open(path);
  if (!sym) {
    return Py_BuildValue("");
  }
  sym_obj = PyCObject_FromVoidPtr((void*)sym, NULL);
  return Py_BuildValue("O", sym_obj);
}


static PyObject *
senna_sen_sym_info(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_sym *sym;
  PyObject *sym_obj;
  int key_size = 0;
  unsigned flags = 0;
  sen_encoding encoding = sen_enc_default;
  unsigned nrecords = 0;
  unsigned file_size = 0;
  static char *kwlist[] = {"sym", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &sym_obj)) {
    return NULL;
  }
  sym = (sen_sym*)PyCObject_AsVoidPtr(sym_obj);
  rc = sen_sym_info(sym, &key_size, &flags, &encoding, &nrecords, &file_size);
  printf("%d\n", flags);
  if (rc != sen_success) {
    return NULL;
  } else {
    return Py_BuildValue("(iiiii)", key_size, flags, encoding, nrecords, file_size);
  }
}


static PyObject *
senna_sen_sym_close(PyObject *self, PyObject *args, PyObject *keywds)
{
  sen_rc rc;
  sen_sym *sym;
  PyObject *sym_obj;
  static char *kwlist[] = {"sym", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &sym_obj)) {
    return NULL;
  }
  sym = (sen_sym*)PyCObject_AsVoidPtr(sym_obj);
  rc = sen_sym_close(sym);
  return Py_BuildValue("i", rc);
}


static PyObject *
senna_sen_sym_size(PyObject *self, PyObject *args, PyObject *keywds)
{
  unsigned int size;
  sen_sym *sym;
  PyObject *sym_obj;
  static char *kwlist[] = {"sym", NULL};

  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O", kwlist, &sym_obj)) {
    return NULL;
  }
  sym = (sen_sym*)PyCObject_AsVoidPtr(sym_obj);
  size = sen_sym_size(sym);
  return Py_BuildValue("i", size);
}


static PyMethodDef SennaMethods[] = {
  {"sen_init",  senna_sen_init, METH_VARARGS,
   "Initialize Senna library."},
  {"sen_fin",  senna_sen_fin, METH_VARARGS,
   "Finalize Senna library."},
  {"sen_index_create",  (PyCFunction)senna_sen_index_create, METH_VARARGS | METH_KEYWORDS,
   "Create Senna index and return it."},
  {"sen_index_open",  (PyCFunction)senna_sen_index_open, METH_VARARGS | METH_KEYWORDS,
   "Open Senna index."},
  {"sen_index_close",  (PyCFunction)senna_sen_index_close, METH_VARARGS | METH_KEYWORDS,
   "Close Senna index."},
  {"sen_index_remove",  (PyCFunction)senna_sen_index_remove, METH_VARARGS | METH_KEYWORDS,
   "Remove Senna index files."},
  {"sen_index_rename",  (PyCFunction)senna_sen_index_rename, METH_VARARGS | METH_KEYWORDS,
   "Rename Senna index files."},
  {"sen_index_upd",  (PyCFunction)senna_sen_index_upd, METH_VARARGS | METH_KEYWORDS,
   "Update (or insert into or delete from) Senna index."},
  {"sen_index_sel",  (PyCFunction)senna_sen_index_sel, METH_VARARGS | METH_KEYWORDS,
   "Search for document whose value contains string."},
  {"sen_records_next",  (PyCFunction)senna_sen_records_next, METH_VARARGS | METH_KEYWORDS,
   "Advance to the next record the current record if it is possible. Return 0 if fail, otherwise return length of the key of current record. "},
  {"sen_records_rewind",  (PyCFunction)senna_sen_records_rewind, METH_VARARGS | METH_KEYWORDS,
   "The current record is cleared. To read records again from the first records, a call to sen_records_next() is needed."},
  {"sen_records_curr_score",  (PyCFunction)senna_sen_records_curr_score, METH_VARARGS | METH_KEYWORDS,
   "Return score of the current record (goodness of relevant for search query)."},
  {"sen_records_curr_key",  (PyCFunction)senna_sen_records_curr_key, METH_VARARGS | METH_KEYWORDS,
   "Return length of the key of current record. If current record doesn't exist, return None"},
  {"sen_records_nhits",  (PyCFunction)senna_sen_records_nhits, METH_VARARGS | METH_KEYWORDS,
   "Return the number of records which are included in records."},
  {"sen_records_find",  (PyCFunction)senna_sen_records_find, METH_VARARGS | METH_KEYWORDS,
   "Find record which corresponds to given key in the records, return score value ifsuch record exist."},
  {"sen_records_close",  (PyCFunction)senna_sen_records_close, METH_VARARGS | METH_KEYWORDS,
   "Release the records instance."},
  {"sen_sym_open",  (PyCFunction)senna_sen_sym_open, METH_VARARGS | METH_KEYWORDS,
   "Open symbol table data structure."},
  {"sen_sym_info",  (PyCFunction)senna_sen_sym_info, METH_VARARGS | METH_KEYWORDS,
   "Return the number of records which are correspond to given key_size, flags and encoding of a sen_sym instance."},
  {"sen_sym_close",  (PyCFunction)senna_sen_sym_close, METH_VARARGS | METH_KEYWORDS,
   "Close symbol table data structure."},
  {"sen_sym_size",  (PyCFunction)senna_sen_sym_size, METH_VARARGS | METH_KEYWORDS,
   "Get the number of records of symbol table."},
  {NULL, NULL, 0, NULL}
};


PyMODINIT_FUNC
initsenna(void)
{
  PyObject *m;

  m = Py_InitModule("senna", SennaMethods);
  SennaError = PyErr_NewException("senna.error", NULL, NULL);
  Py_INCREF(SennaError);
  PyModule_AddObject(m, "error", SennaError);
  PyModule_AddIntConstant(m, "SEN_INDEX_NORMALIZE", 0x0001);
  PyModule_AddIntConstant(m, "SEN_INDEX_SPLIT_ALPHA", 0x0002);
  PyModule_AddIntConstant(m, "SEN_INDEX_SPLIT_DIGIT", 0x0004);
  PyModule_AddIntConstant(m, "SEN_INDEX_SPLIT_SYMBOL", 0x0008);
  PyModule_AddIntConstant(m, "SEN_INDEX_MORPH_ANALYSE", 0x0000);
  PyModule_AddIntConstant(m, "SEN_INDEX_NGRAM", 0x0010);
  PyModule_AddIntConstant(m, "SEN_INDEX_DELIMITED", 0x0020);
  PyModule_AddIntConstant(m, "SEN_INDEX_ENABLE_SUFFIX_SEARCH", 0x0100);
  PyModule_AddIntConstant(m, "SEN_INDEX_DISABLE_SUFFIX_SEARCH", 0x0200);
  PyModule_AddIntConstant(m, "SEN_INDEX_WITH_VACUUM", 0x8000);
  PyModule_AddIntConstant(m, "sen_enc_default", sen_enc_default);
  PyModule_AddIntConstant(m, "sen_enc_none", sen_enc_none);  
  PyModule_AddIntConstant(m, "sen_enc_euc_jp", sen_enc_euc_jp);
  PyModule_AddIntConstant(m, "sen_enc_utf8", sen_enc_utf8);
  PyModule_AddIntConstant(m, "sen_enc_sjis", sen_enc_sjis);
}
