/*!
  \file
  \brief テーマ時計の描画処理

  \author Satofumi KAMIMURA

  $Id$
*/

#include <QtGui>
#include <QTimer>
#include <QTime>
#include <QSettings>
#include <QAction>
#include "ThemeClockWidget.h"
#include "ThemeClockDraw.h"


struct ThemeClockWidget::pImpl {

  static const bool ForceCreate = true;

  QTimer clock_timer_;
  ThemeClockDraw* draw_widget_;
  QString pre_theme_file_;
  QString theme_file_;
  QPoint drag_position_;

  pImpl(ThemeClockWidget* parent)
    : draw_widget_(new ThemeClockDraw(parent)) {
  }

  // フォームの初期化
  void initializeForm(ThemeClockWidget* parent) {

    // 時計の更新設定
    connect(&clock_timer_, SIGNAL(timeout()), parent, SLOT(redraw()));

    // 終了処理
    connect(draw_widget_, SIGNAL(quit()), parent, SLOT(close()));

    // テーマファイルの Drop を許可
    parent->setAcceptDrops(true);
  }

  // 次に秒針が動くまでのミリ秒だけ待機
  void startClock(ThemeClockWidget* parent) {

    QTime current_time = QTime::currentTime();
    clock_timer_.stop();

    QTimer::singleShot(1000 - current_time.msec(), parent, SLOT(redraw()));
  }

  void redraw(ThemeClockWidget* parent) {
    static_cast<void>(parent);

    if (! clock_timer_.isActive()) {
      clock_timer_.start(1000);
    }

    // 再描画処理
    draw_widget_->redraw();
  }

  bool tryLoadTheme(const QString& theme_file) {

    bool ret = false;

    // テーマの読み出し
    if (loadThemeFile(theme_file)) {
      // 今回指定されたテーマ
      theme_file_ = theme_file;
      ret = true;

    } else if (loadThemeFile(pre_theme_file_)) {
      // 前回利用したテーマ
      theme_file_ = pre_theme_file_;
      ret = true;

    } else {
      // テーマのドロップを促すフォームを作成
      loadThemeFile("");
    }

    return ret;
  }

  bool loadThemeFile(const QString& theme_file) {

    draw_widget_->loadThemeFile(theme_file);
    return draw_widget_->isThemeLoaded();
  }

  QString getThemeFile(void) {
    return draw_widget_->getThemeFile();
  }

  void readSettings(bool position_specified, ThemeClockWidget* parent) {

    QSettings settings("Hyakuren Soft LTD.", "qtmclock");
    pre_theme_file_ = settings.value("theme_file", "").toString();

    if (! position_specified) {
      // 位置が指定されてないときのみ、設定を反映させる
      QRect position = settings.value("geometry", parent->geometry()).toRect();
      parent->setGeometry(position);
    }
  }

  void writeSettings(ThemeClockWidget* parent) {

    QSettings settings("Hyakuren Soft LTD.", "qtmclock");
    settings.setValue("theme_file", theme_file_);

    QRect position = parent->geometry();
    QSize size = parent->size();

    settings.setValue("geometry", QRect(position.x(), position.y(),
                                        size.width(), size.height()));
  }

  void initializeMenu(ThemeClockWidget* parent) {

    QAction* quit_action = new QAction(QAction::tr("E&xit"), parent);
    quit_action->setShortcut(tr("Ctrl+Q"));
    connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit()));
    parent->addAction(quit_action);

    parent->setContextMenuPolicy(Qt::ActionsContextMenu);
  }
};


ThemeClockWidget::ThemeClockWidget(const QString& theme_file,
                                   const bool position_specified,
                                   QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this)) {

  // 前回動作時の設定を読み出す
  pimpl->readSettings(position_specified, this);

  // テーマの読み出し
  pimpl->tryLoadTheme(theme_file);
  resize(pimpl->draw_widget_->size());

  // フォームの初期化
  pimpl->initializeForm(this);

  // 右クリックメニューの初期化
  pimpl->initializeMenu(this);

  // 時計の動作開始
  pimpl->startClock(this);
}


ThemeClockWidget::~ThemeClockWidget(void) {

  // 今回動作時の設定を保存
  pimpl->writeSettings(this);
}


void ThemeClockWidget::redraw(void) {
  pimpl->redraw(this);
}


void ThemeClockWidget::resizeEvent(QResizeEvent* event) {
  static_cast<void>(event);

  QRegion* region = pimpl->draw_widget_->getRegion();
  if (region) {
    setMask(*region);
  }

  // !!! または、表示自体の拡大率を変更してしまう？
  // !!! いや、通常はウィンドウが出ないので、画像サイズの変更は無理
  // !!! ドロップ用の処理を記述すべき
}


void ThemeClockWidget::mousePressEvent(QMouseEvent* event) {

  if (event->button() == Qt::LeftButton) {
    pimpl->drag_position_ = event->globalPos() - frameGeometry().topLeft();
    event->accept();
  }
}


void ThemeClockWidget::mouseMoveEvent(QMouseEvent* event) {

  if (event->buttons() & Qt::LeftButton) {
    move(event->globalPos() - pimpl->drag_position_);
    event->accept();
  }
}


void ThemeClockWidget::dragEnterEvent(QDragEnterEvent* event) {

  setBackgroundRole(QPalette::Highlight);
  event->acceptProposedAction();
}


void ThemeClockWidget::dragMoveEvent(QDragEnterEvent* event) {

  event->acceptProposedAction(); static_cast<void>(event);
}


void ThemeClockWidget::dropEvent(QDropEvent* event) {

  const QMimeData* mime_data = event->mimeData();

  if (! mime_data->hasUrls()) {
    return;
  }

  QList<QUrl> url_list = mime_data->urls();
  for (int i = 0; i < url_list.count(); ++i) {

    QString path = url_list.at(i).path();

    // 先頭の '/' を取り除く
    path.replace(0, 1, "");

    // 日本語ロケールの場合、文字列を SJIS に変換する (Windows 用)
    if (! QLocale::system().name().compare("ja_JP")) {
      QTextCodec* codec = QTextCodec::codecForName("Shift-JIS");
      path = codec->fromUnicode(path);
    }
    if (pimpl->draw_widget_->loadThemeFile(path)) {

      // !!! 本来の代入処理とこのあたりの代入処理とを、くくるべき
      pimpl->theme_file_ = path;
      resize(pimpl->draw_widget_->size());
      return;

    } else {
      QString out;
      QMessageBox::warning(this, tr("qtmclock"),
			   tr("%1 is not correct theme file.").arg(path),
			   QMessageBox::Ok);
    }
  }
}
