#ifndef ARCHIVE_CONFIG_H
#define ARCHIVE_CONFIG_H

/*!
  \file
  \brief 設定ファイルの管理

  \author Satofumi KAMIMURA

  $Id$
*/

#include <memory>


/*!
  \brief 設定ファイルの管理クラス
*/
class ArchiveConfig {

  ArchiveConfig(void);
  ArchiveConfig(const ArchiveConfig& rhs);
  ArchiveConfig& operator = (const ArchiveConfig& rhs);

  struct pImpl;
  std::auto_ptr<pImpl> pimpl;

 public:

  /*!
    \brief コンストラクタ

    \param archive_file [i] 読み出すファイル名
  */
  explicit ArchiveConfig(const char* archive_file);
  ~ArchiveConfig(void);


  /*!
    \brief リリースファイル名の取得

    \retval リリースファイル名
  */
  const char* getReleaseName(void) const;


  /*!
    \brief リリースファイル名の設定

    \param release_file [i] リリースファイル名
  */
  void setReleaseName(const char* release_file);


  /*!
    \brief RRDA キーの取得

    \retval RRDA キー
  */
  const char* getRrdaKey(void) const;


  /*!
    \brief RRDA キー

    \param rrda_key [i] RRDA キー
  */
  void setRrdaKey(const char* rrda_key);


  /*!
    \brief データ名の取得

    \param key [i] キー

    \retval データファイル名
  */
  const char* getNormalData(const char* key) const;


  /*!
    \brief データ名の設定

    \param key [i] キー
    \param name [i] データファイル名
  */
  void setNormalData(const char* key, const char* name);


  /*!
    \brief データ名の取得 (暗号化)

    \param key [i] キー

    \retval データファイル名
  */
  const char* getRestrictData(const char* key) const;


  /*!
    \brief データ名の設定 (暗号化)

    \param key [i] キー
    \param name [i] データファイル名
  */
  void setRestrictData(const char* key, const char* name);


  /*!
    \brief アーカイブファイルの更新

    \retval true 正常
    \retval false エラー
  */
  bool save(void);


  /*!
    \brief アーカイブファイル名を変更して更新

    \param name [i] アーカイブファイル名

    \retval true 正常
    \retval false エラー
  */
  bool saveAs(const char* name);


  /*!
    \brief 現在のアーカイブファイル名を取得

    \retval ! NULL アーカイブファイル名
    \retval NULL アーカイブファイル名なし
  */
  const char* getArchiveFileName(void) const;
};

#endif /* !ARCHIVE_CONFIG_H */
