/*!
  \file
  \brief RRDA リソースのデコード

  \author Satofumi KAMIMURA

  $Id$

  \todo ファイルを逐次読み出しながら圧縮する実装に変更する
  \todo zip の圧縮、解凍を抽出してクラス化する
  \todo アルゴリズムを用いたループ処理に変更する
*/

#include <fstream>
#include <string>
#include <vector>
#include <zlib.h>
#include <iterator>
#include "RrdaDecoder.h"

namespace {

  // タグ毎に、ファイル名と圧縮データを管理するクラス
  class TagFile {

    enum {
      BufferSize = 1024,        //!< 圧縮処理におけるバッファサイズ
    };

    std::string error_message_;

  public:
    std::string tag_;
    std::string file_;

    std::vector<char> compressed_data_;

    TagFile(void) : error_message_("no error.") {
    }

    TagFile(const std::string& tag, const std::string& file)
      : tag_(tag), file_(file) {
    }

    const char* what(void) const {

      return error_message_.c_str();
    }

    bool compress(void) {

      // ファイル内容の読み出し
      std::ifstream fin(file_.c_str(), std::fstream::binary);
      if (! fin.is_open()) {
        return false;
      }
      std::vector<unsigned char> original_data;
      copy(std::istreambuf_iterator<char>(fin),
           std::istreambuf_iterator<char>(),
           std::back_inserter(original_data));

      // 圧縮処理
      z_stream z;
      z.zalloc = NULL;
      z.zfree = NULL;
      z.opaque = NULL;

      if (deflateInit2(&z, Z_BEST_COMPRESSION, Z_DEFLATED, -MAX_WBITS,
                       8, Z_DEFAULT_STRATEGY) != Z_OK) {
        error_message_ = z.msg;
        return false;
      }

      unsigned char output_buffer[BufferSize];
      size_t left_size = original_data.size();
      int flush = Z_NO_FLUSH;

      z.next_in = &original_data[0];
      z.avail_in = left_size;
      z.next_out = output_buffer;
      z.avail_out = BufferSize;

      while (true) {

        int ret = deflate(&z, flush);
        if (ret == Z_STREAM_END) {
          // バッファの内容を書き出して終了
          writeOutput(z, output_buffer);
          break;

        } else if (ret != Z_OK) {
          error_message_ = z.msg;
          return false;
        }

        if (z.avail_in == 0) {
          // 入力が終了
          flush = Z_FINISH;
        }

        if (z.avail_out == 0) {
          // 内容をバッファに待避
          writeOutput(z, output_buffer);

          z.next_out = output_buffer;
          z.avail_out = BufferSize;
        }
      }
      deflateEnd(&z);

      return true;
    }


    // 出力バッファの退避
    void writeOutput(z_stream& z, const unsigned char output_buffer[]) {

      size_t add_size = BufferSize - z.avail_out;
      size_t current_size = compressed_data_.size();
      compressed_data_.resize(current_size + add_size);
      memcpy(&compressed_data_[current_size], output_buffer, add_size);
    }
  };
};


struct RrdaDecoder::pImpl {

  std::string release_file_;
  std::string rrda_key_;

  std::vector<TagFile> normal_data_;
  std::vector<TagFile> restrict_data_;


  pImpl(const char* fileName, const char* key)
    : release_file_(fileName), rrda_key_(key) {
  }


  // リリースファイルの書き出し
  bool release(void) {

    // 登録データの圧縮
    compressData(normal_data_);
    compressData(restrict_data_);

    // ディレクトリサイズの取得
    size_t normal_directory_size = directorySize(normal_data_);
    size_t restrict_directory_size = directorySize(restrict_data_);

    // データサイズの合計を取得
    size_t normal_total = dataSize(normal_data_);
    size_t restrict_total = dataSize(restrict_data_);

    std::ofstream fout(release_file_.c_str(), std::fstream::binary);
    if (! fout.is_open()) {
      return false;
    }

    // 最初のヘッダ書き出し
    const char* first_header = "RRDA - Ver1.00";
    fout << first_header << std::endl;

    // ディレクトリサイズ
    size_t directory_size =
      + (strlen(first_header) + 1)
      + (8 + 1)
      + (8 + 1 + 8 + 1)
      + normal_directory_size
      + (3 + 1);
    fout << hex(directory_size) << std::endl;

    // 制限データ位置、サイズの書き出し
    fout << hex(directory_size + normal_total + restrict_total) << ','
         << hex(restrict_directory_size) << std::endl;

    // 通常ディレクトリ文字列の出力
    size_t count = directory_size;
    count = outputDirectoryLines(fout, normal_data_, count);

    // 通常データの出力
    count = outputData(fout, normal_data_, count);

    // 制限データの出力
    count = outputData(fout, restrict_data_, count);

    // 制限ディレクトリの出力
    outputDirectoryLines(fout, restrict_data_, count);

    return true;
  }


  // データの圧縮処理
  void compressData(std::vector<TagFile>& data) {

    // !!! アルゴリズムを使うべき

    for (std::vector<TagFile>::iterator it = data.begin();
         it != data.end(); ++it) {
      it->compress();
    }
  }


  // ディレクトリサイズの計算
  size_t directorySize(std::vector<TagFile>& data) {

    size_t directory_size = 0;

    // !!! アルゴリズムを使うべき

    for (std::vector<TagFile>::iterator it = data.begin();
         it != data.end(); ++it) {
      directory_size += 8 + 1 + 8 + 1 + it->tag_.size() + 1;
    }
    return directory_size;
  }


  // データサイズの取得
  size_t dataSize(std::vector<TagFile>& data) {

    size_t total_size = 0;

    // !!! アルゴリズムを使うべき

    for (std::vector<TagFile>::iterator it = data.begin();
         it != data.end(); ++it) {
      total_size += it->compressed_data_.size();
    }
    return total_size;
  }


  // 8 桁の 16進文字列を返す
  std::string hex(unsigned long value) {

    char buffer[] = "01234567";
    snprintf(buffer, 9, "%08lx", value);

    return buffer;
  }


  // ディレクトリ文字列の出力
  size_t outputDirectoryLines(std::ofstream& fout,
                            std::vector<TagFile>& data, size_t count) {

    size_t next_count = count;

    for (std::vector<TagFile>::iterator it = data.begin();
         it != data.end(); ++it) {

      size_t data_size = it->compressed_data_.size();

      fout << hex(next_count) << ','
           << hex(data_size) << ','
           << it->tag_ << std::endl;

      next_count += data_size;
    }

    fout << "END" << std::endl;

    return next_count;
  }


  // データの出力
  size_t outputData(std::ofstream& fout,
                    std::vector<TagFile>& data, size_t count) {

    size_t next_count = count;

    for (std::vector<TagFile>::iterator it = data.begin();
         it != data.end(); ++it) {

      // バイナリデータの書きだし
      size_t n = it->compressed_data_.size();
      fout.write(&it->compressed_data_[0], n);
      next_count += n;
    }
    return next_count;
  }
};


RrdaDecoder::RrdaDecoder(const char* fileName, const char* key)
  : pimpl(new pImpl(fileName, key)) {
}


RrdaDecoder::~RrdaDecoder(void) {
}


// 通常データの登録
void RrdaDecoder::setNormalData(const std::string& tag,
                                const std::string& file) {

  pimpl->normal_data_.push_back(TagFile(tag, file));
}


// 暗号データの登録
void RrdaDecoder::setRestrictData(const std::string& tag,
                                  const std::string& file) {

  pimpl->restrict_data_.push_back(TagFile(tag, file));
}


// リリースファイルの書き出し
bool RrdaDecoder::release(void) {

  return pimpl->release();
}
