/*!
  \file
  \brief テーマ時計の描画

  \author Satofumi KAMIMURA

  $Id$

  \todo 拡大縮小の機能を実装する
  \todo 月、日、曜日、の表示機能を付加する
  \todo エラーメッセージの処理
  \todo 時刻毎のベース画像切替え機能を実装する
  \todo アニメーション機能を付加する
*/

#include <QLabel>
#include <QVBoxLayout>
#include <QPainter>
#include <QRegion>
#include <QBitmap>
#include <QTime>
#include "ThemeClockDraw.h"
#include "RrdaEncoder.h"
#include "RrdaResource.h"
#include "SettingConfig.h"

#ifdef _MSC_VER
#define snprintf _snprintf
#endif

namespace {

  class Resource {
  public:
    QMap<QString, QPixmap*> pixmap;
    QMap<QString, QRect> offset;
    QMap<QString, QString> option;
  };
};


struct ThemeClockDraw::pImpl {

  static const bool CenterPosition = true; // 画像中心を基準にして描画

  QLabel* message_label_;
  bool theme_loaded_;
  QString theme_file_;
  Resource theme_resource_;
  QRegion* region_;
  QVector<QPixmap> digital_pixmap_;
  QVector<QPoint> digital_positions_;


  pImpl(ThemeClockDraw* parent)
    : message_label_(NULL), theme_loaded_(false), region_(NULL) {

    // テーマファイルのドラッグを促すフォームを作成
    message_label_ =
      new QLabel(QLabel::tr("Drop theme file in this form."), parent);
    message_label_->setAlignment(Qt::AlignCenter);
    QSize size = message_label_->sizeHint();
    parent->resize(QSize(static_cast<int>(size.width() * 1.2),
                         size.height() * 4));
    QVBoxLayout* layout = new QVBoxLayout(parent);
    layout->addWidget(message_label_);
    parent->setLayout(layout);
  }


  // テーマファイルの読み出し
  bool loadThemeFile(const QString& theme_file, const char* key) {

    // リソースの読み出しを行う
    RrdaEncoder rrda_encoder(theme_file.toStdString().c_str(), key);

    std::vector<std::string> resource_files;
    rrda_encoder.getResourceNames(resource_files);

    // "config" がリソースに含まれているかの確認
    if (! isExist("config", resource_files)) {
      // !!! リソースにないよー、と表示して戻る
      // !!!
      // !!! 失敗する可能性としては、
      // !!! キーが違う、テーマファイルでない、壊れている、などか？
      return false;
    }

    // config を読み出してパース
    std::auto_ptr<RrdaResource> config(rrda_encoder.getResource("config"));
    size_t size = 0;
    char* p = reinterpret_cast<char*>(config->get(&size));

    // 設定を読みだし、必要な情報を格納する
    if (! loadResources(p, size, rrda_encoder)) {
      // !!! 「リソースが不足しています」という表示を行う
      // !!! テーマファイルが不完全です。か？ これも意味不明だな...。
      // !!!
      return false;
    }

    theme_loaded_ = true;
    theme_file_ = theme_file;

    return true;
  }


  // リソース名が存在するかを返す
  bool isExist(const char* resource_name,
               const std::vector<std::string>& resources) {

    std::vector<std::string>::const_iterator p =
      std::find(resources.begin(), resources.end(), resource_name);

    return (p == resources.end()) ? false : true;
  }


  // リソースの読み出し
  bool loadResources(char* p, size_t size, RrdaEncoder& rrda_encoder) {

    // 設定ファイルの読み出し
    QByteArray data(p, size);
    QMap<QString, QString> tags;
    SettingConfig setting(tags, data);

    // 画像の格納
    Resource resource;
    loadPixmap(resource, "base/base_org", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_h", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_m", tags, rrda_encoder);
    loadPixmap(resource, "analog/hari_s", tags, rrda_encoder);

    // アナログ部品の位置を格納
    loadOffset(resource, "analog/hari_h",
               "analog/h_xpos", "analog/h_ypos", tags);
    loadOffset(resource, "analog/hari_m",
               "analog/m_xpos", "analog/m_ypos", tags);
    loadOffset(resource, "analog/hari_s",
               "analog/s_xpos", "analog/s_ypos", tags);

    // 設定の格納
    loadOption(resource, "analog/sec", tags);
    loadOption(resource, "digital/colon", tags);
    loadOption(resource, "digital/sec", tags);

    // デジタル文字を、文字毎に格納
    loadDigitalCharacter(resource, tags, rrda_encoder);

    // デジタル文字の描画位置を設定
    loadDigitalPosition(tags);

    // 描画リソースが何もなければ、エラー
    if (resource.pixmap.empty()) {
      return false;
    }

    std::swap(theme_resource_, resource);

    return true;
  }


  // 画像の読み出し
  bool loadPixmap(Resource& resource, const QString& tag_name,
                  const QMap<QString, QString>& tags,
                  RrdaEncoder& rrda_encoder) {

    // タグ名の RRDA リソースからの取得
    QString pixmap_tag_name = tags.value(tag_name, "");
    if (pixmap_tag_name.isEmpty()) {
      // !!! 存在しない
      return false;
    }

    // 画像の読み出し
    std::auto_ptr<RrdaResource>
      config(rrda_encoder.getResource(pixmap_tag_name.toStdString().c_str()));
    size_t size = 0;
    char* p = reinterpret_cast<char*>(config->get(&size));
    if (size == 0) {
      return false;
    }

    // 画像の登録
    QByteArray data(p, size);
    QPixmap* pixmap = new QPixmap;
    pixmap->loadFromData(data);
    resource.pixmap.insert(tag_name, pixmap);

    return true;
  }


  // 対応する画像が既に resouce に登録されていなければならない
  void loadOffset(Resource& resource, const QString& tag_name,
                  const QString& x_tag, const QString& y_tag,
                  const QMap<QString, QString>& tags) {

    // 位置の取得と登録
    int x = tags.value(x_tag, 0).toInt();
    int y = tags.value(y_tag, 0).toInt();

    // サイズの取得
    QPixmap* pixmap = resource.pixmap.value(tag_name, NULL);
    int w = (pixmap) ? pixmap->width() : 0;
    int h = (pixmap) ? pixmap->height() : 0;

    resource.offset.insert(tag_name, QRect(x, y, w, h));
  }


  // オプションの読み出し
  void loadOption(Resource& resource, const QString& tag_name,
                  const QMap<QString, QString>& tags) {

    // オプションの取得と登録
    QString tag_value = tags.value(tag_name, "");
    if (tag_value.isEmpty()) {
      // !!! 存在しない
      return;
    }
    resource.option.insert(tag_name, tag_value);
  }


  // デジタル画像の読み出し
  void loadDigitalCharacter(Resource& resource,
                            QMap<QString, QString>& tags,
                            RrdaEncoder& rrda_encoder) {

    Resource digital_resource;
    if (! loadPixmap(digital_resource, "digital/parts", tags, rrda_encoder)) {
      return;
    }

    QPixmap* pixmap = digital_resource.pixmap.value("digital/parts", NULL);
    if (! pixmap) {
      // !!! 失敗はしないはずだが、一応
      return;
    }
    const QSize size = pixmap->size();

    // 文字毎に等間隔で分割して登録する
    const char* tag_names[] = {
      "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
      " ", ":", "A", "P",
    };
    size_t n = sizeof(tag_names) / sizeof(tag_names[0]);
    digital_pixmap_.clear();
    for (size_t i = 0; i < n; ++i) {

      // !!! list にしておけば、アドレスが変化しないので、より適切か？
      digital_pixmap_.
        push_back(pixmap->copy(size.width() * i / n,
                               0, size.width() / n, size.height()));
    }

    // 登録した vector のアドレスを確保
    for (size_t i = 0; i < n; ++i) {
      resource.pixmap.insert(tag_names[i], &digital_pixmap_[i]);
    }
  }


  // デジタル画像の描画位置を読み出し
  void loadDigitalPosition(QMap<QString, QString>& tags) {

    // "hh:mm ss x";
    digital_positions_.clear();
    digital_positions_.resize(10);

    digitalPositionUpdate(0, tags, "digital/h1_xpos", "digital/h1_ypos");
    digitalPositionUpdate(1, tags, "digital/h2_xpos", "digital/h2_ypos");

    digitalPositionUpdate(2, tags, "digital/cl_xpos", "digital/cl_ypos");

    digitalPositionUpdate(3, tags, "digital/m1_xpos", "digital/m1_ypos");
    digitalPositionUpdate(4, tags, "digital/m2_xpos", "digital/m2_ypos");

    digitalPositionUpdate(6, tags, "digital/s1_xpos", "digital/s1_ypos");
    digitalPositionUpdate(7, tags, "digital/s2_xpos", "digital/s2_ypos");

    digitalPositionUpdate(9, tags, "digital/ap_xpos", "digital/ap_ypos");
  }


  // デジタル文字の描画位置を設定
  void digitalPositionUpdate(int index, QMap<QString, QString>& tags,
                             const QString& x_tag, const QString& y_tag) {

    int x = tags.value(x_tag, 0).toInt();
    int y = tags.value(y_tag, 0).toInt();
    digital_positions_[index] = QPoint(x, y);
  }


  // フォームの初期化
  void initializeWidget(ThemeClockDraw* parent) {

    QPixmap* base_pixmap = theme_resource_.pixmap.value("base/base_org");
    region_ = new QRegion(base_pixmap->mask());
    parent->resize(base_pixmap->size());
  }


  // 再描画
  void paintEvent(ThemeClockDraw* parent) {

    QPainter painter(parent);
    painter.setRenderHint(QPainter::SmoothPixmapTransform);

    // ベース画像の描画
    drawPixmap(painter, "base/base_org");

    // 部品画像の描画
    QTime time = QTime::currentTime();
    updateDigital(painter, time.hour(), time.minute(), time.second());
    updateAnalog(painter, time.hour(), time.minute(), time.second());
  }


  // アナログ部分の再描画
  void updateAnalog(QPainter& painter, int hour, int minute, int second) {

    // 針の角度を再計算
    qreal analog_h = (360.0 * (hour / 12.0)) + (30.0 * (minute / 60.0));
    drawPixmap(painter, "analog/hari_h", analog_h, CenterPosition);

    qreal analog_m = 360.0 * minute / 60.0;
    drawPixmap(painter, "analog/hari_m", analog_m, CenterPosition);

    if (! compare_value("analog/sec", "on")) {
      // 秒針は、指定があるときのみ描画する
      qreal analog_s = 360.0 * second / 60.0;
      drawPixmap(painter, "analog/hari_s", analog_s, CenterPosition);
    }
  }


  // デジタル部分の再描画
  void updateDigital(QPainter& painter, int hour, int minute, int second) {

    enum { BufferSize = 11 };
    char buffer[BufferSize] = "hh:mm ss x";

    if (! compare_value("digital/sec", "on")) {
      // 秒を描画する
      snprintf(buffer, BufferSize, "%02d:%02d_%02d_%c",
               hour, minute, second, (hour >= 12) ? 'P' : 'A');
    } else {
      // 秒は描画しない
      snprintf(buffer, BufferSize, "%02d:%02d____%c",
               hour, minute, (hour >= 12) ? 'P' : 'A');
    }

    if (! compare_value("digital/colon", "on")) {
      // コロンを点滅させる
      if (second & 0x01) {
        buffer[2] = ' ';
      }
    }

    // 描画処理
    char digital_char[] = "_";
    size_t n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      digital_char[0] = buffer[i];
      QPixmap* parts = theme_resource_.pixmap.value(digital_char, NULL);
      if (! parts) {
        continue;
      }

      int x = digital_positions_[i].x();
      int y = digital_positions_[i].y();
      painter.drawPixmap(QPoint(x, y), *parts);
    }
  }


  // 画像の描画
  void drawPixmap(QPainter& painter,
                  const QString& tag_name,
                  qreal rotate_degree = 0,
                  bool center_position = false) {

    QPixmap* pixmap = theme_resource_.pixmap.value(tag_name, NULL);
    if (! pixmap) {
      return;
    }

    QRect offset = theme_resource_.offset.value(tag_name, QRect(0, 0, 0, 0));

    if (center_position) {
      // 回転前の状態をスタックに積んでおく
      painter.save();

      // 画像の中心で回転させてから、所定の場所に描画を行う
      painter.translate(offset.x(), offset.y());
      painter.rotate(rotate_degree);

      int x = -offset.width() / 2;
      int y = -offset.height() / 2;
      painter.drawPixmap(QPoint(x, y), *pixmap);
      painter.restore();

    } else {
      int x = offset.x();
      int y = offset.y();
      painter.drawPixmap(QPoint(x, y), *pixmap);
    }
  }


  // 指定したタグが存在するか、の比較用
  int compare_value(const QString& tag_name, const QString& compare_value) {

    return theme_resource_.option.value(tag_name, "").compare(compare_value);
  }
};


ThemeClockDraw::ThemeClockDraw(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this)) {
}


ThemeClockDraw::~ThemeClockDraw(void) {
}


// テーマファイルの読み出し
bool ThemeClockDraw::loadThemeFile(const QString& theme_file,
                                   const char* key) {

  if (! pimpl->loadThemeFile(theme_file, key)) {
    return false;
  }

  // フォーム初期化を行い、ドラッグ用のメッセージを隠す
  pimpl->initializeWidget(this);
  pimpl->message_label_->hide();

  return true;
}


// テーマファイル名の取得
const QString ThemeClockDraw::getThemeFile(void) {

  // !!! 可能ならば、絶対パスを返すべき
  return pimpl->theme_file_;
}


// 再描画
void ThemeClockDraw::redraw(void) {

  if (pimpl->theme_loaded_) {
    // 描画イベントを発生させる
    update();
  }
}


// 描画イベント
void ThemeClockDraw::paintEvent(QPaintEvent* event) {
  static_cast<void>(event);

  if (pimpl->theme_loaded_) {
    pimpl->paintEvent(this);
  }
}


// 透過領域の取得
QRegion* ThemeClockDraw::getRegion(void) {

  return (pimpl->theme_loaded_) ? pimpl->region_ : NULL;
}
