/*!
  \file
  \brief XML 設定ファイルの管理

  \author Satofumi KAMIMURA

  $Id$
*/

#include <QVector>
#include <QXmlDefaultHandler>
#include <fstream>
#include <string>
#include "SettingConfig.h"


namespace {

  typedef QMap<QString, QString> TagMap;

  class XmlHandler : public QXmlDefaultHandler {
    QString current_text_;
    QVector<QString> tag_stack_;
    TagMap& tag_map_;

  public:

    XmlHandler(TagMap& tags) : tag_map_(tags) {
    }


    // 要素の開始
    bool startElement(const QString& namespaceURI,
                      const QString& localName,
                      const QString& qName,
                      const QXmlAttributes& attributes) {
      static_cast<void>(namespaceURI);
      static_cast<void>(localName);
      static_cast<void>(attributes);

      tag_stack_.push_back(qName);
      current_text_.clear();

      return true;
    }


    // 要素の終了
    bool endElement(const QString& namespaceURI,
                    const QString& localName,
                    const QString& qName) {
      static_cast<void>(namespaceURI);
      static_cast<void>(localName);
      static_cast<void>(qName);

      QString value = current_text_.trimmed();
      current_text_.clear();

      // 前後の空白を除去して空になったら、終端とみなして何もしない
      if (! value.isEmpty()) {

        QString key;
        for (QVector<QString>::const_iterator it = tag_stack_.begin();
             it != tag_stack_.end(); ++it) {
          if (it == tag_stack_.begin()) {
            // 最初のタグ <lm_tmclock> は、無視する
            continue;
          }
          key += *it + '/';
        }
        key.chop(1);

        // タグ名の登録
        tag_map_.insert(key, value);
      }
      tag_stack_.pop_back();

      return true;
    }


    // タグの取得
    bool characters(const QString& str) {
      current_text_ += str;
      return true;
    }


    // エラー処理
    bool fatalError(const QXmlParseException& exception) {
      static_cast<void>(exception);

      // XML のデータ末尾に不正なデータがあっても読み飛ばせるように、false
      return false;
    }
  };
};


struct SettingConfig::pImpl {

  std::string config_file_;
  TagMap tags_;


  pImpl(const std::string& config_file)
    : config_file_(config_file) {

    // デフォルトパラメータの設定
    setDefaults();

    // 読み出し
    load(config_file_);
  }


  pImpl(TagMap& tags, QByteArray& data) {

    // private メンバに対する処理結果が返されるようにする
    std::swap(tags_, tags);

    // 読み出し
    load(data);

    std::swap(tags, tags_);
  }


  // デフォルト値の設定
  void setDefaults(void) {

    // ベース画像
    tags_.insert("base/base_org", "base");

    // アナログ画像
    tags_.insert("analog/hari_h", "hari_h");
    tags_.insert("analog/hari_m", "hari_m");
    tags_.insert("analog/hari_s", "hari_s");

    tags_.insert("analog/sec", "on");

    tags_.insert("analog/h_xpos", "0");
    tags_.insert("analog/h_ypos", "0");
    tags_.insert("analog/m_xpos", "0");
    tags_.insert("analog/m_ypos", "0");
    tags_.insert("analog/s_xpos", "0");
    tags_.insert("analog/s_ypos", "0");

    // デジタル画像
    tags_.insert("digital/parts", "digital");
    tags_.insert("digital/colon", "on");
    tags_.insert("digital/sec", "on");

    tags_.insert("digital/ap_xpos", "0");
    tags_.insert("digital/ap_ypos", "0");
    tags_.insert("digital/h1_xpos", "0");
    tags_.insert("digital/h1_ypos", "0");
    tags_.insert("digital/h2_xpos", "0");
    tags_.insert("digital/h2_ypos", "0");
    tags_.insert("digital/cl_xpos", "0");
    tags_.insert("digital/cl_ypos", "0");
    tags_.insert("digital/m1_xpos", "0");
    tags_.insert("digital/m1_ypos", "0");
    tags_.insert("digital/m2_xpos", "0");
    tags_.insert("digital/m2_ypos", "0");
    tags_.insert("digital/s1_xpos", "0");
    tags_.insert("digital/s1_ypos", "0");
    tags_.insert("digital/s2_xpos", "0");
    tags_.insert("digital/s2_ypos", "0");
  }


  // 読み出し
  bool load(std::string& file) {

    std::ifstream fin(file.c_str());
    if (! fin.is_open()) {
      return false;
    }

    // 全内容のデータを作成する
    QString all_lines;
    std::string line;
    while (getline(fin, line)) {
      all_lines += line.c_str();
    }

    QXmlInputSource xml_resource;
    xml_resource.setData(all_lines);
    return loadXml(xml_resource);
  }


  // 読み出し
  bool load(QByteArray& data) {

    QXmlInputSource xml_resource;
    xml_resource.setData(data);
    return loadXml(xml_resource);
  }


  // XML の読み出し処理
  bool loadXml(QXmlInputSource& xml_resource) {

    // XML 内容の読み出し
    XmlHandler xml_handler(tags_);
    QXmlSimpleReader reader;
    reader.setContentHandler(&xml_handler);
    reader.setErrorHandler(&xml_handler);

    reader.parse(xml_resource);

    return true;
  }


  // 保存
  bool save(std::string& file) {

    std::ofstream fout(file.c_str());
    if (! fout.is_open()) {
      return false;
    }

    fout << "<?xml version=\"1.0\"?>" << std::endl
         << "<lm_tmclock>" << std::endl;

    // ベース情報
    const char* base_tags[] = { "base_org" };
    size_t n = sizeof(base_tags) / sizeof(base_tags[0]);
    writeTagValues(fout, "base", base_tags, n);

    fout << std::endl;

    // アナログ情報
    const char* analog_tags[] = {
      "hari_h", "hari_m", "hari_s",
      "sec",
      "h_xpos", "h_ypos", "m_xpos", "m_ypos", "s_xpos", "s_ypos",
    };
    n = sizeof(analog_tags) / sizeof(analog_tags[0]);
    writeTagValues(fout, "analog", analog_tags, n);

    fout << std::endl;

    // デジタル情報
    const char* digital_tags[] = {
      "parts",
      "colon", "sec",
      "ap_xpos", "ap_ypos",
      "h1_xpos", "h1_ypos", "h2_xpos", "h2_ypos", "cl_xpos", "cl_ypos",
      "m1_xpos", "m1_ypos", "m2_xpos", "m2_ypos",
      "s1_xpos", "s1_ypos", "s2_xpos", "s2_ypos",
    };
    n = sizeof(digital_tags) / sizeof(digital_tags[0]);
    writeTagValues(fout, "digital", digital_tags, n);

    fout << "<lm_tmclock" << std::endl;

    return true;
  }


  // タグと値の書き出し
  void writeTagValues(std::ofstream& fout,
                      const char* base, const char* tags[], size_t n) {

    fout << "  <" << base << ">" << std::endl;

    for (size_t i = 0; i < n; ++i) {
      const char* tag = tags[i];
      QString key = QString(base) + "/" + QString(tag);

      fout << "    <" << tag << ">"
           << " " << tags_[key].toStdString() << " "
           << "</" << tag << ">" << std::endl;
    }

    fout << "  </" << base << ">" << std::endl;
  }
};


SettingConfig::SettingConfig(const std::string& config_file)
  : pimpl(new pImpl(config_file)) {
}


SettingConfig::SettingConfig(QMap<QString, QString>& tags, QByteArray& data)
  : pimpl(new pImpl(tags, data)) {
}


SettingConfig::~SettingConfig(void) {
}


// 値の読み出し
const std::string SettingConfig::getValue(const std::string& tag) {

  TagMap::iterator p = pimpl->tags_.find(tag.c_str());
  return (p != pimpl->tags_.end()) ? p.value().toStdString() : "";
}


// 値の書き込み
void SettingConfig::setValue(const std::string& tag, const std::string& value) {

  TagMap::iterator p = pimpl->tags_.find(tag.c_str());
  if (p == pimpl->tags_.end()) {
    // 新しいタグは登録させない
    return;
  }

  // 既存のタグへの更新のみ許可する
  pimpl->tags_[tag.c_str()] = value.c_str();
}


// 保存
void SettingConfig::save(void) {

  pimpl->save(pimpl->config_file_);
}
