/*!
  \file
  \brief テーマ時計の部品描画

  \author Satofumi KAMIMURA

  $Id$

  \todo 拡大縮小を毎回しているのを、なんとかする
*/

#include <QPainter>
#include <QTime>
#include <cmath>
#include "ClockPartsDraw.h"
#include "ThemeResource.h"
#include "ThemeClockDraw.h"


namespace {
  static const bool CenterPosition = true; // 画像中心を基準にした描画を指示

  // 位置の取得
  QRect getPosition(const ThemeResource& resource,
                    const std::string& tag_name) {

    return resource.offset.value(tag_name, QRect(0, 0, 0, 0));
  }


  // 画像の描画
  void drawPixmap(QPainter& painter,
                  const ThemeResource& resource,
                  const std::string& tag_name,
                  const QRect& offset,
                  size_t scaling_percent,
                  qreal rotate_degree = 0,
                  bool center_position = false) {

    QPixmap* pixmap = resource.pixmap.value(tag_name, NULL);
    if (! pixmap) {
      return;
    }
    QSize size = pixmap->size();
    QPixmap scaled_pixmap =
      pixmap->scaled(static_cast<int>(scaling_percent * size.width() / 100.0),
                     static_cast<int>(scaling_percent * size.height() / 100.0));

    int offset_x =
      static_cast<int>(round(scaling_percent * offset.x() / 100.0));
    int offset_y =
      static_cast<int>(round(scaling_percent * offset.y() / 100.0));

    if (center_position) {
      // 回転前の状態をスタックに積んでおく
      painter.save();

      // 画像の中心で回転させてから、所定の場所に描画を行う
      painter.translate(offset_x, offset_y);
      painter.rotate(rotate_degree);

      int x = -static_cast<int>(scaling_percent * offset.width() / 2 / 100.0);
      int y = -static_cast<int>(scaling_percent * offset.height() / 2 / 100.0);
      painter.drawPixmap(QPoint(x, y), scaled_pixmap);
      painter.restore();

    } else {
      painter.drawPixmap(QPoint(offset_x, offset_y), scaled_pixmap);
    }
  }


  // アナログ部分の再描画
  void updateAnalog(QPainter& painter, const ThemeResource& resource,
                    int hour, int minute, int second, size_t scaling_percent) {

    // 針の角度を再計算
    qreal analog_h_degree = (360.0 * (hour / 12.0)) + (30.0 * (minute / 60.0));
    QRect analog_h_offset = getPosition(resource, "analog/hari_h");
    drawPixmap(painter, resource, "analog/hari_h", analog_h_offset,
               scaling_percent, analog_h_degree, CenterPosition);

    qreal analog_m_degree = 360.0 * minute / 60.0;
    QRect analog_m_offset = getPosition(resource, "analog/hari_m");
    drawPixmap(painter, resource, "analog/hari_m", analog_m_offset,
               scaling_percent, analog_m_degree, CenterPosition);

    if (! resource.option.value("analog/sec", "").compare("on")) {
      // 秒針は、指定があるときのみ描画する
      qreal analog_s_degree = 360.0 * second / 60.0;
      QRect analog_s_offset = getPosition(resource, "analog/hari_s");
      drawPixmap(painter, resource, "analog/hari_s", analog_s_offset,
                 scaling_percent, analog_s_degree, CenterPosition);
    }
  }


  // デジタル部分の再描画
  void updateDigital(QPainter& painter, const ThemeResource& resource,
                     const QVector<QPoint>& digital_positions,
                     int hour, int minute, int second, size_t scaling_percent) {

    enum { BufferSize = 11 };
    char buffer[BufferSize] = "hh:mm ss x";

    if (! resource.option.value("digital/sec", "").compare("on")) {
      // 秒を描画する
      snprintf(buffer, BufferSize, "%02d:%02d_%02d_%c",
               hour, minute, second, (hour >= 12) ? 'P' : 'A');
    } else {
      // 秒は描画しない
      snprintf(buffer, BufferSize, "%02d:%02d____%c",
               hour, minute, (hour >= 12) ? 'P' : 'A');
    }

    if (! resource.option.value("digital/colon", "").compare("on")) {
      // コロンを点滅させる
      if (second & 0x01) {
        buffer[2] = ' ';
      }
    }

    // 描画処理
    char digital_char[] = "_";
    size_t n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      digital_char[0] = buffer[i];
      QPixmap* parts = resource.pixmap.value(digital_char, NULL);
      if (! parts) {
        continue;
      }

      QSize size = parts->size();
      int x = digital_positions[i].x();
      int y = digital_positions[i].y();
      drawPixmap(painter, resource, digital_char,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }
  }
};


// 再描画
void ClockPartsDraw::redraw(QPainter& painter,
                            const ThemeResource& resource,
                            const QVector<QPoint>& digital_positions,
                            size_t scaling_percent, int sec_offset) {

  painter.setRenderHint(QPainter::SmoothPixmapTransform);

  // ベース画像の描画
  QRect offset = getPosition(resource, "base/base_org");
  drawPixmap(painter, resource, "base/base_org", offset, scaling_percent);

  // 部品画像の描画
  QTime current_time = QTime::currentTime();
  QTime time = current_time.addSecs(sec_offset);
  updateDigital(painter, resource, digital_positions,
                time.hour(), time.minute(), time.second(),
                scaling_percent);
  updateAnalog(painter, resource,
               time.hour(), time.minute(), time.second(), scaling_percent);
}
