/*!
  \file
  \brief 設定ファイルの処理クラス

  \author Satofumi KAMIMURA

  $Id$
*/

#include <fstream>
#include <map>
#include "ArchiveConfig.h"
#include "SplitString.h"

namespace {
  typedef std::map<std::string, std::string> DataMap;
};

struct ArchiveConfig::pImpl {

  std::string archive_file_;    //!< 設定ファイル名
  std::string dir_path_;        //!< 設定ファイルのあるディレクトリへの絶対パス
  std::string release_file_;    //!< リリース時の出力ファイル名
  std::string rrda_key_;        //!< RRDA キー

  DataMap normal_data_;         //!< 通常データ
  DataMap restricted_data_;     //!< 制限データ


  pImpl(const std::string& archive_file) : archive_file_(archive_file) {

    // デフォルト値を設定してから、設定ファイルの内容を読み出す
    setDefault();
    loadArchiveFile();

    // 初期のタイトル表示を読み出す
    // !!!
    // !!! 現在編集中のファイル名を表示するときに使う
  }


  // デフォルト設定
  void setDefault(void) {

    release_file_ = "lm_clocktheme_xxxx.dat";
    rrda_key_ = "LMTC";

    normal_data_["config"] = "./lm_tc.xml";
    normal_data_["base"] = "./clock_base.png";
    normal_data_["digital"] = "./segment.png";
    normal_data_["hari_h"] = "./hand_h.png";
    normal_data_["hari_m"] = "./hand_m.png";
    normal_data_["hari_s"] = "./hand_s.png";

    restricted_data_["dummy"] = "./archive_list.txt";
  }


  // 設定ファイルの読み出し
  void loadArchiveFile(void) {

    std::ifstream fin(archive_file_.c_str());
    if (! fin.is_open()) {
      return;
    }

    // ディレクトリパスの取得
    size_t last_slash = archive_file_.rfind("/");
    dir_path_ = (last_slash != std::string::npos) ?
      archive_file_.substr(0, last_slash + 1) : "./";

    bool in_normal_data = true;
    std::string line;
    while (getline(fin, line)) {
      if (line.empty() || (line[0] == '#')) {
        // 空行, 先頭が '#' の行は読み飛ばす
        continue;
      }

      // \name リリース時の出力ファイル名
      if (registerKeyValue(release_file_, line, "\\name ")) {
        continue;
      }

      // \key レジストリキー
      if (registerKeyValue(rrda_key_, line, "\\key ")) {
        continue;
      }

      // \restriction 以降のデータは、制限データ
      if (! line.compare("\\restriction")) {
        in_normal_data = false;
        continue;
      }

      // <key>, <value> データの登録
      if (in_normal_data) {
        registerData(normal_data_, line);
      } else {
        registerData(restricted_data_, line);
      }
    }
  }


  // データの登録
  void registerData(DataMap& data, std::string& line) {

    std::vector<std::string> tokens;
    size_t n = qrk::split(tokens, line, ", \t");
    if (n != 2) {
      return;
    }

    setData(data, tokens[0], tokens[1]);
  }


  // キーワードが一致する場合、以降の変数を登録
  bool registerKeyValue(std::string& output_line,
                        const std::string& input_line, const char* key) {
    size_t key_length = strlen(key);
    if (input_line.compare(0, key_length, key)) {
      return false;
    }

    output_line = input_line.substr(key_length, std::string::npos);
    return true;
  }


  // 設定ファイルの保存
  bool saveArchiveFile(const std::string& name) {

    std::ofstream fout(name.c_str());
    if (! fout.is_open()) {
      return false;
    }

    // 再出力
    fout << "# RRDA archive list" << std::endl
         << "#" << std::endl
         << std::endl
         << "# --- release file name" << std::endl
         << "\\name " << release_file_ << std::endl
         << std::endl
         << "# --- register key" << std::endl
         << "\\key " << rrda_key_ << std::endl
         << std::endl;

    outputData(fout, normal_data_);

    fout << std::endl
         << "\\restriction" << std::endl
         << std::endl;

    outputData(fout, restricted_data_);

    // 最後に保存した名前で更新
    archive_file_ = name;
    return true;
  }


  // 保持データの出力
  void outputData(std::ofstream& fout, DataMap& data) {

    for (DataMap::iterator it = data.begin(); it != data.end(); ++it) {
      fout << it->first << ", " << it->second << std::endl;
    }
  }


  // 登録データの読み出し
  const std::string getData(DataMap& data, const std::string& key) const {

    DataMap::iterator p = data.find(key);
    return (p != data.end()) ? p->second : "";
  }


  // 登録データの更新
  void setData(DataMap& data, const std::string& key, const std::string& name) {
    data[key] = name;
  }
};


ArchiveConfig::ArchiveConfig(const std::string& archive_file)
  : pimpl(new pImpl(archive_file)) {
}


ArchiveConfig::~ArchiveConfig(void) {
}


// リリース時のファイル名を取得
const std::string ArchiveConfig::getReleaseName(void) const {

  return (pimpl->release_file_.empty()) ? "" :
    pimpl->dir_path_ + pimpl->release_file_;
}


// リリースファイルのディレクトリ名を取得
const std::string ArchiveConfig::getReleaseDirectoryName(void) const {

  return pimpl->dir_path_;
}


// リリース時のファイル名を設定
void ArchiveConfig::setReleaseName(const std::string& release_file) {

  pimpl->release_file_ = release_file;
}


// RRDA キーの取得
const std::string ArchiveConfig::getRrdaKey(void) const {

  return pimpl->rrda_key_;
}


// RRDA キーの設定
void ArchiveConfig::setRrdaKey(const std::string& rrda_key) {

  pimpl->rrda_key_ = rrda_key;
}


// 通常データ名の取得
const std::string ArchiveConfig::getNormalData(const std::string& key) const {

  std::string name = pimpl->getData(pimpl->normal_data_, key);
  return name.empty() ? "" : pimpl->dir_path_ + name;
}


// 通常データ名の登録
void ArchiveConfig::setNormalData(const std::string& key,
                                  const std::string& name) {
  pimpl->setData(pimpl->normal_data_, key, name);
}


// 制限データ名の取得
const std::string ArchiveConfig::getRestrictData(const std::string& key) const {

  std::string name = pimpl->getData(pimpl->restricted_data_, key);
  return name.empty() ? "" : pimpl->dir_path_ + name;
}


// 制限データ名の設定
void ArchiveConfig::setRestrictData(const std::string& key,
                                    const std::string& name) {
  pimpl->setData(pimpl->restricted_data_, key, name);
}


// 保存
bool ArchiveConfig::save(void) {
  return pimpl->saveArchiveFile(pimpl->archive_file_);
}


// 名前を付けて保存
bool ArchiveConfig::saveAs(const std::string& name) {
  return pimpl->saveArchiveFile(name);
}


// 保存するファイル名の読み出し
const std::string ArchiveConfig::getArchiveFileName(void) const {

  return pimpl->archive_file_;
}
