/*!
  \file
  \brief テーマ時計の部品描画

  \author Satofumi KAMIMURA

  $Id: ClockPartsDraw.cpp 283 2011-07-03 19:09:08Z satofumi $
*/

#include "ThemeResource.h"
#include "ThemeClockDraw.h"
#include "AnimationPattern.h"
#include <QPainter>
#include <QBitmap>
#include <QTime>
#include <cmath>
#include <cstdio>

using namespace std;


namespace
{
  static const bool CenterPosition = true; // 画像中心を基準にした描画を指示

  static int last_cell_index_ = ClockPartsDraw::NoAnimationCell;

  class SecondPixmap
  {
  public:
    qint64 cache_key_;
    size_t scaling_percent_;
    QPixmap pixmap_;
    size_t size_width_;
    size_t size_height_;

    SecondPixmap(void) : cache_key_(0), scaling_percent_(0)
    {
    }
  };


  // 画面に描画するよりもコピーの方が時間コストが安いため、コピーで対処する
  class DrawPartsData
  {
  public:
    QPixmap pixmap_;
    QRect draw_rect_;
    QRect source_rect_;

    DrawPartsData(const QPixmap& pixmap,
                  const QRect& draw_rect, const QRect& source_rect)
      : pixmap_(pixmap), draw_rect_(draw_rect), source_rect_(source_rect)
    {
    }
  };
  static vector<DrawPartsData> draw_parts_;
  static bool is_drawn_ = false;


  // 位置の取得
  QRect getPosition(const ThemeResource& resource,
                    const string& tag_name)
  {

    return resource.offset.value(tag_name, QRect(0, 0, 0, 0));
  }


  // 画像の描画
  void drawPixmap(QRegion& region,
                  const ThemeResource& resource,
                  const string& tag_name,
                  const QRect& offset,
                  size_t scaling_percent,
                  qreal rotate_degree = 0.0,
                  bool center_position = false,
                  bool erase_bottom_line = false)
  {
    // ローカルのキャッシュサイズを確保
    static QPixmap no_second_pixmap;
    static vector<SecondPixmap> second_pixmaps;
    if (second_pixmaps.empty()) {
      second_pixmaps.resize(60);
    }

    QPixmap* pixmap = resource.pixmap.value(tag_name, NULL);
    if (! pixmap) {
      return;
    }

    // キャッシュ対象のタグ名
    static const char* CacheTag = "analog/hari_s";

    // 秒の画像がキャッシュにヒットしたら、画像は再生成しない
    int size_width = 0, size_height = 0;
    QPixmap* converted_pixmap = NULL;
    qint64 pixmap_cache_key = pixmap->cacheKey();
    int second_index = static_cast<int>(60 * rotate_degree / 360.0);
    if ((! tag_name.compare(CacheTag)) &&
        (second_pixmaps[second_index].cache_key_ == pixmap_cache_key) &&
        (second_pixmaps[second_index].scaling_percent_ == scaling_percent)) {
      // キャッシュにヒット。画像生成は行わない
      converted_pixmap = &second_pixmaps[second_index].pixmap_;
      size_width = second_pixmaps[second_index].size_width_;
      size_height = second_pixmaps[second_index].size_height_;
    }

    // 指定割合の回転画像を生成 (キャッシュされていない場合の処理)
    if (! converted_pixmap) {
      QMatrix matrix;
      matrix = matrix.rotate(rotate_degree);
      QPixmap rotated_pixmap =
        pixmap->transformed(matrix, Qt::SmoothTransformation);
      if (rotated_pixmap.isNull()) {
        return;
      }

      QSize size = rotated_pixmap.size();
      size_width = size.width();
      size_height = size.height();
      int scaled_x =
        static_cast<int>(round(size_width * scaling_percent / 100.0));
      int scaled_y =
        static_cast<int>(round(size_height * scaling_percent / 100.0));

      if (! tag_name.compare(CacheTag)) {
        // キャッシュを更新
        second_pixmaps[second_index].pixmap_ =
          rotated_pixmap.scaled(scaled_x, scaled_y,
                                Qt::IgnoreAspectRatio,
                                Qt::SmoothTransformation);
        converted_pixmap = &second_pixmaps[second_index].pixmap_;
        second_pixmaps[second_index].cache_key_ = pixmap->cacheKey();
        second_pixmaps[second_index].scaling_percent_ = scaling_percent;
        second_pixmaps[second_index].size_width_ = size_width;
        second_pixmaps[second_index].size_height_ = size_height;

      } else {
        // キャッシュ対象以外の場合
        no_second_pixmap = rotated_pixmap.scaled(scaled_x, scaled_y,
                                                 Qt::IgnoreAspectRatio,
                                                 Qt::SmoothTransformation);
        converted_pixmap = &no_second_pixmap;
      }
    }

    // 描画位置の計算
    int x, y;
    if (center_position) {
      x = static_cast<int>(round(-(size_width / 2.0 * scaling_percent / 100.0)
                                 + (offset.x() * scaling_percent / 100.0)));
      y = static_cast<int>(round(-(size_height / 2.0 * scaling_percent / 100.0)
                                 + (offset.y() * scaling_percent / 100.0)));
    } else {
      x = static_cast<int>(round(offset.x() * scaling_percent / 100.0));
      y = static_cast<int>(round(offset.y() * scaling_percent / 100.0));
    }
    QSize size = converted_pixmap->size();
    int w = size.width();
    int h = size.height();
    int fixed_h = (erase_bottom_line) ? (h - 1) : h;
    draw_parts_.push_back(DrawPartsData(*converted_pixmap,
                                        QRect(x, y, w, fixed_h),
                                        QRect(0, 0, w, fixed_h)));
    is_drawn_ = false;

    // 描画領域の更新
    QRegion parts_region = converted_pixmap->mask();
    parts_region.translate(x, y);
    region += parts_region;
  }


  // アナログ部分の再描画
  void updateAnalog(QRegion& region, const ThemeResource& resource,
                    int hour, int minute, int second, size_t scaling_percent)
  {
    // 針の角度を再計算
    qreal analog_h_degree = (360.0 * (hour / 12.0)) + (30.0 * (minute / 60.0));
    QRect analog_h_offset = getPosition(resource, "analog/hari_h");
    drawPixmap(region, resource, "analog/hari_h", analog_h_offset,
               scaling_percent, analog_h_degree, CenterPosition);

    qreal analog_m_degree = 360.0 * minute / 60.0;
    QRect analog_m_offset = getPosition(resource, "analog/hari_m");
    drawPixmap(region, resource, "analog/hari_m", analog_m_offset,
               scaling_percent, analog_m_degree, CenterPosition);

    if (! resource.option.value("analog/sec", "").compare("on")) {
      // 秒針は、指定があるときのみ描画する
      qreal analog_s_degree = 360.0 * second / 60.0;
      QRect analog_s_offset = getPosition(resource, "analog/hari_s");
      drawPixmap(region, resource, "analog/hari_s", analog_s_offset,
                 scaling_percent, analog_s_degree, CenterPosition);
    }
  }


  // デジタル部分の再描画
  void updateDigital(QRegion& region,
                     const ThemeResource& resource,
                     const QVector<QPoint>& digital_positions,
                     int hour, int minute, int second, size_t scaling_percent)
  {
    enum { BufferSize = 11 };
    char buffer[BufferSize] = "hh:mm ss x";

    if (! resource.option.value("digital/sec", "").compare("on")) {
      // 秒を描画する
      snprintf(buffer, BufferSize, "%02d:%02d_%02d_%c",
               hour, minute, second, (hour >= 12) ? 'P' : 'A');
    } else {
      // 秒は描画しない
      snprintf(buffer, BufferSize, "%02d:%02d____%c",
               hour, minute, (hour >= 12) ? 'P' : 'A');
    }

    if (! resource.option.value("digital/ampm", "").compare("off")) {
      // AM / PM を表示しない
      buffer[9] = '_';
    }

    if (resource.option.value("digital/colon", "").compare("on")) {
      // コロンを表示しない
      buffer[2] = ' ';

    } else if (! resource.option.value("digital/blink", "").compare("on")) {
      // コロンを点滅させる
      if (second & 0x01) {
        buffer[2] = ' ';
      }
    }

    // 描画処理
    char digital_char[] = "_";
    size_t n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      digital_char[0] = buffer[i];
      QPixmap* parts = resource.pixmap.value(digital_char, NULL);
      if (! parts) {
        continue;
      }

      QSize size = parts->size();
      int x = digital_positions[i].x();
      int y = digital_positions[i].y();
      drawPixmap(region, resource, digital_char,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }
  }


  // カレンダー部品の描画
  void updateCalendar(QRegion& region, const ThemeResource& resource,
                      const QVector<QPoint>& calendar_positions,
                      int month, int day, int day_of_week,
                      size_t scaling_percent)
  {
    enum { BufferSize = 5 };
    char buffer[BufferSize] = "mmdd";

    bool calendar_month =
      ! resource.option.value("calendar/month", "").compare("on");
    bool zero_digit =
      ! resource.option.value("calendar/zero", "").compare("on");
    if (calendar_month) {
      // 月を描画する
      if (! zero_digit) {
        // 10 の位がゼロのときに表示しない
        snprintf(buffer, BufferSize, "%2d%2d", month, day);
      } else {
        snprintf(buffer, BufferSize, "%02d%02d", month, day);
      }
    } else {
      // 月は描画しない
      if (! zero_digit) {
        // 10 の位がゼロのときに表示しない
        snprintf(buffer, BufferSize, "__%2d", day);
      } else {
        snprintf(buffer, BufferSize, "__%02d", day);
      }
    }

    // セパレータの描画
    const char* separator_tags[] = { "MONTH", "DAY" };
    size_t n = sizeof(separator_tags) / sizeof(separator_tags[0]);
    for (int i = 0; i < 2; ++i) {
      if ((i == 0) && (! calendar_month)) {
        // !!! 月を表示しないときは、っていう意味だが、わかりにくい
        // !!! どうにかすべき
        continue;
      }

      const char* tag = separator_tags[i];
      QPixmap* parts = resource.pixmap.value(tag, NULL);
      if (! parts) {
        continue;
      }

      // !!! マジックナンバー 4 をなんとかする
      QSize size = parts->size();
      int x = calendar_positions[4 + i].x();
      int y = calendar_positions[4 + i].y();
      drawPixmap(region, resource, tag,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }

    // 曜日の描画
    const char* week_names[] = {
      "SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT", "SUN",
    };
    const char* tag = week_names[day_of_week];
    QPixmap* parts = resource.pixmap.value(tag, NULL);
    if (parts) {
      // !!! マジックナンバー 6 をなんとかする
      QSize size = parts->size();
      int x = calendar_positions[6].x();
      int y = calendar_positions[6].y();
      drawPixmap(region, resource, tag,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }

    // 数値の描画処理
    char number_char[] = "Cx";
    n = strlen(buffer);
    for (size_t i = 0; i < n; ++i) {
      number_char[1] = buffer[i];
      QPixmap* parts = resource.pixmap.value(number_char, NULL);
      if (! parts) {
        continue;
      }

      QSize size = parts->size();
      int x = calendar_positions[i].x();
      int y = calendar_positions[i].y();
      drawPixmap(region, resource, number_char,
                 QRect(x, y, size.width(), size.height()), scaling_percent);
    }
  }


  // 追加部品の描画
  void updateAdditional(QRegion& region, const ThemeResource& resource,
                        size_t scaling_percent)
  {
    QRect additional_offset = getPosition(resource, "additional/near");
    drawPixmap(region, resource, "additional/near",
               additional_offset, scaling_percent);
  }


  // アニメーションセルの描画
  void updateAnimationCell(QRegion& region, const ThemeResource& resource,
                           size_t cell_index, size_t scaling_percent)
  {
    if (last_cell_index_ == ClockPartsDraw::NoAnimationCell) {
      return;
    }

    AnimationCellTag cell_tags(cell_index);
    QRect cell_offset = getPosition(resource, cell_tags.cell_tag);
    drawPixmap(region, resource, cell_tags.cell_tag,
               cell_offset, scaling_percent, 0.0, false, true);
  }
}


// 再描画
void ClockPartsDraw::updatePixmaps(QRegion& region,
                                   const ThemeResource& resource,
                                   const QVector<QPoint>& digital_positions,
                                   const QVector<QPoint>& calendar_positions,
                                   size_t scaling_percent, int sec_offset,
                                   int cell_index)
{
  // 描画部品をクリア
  draw_parts_.clear();

  // ベース画像の描画
  QRect offset = getPosition(resource, "base/base_org");
  drawPixmap(region, resource, "base/base_org", offset, scaling_percent);

  // 部品画像の描画
  QTime current_time = QTime::currentTime();
  QDateTime current_date = QDateTime(QDate::currentDate(), current_time);
  QTime time = current_time.addSecs(sec_offset);
  QDate date = (current_date.addSecs(sec_offset)).date();

  // アニメーションセルの更新
  if ((cell_index >= 0) || (cell_index == InvalidCellIndex)) {
    // 明示的に指定されない限り、前回指定されたセルを描画し続ける
    if (cell_index >= 0) {
      last_cell_index_ = cell_index;
    }
    updateAnimationCell(region, resource, last_cell_index_, scaling_percent);

  } else {
    // 何も描画させない
    last_cell_index_ = cell_index;
  }

  // デジタル文字を手前に表示するかどうか
  bool digital_above =
    ! resource.option.value("digital/above", "off").compare("on");

  if (! digital_above) {
    updateCalendar(region, resource, calendar_positions,
                   date.month(), date.day(), date.dayOfWeek(), scaling_percent);
    updateDigital(region, resource, digital_positions,
                  time.hour(), time.minute(), time.second(), scaling_percent);
  }
  updateAnalog(region, resource,
               time.hour(), time.minute(), time.second(), scaling_percent);
  updateAdditional(region, resource, scaling_percent);
  if (digital_above) {
    updateCalendar(region, resource, calendar_positions,
                   date.month(), date.day(), date.dayOfWeek(), scaling_percent);
    updateDigital(region, resource, digital_positions,
                  time.hour(), time.minute(), time.second(), scaling_percent);
  }
}


// 画像部品の描画
void ClockPartsDraw::drawPixmaps(QPainter& painter, bool force_draw)
{
  if (! force_draw) {
    if (draw_parts_.empty() || is_drawn_) {
      // 描画すべき部品が何もなければ、戻る
      return;
    }
  }

  painter.setRenderHint(QPainter::SmoothPixmapTransform);

  // 実際の部品描画
  for (vector<DrawPartsData>::iterator it = draw_parts_.begin();
       it != draw_parts_.end(); ++it) {

    QPixmap& pixmap = it->pixmap_;
    painter.drawPixmap(it->draw_rect_, pixmap, it->source_rect_);
  }

  // 描画済み
  is_drawn_ = true;
}
