#ifndef ARCHIVE_CONFIG_H
#define ARCHIVE_CONFIG_H

/*!
  \file
  \brief 設定ファイルの管理

  \author Satofumi KAMIMURA

  $Id: ArchiveConfig.h 262 2009-01-30 13:25:37Z satofumi $
*/

#include <memory>
#include <string>


/*!
  \brief 設定ファイルの管理クラス
*/
class ArchiveConfig
{
  ArchiveConfig(void);
  ArchiveConfig(const ArchiveConfig& rhs);
  ArchiveConfig& operator = (const ArchiveConfig& rhs);

  struct pImpl;
  std::auto_ptr<pImpl> pimpl;

 public:

  /*!
    \brief コンストラクタ

    \param[in] archive_file 読み出すファイル名
  */
  explicit ArchiveConfig(const std::string& archive_file);
  ~ArchiveConfig(void);


  /*!
    \brief リリースファイル名の取得

    \return リリースファイル名
  */
  const std::string getReleaseName(void) const;


  /*!
    \brief リリースファイルのディレクトリ名を取得

    \return リリースファイルのディレクトリ名
  */
  const std::string getReleaseDirectoryName(void) const;


  /*!
    \brief リリースファイル名の設定

    \param[in] release_file リリースファイル名
  */
  void setReleaseName(const std::string& release_file);


  /*!
    \brief RRDA キーの取得

    \retval RRDA キー
  */
  const std::string getRrdaKey(void) const;


  /*!
    \brief RRDA キー

    \param[in] rrda_key RRDA キー
  */
  void setRrdaKey(const std::string& rrda_key);


  /*!
    \brief データ名の取得

    \param[in] key キー

    \retval データファイル名
  */
  const std::string getNormalData(const std::string& key) const;


  /*!
    \brief データ名の設定

    \param[in] key キー
    \param[in] name データファイル名
  */
  void setNormalData(const std::string& key, const std::string& name);


  /*!
    \brief データ名の取得 (制限データ)

    \param[in] key キー

    \retval データファイル名
  */
  const std::string getRestrictData(const std::string& key) const;


  /*!
    \brief データ名の設定 (制限データ)

    \param[in] key キー
    \param[in] name データファイル名
  */
  void setRestrictData(const std::string& key, const std::string& name);


  /*!
    \brief アーカイブファイルの更新

    \retval true 正常
    \retval false エラー
  */
  bool save(void);


  /*!
    \brief アーカイブファイル名を変更して更新

    \param[in] name アーカイブファイル名

    \retval true 正常
    \retval false エラー
  */
  bool saveAs(const std::string& name);


  /*!
    \brief 現在のアーカイブファイル名を取得

    \retval ! NULL アーカイブファイル名
    \retval NULL アーカイブファイル名なし
  */
  const std::string getArchiveFileName(void) const;
};

#endif /* !ARCHIVE_CONFIG_H */
