package junit.extensions.eclipse.quick;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.actions.SelectionConverter;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.ui.PartInitException;

public class OpenTestingPairAction extends QuickJUnitAction {

    private IType getTypeOfJavaEditor() throws JavaModelException {
        if (javaEditor == null)
            return null;
        return SelectionConverter.getTypeAtOffset(javaEditor);
    }

    private IType getTypeOfJavaElement() {
        if (javaElement == null)
            return null;
        ICompilationUnit cu = null;
        if (javaElement instanceof ICompilationUnit) {
            cu = (ICompilationUnit) javaElement;
        } else if (javaElement instanceof IMember) {
            cu = ((IMember) javaElement).getCompilationUnit();
        }
        return cu != null ? cu.findPrimaryType() : null; 
    }

    private static class TypeLabelProvider extends LabelProvider {
        public String getText(Object element) {
            return ((IType) element).getFullyQualifiedName();
        }
    }

    private IType selectType(IAction action, List types) {
        PopupTableSelector selector = new PopupTableSelector(getShell(), types);
        selector.setTitle(action.getText());
        selector.setCommandForward(action.getActionDefinitionId());
        selector.setLabelProvider(new TypeLabelProvider());
        return (IType) selector.select();
    }

    public void run(IAction action) {
        try {
            IType targetType = getTypeOfJavaEditor();
            if (targetType == null)
                targetType = getTypeOfJavaElement();
            if (targetType == null)
                return;

            IJavaProject[] projects = getJavaProjects();
            List testPairTypes = findPairTypes(createTestPair(), projects, targetType);
            if (testPairTypes.isEmpty()) {
                openInformation(action, Messages.getString("OpenTestingPairAction.testPairNotFound", targetType.getFullyQualifiedName())); //$NON-NLS-1$
                return;
            } else {
                IType selectedType = null;
                if (testPairTypes.size() >= 2) {
                    selectedType = selectType(action, testPairTypes);
                } else {
                    selectedType = (IType) testPairTypes.get(0);
                }
                if (selectedType != null)
                    JavaUI.openInEditor(selectedType);    
            }
        } catch (JavaModelException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        } catch (PartInitException e) {
            QuickJUnitPlugin.getDefault().handleSystemError(e, this);
        }
    }

    private TestingPair createTestPair() {
        TestingPair testPair = new TestingPair();
        IPreferenceStore store = QuickJUnitPlugin.getDefault().getPreferenceStore();
        String[] rules = new NamingRules(store).getEnableValues();
        for (int i = 0; i < rules.length; ++i) {
            testPair.addNamingRule(rules[i]);            
        }
        return testPair;
    }

    private List findPairTypes(TestingPair testPair, IJavaProject[] projects, IType type) throws JavaModelException {
        Set result = new LinkedHashSet();
        String[] pairNames = testPair.getPairClassNames(type.getFullyQualifiedName());
        for (int i = 0; i < projects.length; ++i) {
            IJavaProject project = projects[i];
            for (int j = 0; j < pairNames.length; ++j) {
                IType pairType= project.findType(pairNames[j]);
                if (pairType != null) {
                    result.add(pairType);
                }
            }
        }
        return new ArrayList(result);
    }
}
