package junit.extensions.eclipse.quick;

import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.jdt.internal.junit.util.SWTUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

public class QuickJUnitPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

    private List namingRulesValue;
    private NamingRules namingRules;

	private CheckboxTableViewer namingRuleViewer;
    private Button addNamingRuleButton;
    private Button removeNamingRuleButton;
    private Button editNamingRuleButton;
    private Button moveUpNamingRuleButton;
    private Button moveDownNamingRuleButton;

    public QuickJUnitPreferencePage() {
        setPreferenceStore(QuickJUnitPlugin.getDefault().getPreferenceStore());
    }

    public void init(IWorkbench workbench)  {
        IPreferenceStore store = getPreferenceStore();
        namingRules = new NamingRules(store);
        namingRulesValue = namingRules.get();
	}

	protected Control createContents(Composite parent)  {
        Composite composite= new Composite(parent, SWT.NULL);
        GridLayout layout= new GridLayout();
        layout.numColumns= 1;
        layout.marginHeight= 0;
        layout.marginWidth= 0;
        composite.setLayout(layout);
        GridData data= new GridData();
        data.verticalAlignment= GridData.FILL;
        data.horizontalAlignment= GridData.FILL;
        composite.setLayoutData(data);

        createNamingRulePreferences(composite);
        Dialog.applyDialogFont(composite);
        return composite;
	}

    private void createNamingRulePreferences(Composite composite) {
        Composite container= new Composite(composite, SWT.NONE);
        GridLayout layout= new GridLayout();
        layout.numColumns= 2;
        layout.marginHeight= 0;
        layout.marginWidth= 0;
        container.setLayout(layout);
        GridData gd= new GridData(GridData.FILL_BOTH);
        container.setLayoutData(gd);
        createNamingRuleTable(container);
        createNamingRuleButtons(container);
        updateNamingRuleViewer();
    }

    private void createNamingRuleButtons(Composite container) {
        Composite buttonContainer= new Composite(container, SWT.NONE);
        GridData gd= new GridData(GridData.FILL_VERTICAL);
        buttonContainer.setLayoutData(gd);
        GridLayout buttonLayout= new GridLayout();
        buttonLayout.numColumns= 1;
        buttonLayout.marginHeight= 0;
        buttonLayout.marginWidth= 0;
        buttonContainer.setLayout(buttonLayout);

        Listener listener;

        listener = new Listener() {
            public void handleEvent(Event e) {
                addNamingRule();
            }};
        addNamingRuleButton = createButton("addNamingRuleButton", buttonContainer, listener, true); //$NON-NLS-1$

        listener = new Listener() {
            public void handleEvent(Event e) {
                removeNamingRules();
            }};
        removeNamingRuleButton = createButton("removeNamingRuleButton", buttonContainer, listener, false); //$NON-NLS-1$
        removeNamingRuleButton.setEnabled(false);

        listener = new Listener() {
            public void handleEvent(Event e) {
                editNamingRule();
            }};
        editNamingRuleButton = createButton("editNamingRuleButton", buttonContainer, listener, false); //$NON-NLS-1$
        editNamingRuleButton.setEnabled(false);

        listener = new Listener() {
            public void handleEvent(Event e) {
                moveNamingRule(true);
            }};
        moveUpNamingRuleButton = createButton("moveUpNamingRuleButton", buttonContainer, listener, false); //$NON-NLS-1$
        moveUpNamingRuleButton.setEnabled(false);

        listener = new Listener() {
            public void handleEvent(Event e) {
                moveNamingRule(false);
            }};
        moveDownNamingRuleButton = createButton("moveDownNamingRuleButton", buttonContainer, listener, false); //$NON-NLS-1$
        moveDownNamingRuleButton.setEnabled(false);

    }

    private Button createButton(String buttonId, Composite buttonContainer, Listener listener, boolean isTop) {
        Button button= new Button(buttonContainer, SWT.PUSH);
        button.setText(Messages.getString("QuickJUnitPreferencePage." + buttonId + ".label")); //$NON-NLS-1$ //$NON-NLS-2$
        button.setToolTipText(Messages.getString("QuickJUnitPreferencePage." + buttonId + ".tooltip")); //$NON-NLS-1$ //$NON-NLS-2$
        GridData gd;
        if (isTop)
            gd = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
        else
            gd = getButtonGridData(button);
        button.setLayoutData(gd);
        SWTUtil.setButtonDimensionHint(button);
        button.addListener(SWT.Selection, listener);
        return button;
    }

    private void addNamingRule() {
        InputDialog dialog = createNamingRuleEditDialog("addNamingRule", ""); //$NON-NLS-1$ //$NON-NLS-2$
        if (dialog.open() == Window.OK) {
            String value = dialog.getValue();
            if (value.trim().length() != 0) {
                NamingRule rule = new NamingRule(value, true);
                namingRulesValue.add(rule);
                updateNamingRuleViewer();
            }
        }
    }

    private void removeNamingRules() {
        IStructuredSelection selection = (IStructuredSelection) namingRuleViewer.getSelection();
        if (selection.isEmpty())
            return;
        for (Iterator i = selection.iterator(); i.hasNext(); ) {
             namingRulesValue.remove(i.next());
        }
        updateNamingRuleViewer();
    }

    private static class NamingRuleValidator implements IInputValidator {
        public String isValid(String newText) {
            newText = newText.trim();
            if (newText.length() == 0)
                return Messages.getString("QuickJUnitPreferencePage.namingRuleValidator.empty"); //$NON-NLS-1$

            newText = newText.replaceAll("\\$\\{package\\}", "package"); //$NON-NLS-1$ //$NON-NLS-2$
            newText = newText.replaceAll("\\$\\{type\\}", "type"); //$NON-NLS-1$ //$NON-NLS-2$

            StringTokenizer st = new StringTokenizer(newText, ".", true); //$NON-NLS-1$
            boolean dot = false;
            while(st.hasMoreTokens()) {
                String token = st.nextToken();
                if (dot) {
                    if (!token.equals(".")) { //$NON-NLS-1$
                        return Messages.getString("QuickJUnitPreferencePage.namingRuleValidator.error"); //$NON-NLS-1$
                    }
                    dot = false;
                } else {
                    if (!isJavaIdentifier(token))
                        return Messages.getString("QuickJUnitPreferencePage.namingRuleValidator.tokenError", token); //$NON-NLS-1$
                    dot = true;
                }
            }
            return null;
        }
        private boolean isJavaIdentifier(String token) {
            if (!Character.isJavaIdentifierStart(token.charAt(0)))
                return false;
            for (int i = 1; i < token.length(); ++i) {
                if (!Character.isJavaIdentifierPart(token.charAt(i)))
                return false;
            }
            return true;
        }
    }

    private void editNamingRule() {
        IStructuredSelection selection = (IStructuredSelection) namingRuleViewer.getSelection();
        if (selection.isEmpty())
            return;
        NamingRule rule = (NamingRule) selection.getFirstElement();
        InputDialog dialog = createNamingRuleEditDialog("editNamingRule", rule.getValue()); //$NON-NLS-1$
        if (dialog.open() == Window.OK) {
            String value = dialog.getValue();
            if (value.trim().length() != 0) {
                rule.setValue(value);
                updateNamingRuleViewer();
            }
        }
    }

    private InputDialog createNamingRuleEditDialog(String messageId, String initValue) {
        String title = Messages.getString("QuickJUnitPreferencePage." + messageId + ".dialog.title"); //$NON-NLS-1$ //$NON-NLS-2$
        String message = Messages.getString("QuickJUnitPreferencePage." + messageId + ".dialog.message"); //$NON-NLS-1$ //$NON-NLS-2$
        return  new InputDialog(getShell(), title, message, initValue, new NamingRuleValidator());
    }

    protected void moveNamingRule(boolean up) {
        IStructuredSelection selection = (IStructuredSelection) namingRuleViewer.getSelection();
        if (selection.isEmpty() || selection.size() > 1)
            return;
        Object selected = selection.getFirstElement();
        int oldIndex = namingRulesValue.indexOf(selected);
        int newIndex = up ? oldIndex - 1: oldIndex + 1;
        if (newIndex < 0 || namingRulesValue.size() <= newIndex)
            return;
        namingRulesValue.remove(oldIndex);
        namingRulesValue.add(newIndex, selected);
        updateNamingRuleViewer();
    }

    private GridData getButtonGridData(Button button) {
        GridData gd= new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
        int widthHint= convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
        gd.widthHint= Math.max(widthHint, button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true).x);
        gd.heightHint= convertVerticalDLUsToPixels(IDialogConstants.BUTTON_HEIGHT);
        return gd;
    }

    private void createNamingRuleTable(Composite container) {
        Label tableLabel= new Label(container, SWT.NONE);
        tableLabel.setText(Messages.getString("QuickJUnitPreferencePage.namingRule.label")); //$NON-NLS-1$
        GridData gd = new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING);
        gd.horizontalSpan = 2;
        tableLabel.setLayoutData(gd);

        Table namingRuleTable = new Table(container, SWT.CHECK | SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);

        gd= new GridData(GridData.FILL_HORIZONTAL);
        namingRuleTable.setLayoutData(gd);

        TableLayout tableLayout= new TableLayout();
        ColumnLayoutData[] columnLayoutData= new ColumnLayoutData[1];
        columnLayoutData[0]= new ColumnWeightData(100);
        tableLayout.addColumnData(columnLayoutData[0]);
        namingRuleTable.setLayout(tableLayout);
        new TableColumn(namingRuleTable, SWT.NONE);
        namingRuleViewer = new CheckboxTableViewer(namingRuleTable);
        namingRuleViewer.setLabelProvider(new NamingRuleTableLabelProvider());
        namingRuleViewer.setContentProvider(new NamingRuleTableContentProvider());
        namingRuleViewer.setInput(this);

        gd = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL);
        namingRuleViewer.getTable().setLayoutData(gd);
        namingRuleViewer.addCheckStateListener(new ICheckStateListener() {
            public void checkStateChanged(CheckStateChangedEvent event) {
                NamingRule namingRule = (NamingRule) event.getElement();
                namingRule.setEnabled(event.getChecked());
                updateNamingRuleViewer();
            }
        });
        namingRuleViewer.addSelectionChangedListener(new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
                // updateNmingRuleViewerĂԂƁC`FbN{bNX̃IEIt
                // łȂȂĂ܂D
                updateNamingRuleButtons();
            }
        });
        namingRuleViewer.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                editNamingRule();
            }
        });
    }

    private void updateNamingRuleViewer() {
        namingRuleViewer.refresh();
        for (int i = 0; i < namingRulesValue.size(); ++i) {
            NamingRule namingRule = (NamingRule) namingRulesValue.get(i);
            namingRuleViewer.setChecked(namingRule, namingRule.isEnabled());
        }
        updateNamingRuleButtons();
    }

    private void updateNamingRuleButtons() {
        IStructuredSelection selection = (IStructuredSelection) namingRuleViewer.getSelection();
        removeNamingRuleButton.setEnabled(!selection.isEmpty());
        editNamingRuleButton.setEnabled(selection.size() == 1);
        int rowCount = namingRuleViewer.getTable().getItemCount();
        boolean canMove = selection.size() == 1 &&  rowCount > 1;
        if (!canMove) {
            moveUpNamingRuleButton.setEnabled(false);
            moveDownNamingRuleButton.setEnabled(false);
        } else {
           int selectedIndex = namingRulesValue.indexOf(selection.getFirstElement());
           moveUpNamingRuleButton.setEnabled(0 < selectedIndex);
           moveDownNamingRuleButton.setEnabled(selectedIndex < rowCount - 1);
        }
    }

    private static class NamingRuleTableLabelProvider extends LabelProvider implements ITableLabelProvider {
        public String getColumnText(Object o, int column) {
            return column == 0 ? ((NamingRule) o).getValue() : ""; //$NON-NLS-1$
        }
        public String getText(Object element) {
            return ((NamingRule) element).getValue();
        }
        public Image getColumnImage(Object element, int columnIndex) {
            return null;
        }
    }
    private class NamingRuleTableContentProvider implements IStructuredContentProvider {
        public Object[] getElements(Object inputElement) {
            return namingRulesValue.toArray();
        }
        public void dispose() {
        }
        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }
    protected void performDefaults() {
        super.performDefaults();
        namingRulesValue = namingRules.getDefault();
        updateNamingRuleViewer();
    }

    public boolean performOk() {
        namingRules.set(namingRulesValue);
        QuickJUnitPlugin.getDefault().savePluginPreferences();
        return true;
    }
}
