#include <glib.h>
#include <SDL.h>
#include "racanhack.h"
#include "cave.h"

enum {
  CAVE_LX = 2,
  CAVE_LY = 2,
  CAVE_HX = MAP_W - 2,
  CAVE_HY = MAP_H - 2,
  MINIMUM_ROOM_SIZE = 4,
  MARGIN_BETWEEN_RECT_ROOM = 2,
  MINIMUM_RECT_SIZE = MINIMUM_ROOM_SIZE + (MARGIN_BETWEEN_RECT_ROOM * 2),
  JUNCTION_VERTICAL = 0,
  JUNCTION_HORIZONAL = 1
};

static struct _cave *cave;
static void line_map_pass(int x0, int y0, int x1, int y1);
static void line_waynavi(int x0, int y0, int x1, int y1);
static void cave_rect_split(struct _rect *rect_parent);
static void cave_junction_more();
static void cave_room_make();
static void cave_junction_make();
static void cave_map_set();
static void cave_map_print_debug();
static struct _rect *cave_rect_add(int lx, int ly, int hx, int hy);
static struct _junction *cave_junction_add(int v_or_h, struct _rect *rect0, struct _rect *rect1);
static struct _room *cave_room_add(int lx, int ly, int hx, int hy, gboolean route);

static void line_map_pass(int x0, int y0, int x1, int y1)
{
  int min_x, max_x, min_y, max_y, i, j;
  min_x = MIN(x0, x1);
  max_x = MAX(x0, x1);
  min_y = MIN(y0, y1);
  max_y = MAX(y0, y1);
  g_assert((min_x >= 0) && (max_x < MAP_W) && (min_y >= 0) && (max_y < MAP_H));
  if ((x0 <= x1) && (y0 >= y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][max_y].pass = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[max_x][j].pass = TRUE;
    return;
  };
  if ((x0 > x1) && (y0 > y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][min_y].pass = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[max_x][j].pass = TRUE;
    return;
  };
  if ((x0 > x1) && (y0 <= y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][min_y].pass = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[min_x][j].pass = TRUE;
    return;
  };
  if ((x0 <= x1) && (y0 < y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][max_y].pass = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[min_x][j].pass = TRUE;
    return;
  };
}

static void line_waynavi(int x0, int y0, int x1, int y1)
{
  int min_x, max_x, min_y, max_y, i, j;
  min_x = MIN(x0, x1);
  max_x = MAX(x0, x1);
  min_y = MIN(y0, y1);
  max_y = MAX(y0, y1);
  g_assert((min_x >= 0) && (max_x < MAP_W) && (min_y >= 0) && (max_y < MAP_H));
  if ((x0 <= x1) && (y0 >= y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][max_y].waynavi = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[max_x][j].waynavi = TRUE;
    return;
  };
  if ((x0 > x1) && (y0 > y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][min_y].waynavi = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[max_x][j].waynavi = TRUE;
    return;
  };
  if ((x0 > x1) && (y0 <= y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][min_y].waynavi = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[min_x][j].waynavi = TRUE;
    return;
  };
  if ((x0 <= x1) && (y0 < y1)) {
    for (i = min_x; i <= max_x; i++) cave->map[i][max_y].waynavi = TRUE;
    for (j = min_y; j <= max_y; j++) cave->map[min_x][j].waynavi = TRUE;
    return;
  };
}

void cave_generate_tutorial(struct _cave *c)
{
  int i, j;
  struct _room *room;
  cave = c;
  for (i = 0; i < MAP_W; i++) {
    for (j = 0; j < MAP_H; j++) {
      cave->map[i][j].pass = FALSE;
      cave->map[i][j].waynavi = FALSE;
      cave->map[i][j].passed = FALSE;
      cave->map[i][j].disturb = FALSE;
      cave->map[i][j].room = NULL;
    };
  };
  cave->rect_header = NULL;
  cave->junction_header = NULL;
  cave->room_header = NULL;
  room = cave_room_add(CAVE_LX, CAVE_LY, 10, 10, FALSE);
  for (i = room->lx; i <= room->hx; i++) {
    for (j = room->ly; j <= room->hy; j++) {
      cave->map[i][j].pass = TRUE;
      cave->map[i][j].room = room;
    };
  };
}

void cave_free(struct _cave *c)
{
  g_list_foreach(c->rect_header, (GFunc)my_free, NULL);
  g_list_free(c->rect_header);
  g_list_foreach(c->room_header, (GFunc)my_free, NULL);
  g_list_free(c->room_header);
  g_list_foreach(c->junction_header, (GFunc)my_free, NULL);
  g_list_free(c->junction_header);
}

void cave_generate(struct _cave *c)
{
  int i, j;
  GList *li, *next;
  struct _room *room;
  cave = c;
  for (i = 0; i < MAP_W; i++) {
    for (j = 0; j < MAP_H; j++) {
      cave->map[i][j].pass = FALSE;
      cave->map[i][j].waynavi = FALSE;
      cave->map[i][j].passed = FALSE;
      cave->map[i][j].disturb = FALSE;
      cave->map[i][j].room = NULL;
    };
  };
  cave->rect_header = NULL;
  cave->junction_header = NULL;
  cave->room_header = NULL;
  cave_rect_split(cave_rect_add(CAVE_LX, CAVE_LY, CAVE_HX, CAVE_HY));
  cave_junction_more();
  cave_room_make();
  cave_junction_make();
  cave_map_set();
  cave_map_print_debug();
  for (li = g_list_first(cave->room_header); li != NULL; li = next) {
    next = g_list_next(li);
    room = (struct _room *)li->data;
    if (room->route == TRUE) {
      cave->room_header = g_list_remove(cave->room_header, room);
      my_free(room);
    };
  };
}

static void cave_rect_split(struct _rect *rect_parent)
{

  struct _rect *rect_child;
  int split_coord_x;
  int split_coord_y;
  if (rect_parent->hy - rect_parent->ly <= MINIMUM_RECT_SIZE * 2) {
    rect_parent->cant_split_vertical = TRUE;
  };
  if (rect_parent->hx - rect_parent->lx <= MINIMUM_RECT_SIZE * 2) {
    rect_parent->cant_split_horizonal = TRUE;
  };
  if (g_random_int_range(0, 64) == 0) {
    rect_parent->cant_split_vertical = TRUE;
    rect_parent->cant_split_horizonal = TRUE;
  };
  if ((rect_parent->cant_split_vertical) &&
      (rect_parent->cant_split_horizonal)) {
    return;
  };
  rect_child = cave_rect_add(rect_parent->lx, rect_parent->ly,
			     rect_parent->hx, rect_parent->hy);
  if (rect_parent->cant_split_vertical == FALSE) {
    split_coord_y = g_random_int_range(rect_parent->ly + MINIMUM_RECT_SIZE, rect_parent->hy - MINIMUM_RECT_SIZE);
    rect_parent->hy = split_coord_y;
    rect_child->ly = split_coord_y;
    rect_parent->cant_split_vertical = TRUE;
    rect_child->cant_split_vertical = TRUE;
    cave_junction_add(JUNCTION_VERTICAL, rect_parent, rect_child);
    cave_rect_split(rect_parent);
    cave_rect_split(rect_child);
    return;
  };
  if (rect_parent->cant_split_horizonal == FALSE) {
    split_coord_x = g_random_int_range(rect_parent->lx + MINIMUM_RECT_SIZE, rect_parent->hx - MINIMUM_RECT_SIZE);
    rect_parent->hx = split_coord_x;
    rect_child->lx = split_coord_x;
    rect_parent->cant_split_horizonal = TRUE;
    rect_child->cant_split_horizonal = TRUE;
    cave_junction_add(JUNCTION_HORIZONAL, rect_parent, rect_child);
    cave_rect_split(rect_parent);
    cave_rect_split(rect_child);
    return;
  };
}

static void cave_junction_more()
{
  GList *li;
  struct _rect *rect;
  struct _rect *rectmap[MAP_W][MAP_H];
  int i, j;
  for (li = g_list_first(cave->rect_header); li != NULL; li = g_list_next(li)) {
    rect = (struct _rect *)li->data;
    for (i = rect->lx; i < rect->hx; i++) {
      for (j = rect->ly; j < rect->hy; j++) {
      	rectmap[i][j] = rect;
      };
    };
  };
  for (i = CAVE_LX; i < CAVE_HX - 1; i++) {
    for (j = CAVE_LY; j < CAVE_HY - 1; j++) {
      g_assert(rectmap[i][j] != NULL);
      g_assert(rectmap[i + 1][j] != NULL);
      g_assert(rectmap[i][j + 1] != NULL);
      if (rectmap[i][j] != rectmap[i][j + 1]) {
	if (g_random_int_range(0, 64) == 0) {
	  cave_junction_add(JUNCTION_VERTICAL, rectmap[i][j], rectmap[i][j + 1]);
	};
      };
      if (rectmap[i][j] != rectmap[i + 1][j]) {
	if (g_random_int_range(0, 64) == 0) {
	  cave_junction_add(JUNCTION_HORIZONAL, rectmap[i][j], rectmap[i + 1][j]);
	};
      };
    };
  };
}

static void cave_room_make()
{

  GList *li;
  struct _rect *rect;
  int x, y, w, h;
  gboolean route;
  for (li = g_list_first(cave->rect_header); li != NULL; li = g_list_next(li)) {
    rect = (struct _rect *)li->data;
    w = g_random_int_range(MINIMUM_ROOM_SIZE, rect->hx - rect->lx - (MARGIN_BETWEEN_RECT_ROOM * 2) + 1);
    h = g_random_int_range(MINIMUM_ROOM_SIZE, rect->hy - rect->ly - (MARGIN_BETWEEN_RECT_ROOM * 2) + 1);
    x = g_random_int_range(rect->lx + MARGIN_BETWEEN_RECT_ROOM, rect->hx - MARGIN_BETWEEN_RECT_ROOM - w + 1);
    y = g_random_int_range(rect->ly + MARGIN_BETWEEN_RECT_ROOM, rect->hy - MARGIN_BETWEEN_RECT_ROOM - h + 1);
    route = FALSE;
    if (g_random_int_range(0, 32) == 0) {
      if (g_list_length(cave->rect_header) > 1) {
	route = TRUE;
      };
    };
    rect->room = cave_room_add(x, y, x + w, y + h, route);
  };
}

static void cave_junction_make()
{
  GList *li;
  struct _junction *junction;
  for (li = g_list_first(cave->junction_header); li != NULL; li = g_list_next(li)) {
    junction = (struct _junction *)li->data;
    switch (junction->v_or_h) {
    case JUNCTION_HORIZONAL:
      junction->couner0.x = junction->rect0->hx;
      junction->couner0.y = g_random_int_range(junction->rect0->room->ly + 1, junction->rect0->room->hy);
      junction->couner1.x = junction->rect1->lx;
      junction->couner1.y = g_random_int_range(junction->rect1->room->ly + 1, junction->rect1->room->hy);
      break;
    case JUNCTION_VERTICAL:
      junction->couner0.x = g_random_int_range(junction->rect0->room->lx + 1, junction->rect0->room->hx);
      junction->couner0.y = junction->rect0->hy;
      junction->couner1.x = g_random_int_range(junction->rect1->room->lx + 1, junction->rect1->room->hx);
      junction->couner1.y = junction->rect1->ly;
    };
  };
}

static void cave_map_set()
{
  GList *li;
  struct _room *room;
  int i, j;
  struct _junction *junction;
  for (li = g_list_first(cave->room_header); li != NULL; li = g_list_next(li)) {
    room = (struct _room *)li->data;
    if (room->route == TRUE) continue;
    for (i = room->lx; i <= room->hx; i++) {
      for (j = room->ly; j <= room->hy; j++) {
      	cave->map[i][j].pass = TRUE;
      	cave->map[i][j].room = room;
      };
    };
  };
  for (li = g_list_first(cave->junction_header); li != NULL; li = g_list_next(li)) {
    junction = (struct _junction *)li->data;
    switch (junction->v_or_h) {
    case JUNCTION_HORIZONAL:
      g_assert(junction->rect0->hx == junction->rect1->lx);
      cave->map[junction->couner0.x][junction->couner0.y].disturb = TRUE;
      cave->map[junction->couner1.x][junction->couner1.y].disturb = TRUE;
      cave->map[junction->rect0->room->hx][junction->couner0.y].disturb = TRUE;
      cave->map[junction->rect0->room->hx + 1][junction->couner0.y].disturb = TRUE;
      cave->map[junction->rect1->room->lx][junction->couner1.y].disturb = TRUE;
      cave->map[junction->rect1->room->lx - 1][junction->couner1.y].disturb = TRUE;
      line_map_pass(junction->couner0.x, junction->couner0.y,
 		   junction->couner1.x, junction->couner1.y);
      line_map_pass(junction->rect0->room->hx, junction->couner0.y,
		   junction->couner0.x, junction->couner0.y);
      line_map_pass(junction->rect1->room->lx, junction->couner1.y,
		   junction->couner1.x, junction->couner1.y);
      if (junction->rect0->room->route == TRUE) {
	line_map_pass(junction->rect0->room->hx, junction->couner0.y,
		     (junction->rect0->room->lx + junction->rect0->room->hx) / 2,
		     (junction->rect0->room->ly + junction->rect0->room->hy) / 2);
      };
      if (junction->rect1->room->route == TRUE) {
	line_map_pass(junction->rect1->room->lx, junction->couner1.y,
		     (junction->rect1->room->lx + junction->rect1->room->hx) / 2,
		     (junction->rect1->room->ly + junction->rect1->room->hy) / 2);
      };
      line_waynavi(junction->couner0.x, junction->couner0.y,
			junction->couner1.x, junction->couner1.y);
      line_waynavi(junction->rect0->room->hx, junction->couner0.y,
			junction->couner0.x, junction->couner0.y);
      line_waynavi(junction->rect1->room->lx, junction->couner1.y,
			junction->couner1.x, junction->couner1.y);
      line_waynavi(junction->rect0->room->hx, junction->couner0.y,
			(junction->rect0->room->lx + junction->rect0->room->hx) / 2,
			(junction->rect0->room->ly + junction->rect0->room->hy) / 2);
      line_waynavi(junction->rect1->room->lx, junction->couner1.y,
			(junction->rect1->room->lx + junction->rect1->room->hx) / 2,
			(junction->rect1->room->ly + junction->rect1->room->hy) / 2);
      break;
    case JUNCTION_VERTICAL:
      g_assert(junction->rect0->hy == junction->rect1->ly);
      cave->map[junction->couner0.x][junction->couner0.y].disturb = TRUE;
      cave->map[junction->couner1.x][junction->couner1.y].disturb = TRUE;
      cave->map[junction->couner0.x][junction->rect0->room->hy].disturb = TRUE;
      cave->map[junction->couner0.x][junction->rect0->room->hy + 1].disturb = TRUE;
      cave->map[junction->couner1.x][junction->rect1->room->ly].disturb = TRUE;
      cave->map[junction->couner1.x][junction->rect1->room->ly - 1].disturb = TRUE;
      line_map_pass(junction->couner0.x, junction->couner0.y,
		   junction->couner1.x, junction->couner1.y);
      line_map_pass(junction->couner0.x, junction->rect0->room->hy,
		   junction->couner0.x, junction->couner0.y);
      line_map_pass(junction->couner1.x, junction->rect1->room->ly,
		   junction->couner1.x, junction->couner1.y);
      if (junction->rect0->room->route == TRUE) {
	line_map_pass(junction->couner0.x, junction->rect0->room->hy,
		      (junction->rect0->room->lx + junction->rect0->room->hx) / 2,
		      (junction->rect0->room->ly + junction->rect0->room->hy) / 2);
      };
      if (junction->rect1->room->route == TRUE) {
	line_map_pass(junction->couner1.x, junction->rect1->room->ly,
		     (junction->rect1->room->lx + junction->rect1->room->hx) / 2,
		     (junction->rect1->room->ly + junction->rect1->room->hy) / 2);
      };
      line_waynavi(junction->couner0.x, junction->couner0.y,
			junction->couner1.x, junction->couner1.y);
      line_waynavi(junction->couner0.x, junction->rect0->room->hy,
			junction->couner0.x, junction->couner0.y);
      line_waynavi(junction->couner1.x, junction->rect1->room->ly,
			junction->couner1.x, junction->couner1.y);
      line_waynavi(junction->couner0.x, junction->rect0->room->hy,
			(junction->rect0->room->lx + junction->rect0->room->hx) / 2,
			(junction->rect0->room->ly + junction->rect0->room->hy) / 2);
      line_waynavi(junction->couner1.x, junction->rect1->room->ly,
			(junction->rect1->room->lx + junction->rect1->room->hx) / 2,
			(junction->rect1->room->ly + junction->rect1->room->hy) / 2);
      break;
    };
  };
}

static void cave_map_print_debug()
{
  int i, j;
  int debug_map[MAP_W][MAP_H];
  GList *li;
  struct _rect *rect;
  return;
  for (i = 0; i < MAP_W; i++) {
    for (j = 0; j < MAP_H; j++) {
      debug_map[i][j] = 0;
    };
  };
  for (li = g_list_first(cave->rect_header); li != NULL; li = g_list_next(li)) {
    rect = (struct _rect *)li->data;
    for (i = rect->lx, j = rect->ly; i <= rect->hx; i++) { 
      debug_map[i][j] = 1; 
    }; 
    for (i = rect->lx, j = rect->hy; i <= rect->hx; i++)  { 
      debug_map[i][j] = 1;  
    };   
    for (i = rect->lx, j = rect->ly; j <= rect->hy; j++)  {   
      debug_map[i][j] = 1;  
    };   
    for (i = rect->hx, j = rect->ly; j <= rect->hy; j++)  {   
      debug_map[i][j] = 1;
    };   
  };
  for (i = 0; i < MAP_W; i++) {
    for (j = 0; j < MAP_H; j++) {
      if (cave->map[i][j].pass == TRUE) {
	debug_map[i][j] = 2;
	if (cave->map[i][j].waynavi == TRUE) {
	  debug_map[i][j] = 3;
	};
	if (cave->map[i][j].disturb == TRUE) {
	  debug_map[i][j] = 4;
	};
      };
    };
  };
  for (j = 0; j < MAP_H; j++) { 
    for (i = 0; i < MAP_W; i++) { 
      switch (debug_map[i][j]) { 
      case 0:
	/* MAP_WALL */
	g_print(" "); 
	break; 
      case 1:
	/* RECT'S BORDER */
	g_print("."); 
	break; 
      case 2:
	/* MAP_PASS */ 
	g_print("#"); 
	break; 
      case 3:
	/* MAP_PASS AND WAYNAVI(monster's route) */
	g_print("+"); 
	break; 
      case 4:
	/* MAP_PASS AND DISTURB */
	g_print("*");
	break;
      };
    }; 
    g_print("\n"); 
  }; 
}

static struct _rect *cave_rect_add(int lx, int ly, int hx, int hy)
{
  struct _rect *rect;
  rect = my_new(struct _rect, 1);
  rect->lx = lx;
  rect->ly = ly;
  rect->hx = hx;
  rect->hy = hy;
  rect->cant_split_vertical = FALSE;
  rect->cant_split_horizonal = FALSE;
  cave->rect_header = g_list_append(cave->rect_header, rect);
  return(rect);
}

static struct _junction *cave_junction_add(int v_or_h, struct _rect *rect0, struct _rect *rect1)
{
  struct _junction *junction;
  junction = my_new(struct _junction, 1);
  junction->v_or_h = v_or_h;
  junction->rect0 = rect0;
  junction->rect1 = rect1;
  cave->junction_header = g_list_append(cave->junction_header, junction);
  return(junction);
}

static struct _room *cave_room_add(int lx, int ly, int hx, int hy, gboolean route)
{
  struct _room *room;
  room = my_new(struct _room, 1);
  room->lx = lx;
  room->ly = ly;
  room->hx = hx;
  room->hy = hy;
  room->route = route;
  room->monsterhouse = FALSE;
  cave->room_header = g_list_append(cave->room_header, room);
  return(room);
}
