/*
 * Decompiled with CFR 0.152.
 */
package rescuecore2.log;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import rescuecore2.config.Config;
import rescuecore2.log.AbstractLogReader;
import rescuecore2.log.CommandsRecord;
import rescuecore2.log.ConfigRecord;
import rescuecore2.log.InitialConditionsRecord;
import rescuecore2.log.LogException;
import rescuecore2.log.Logger;
import rescuecore2.log.PerceptionRecord;
import rescuecore2.log.RecordType;
import rescuecore2.log.UpdatesRecord;
import rescuecore2.misc.EncodingTools;
import rescuecore2.registry.Registry;
import rescuecore2.worldmodel.ChangeSet;
import rescuecore2.worldmodel.DefaultWorldModel;
import rescuecore2.worldmodel.Entity;
import rescuecore2.worldmodel.EntityID;
import rescuecore2.worldmodel.WorldModel;

public class FileLogReader
extends AbstractLogReader {
    private static final int KEY_FRAME_BUFFER_MAX_SIZE = 10;
    private RandomAccessFile file;
    private int maxTime;
    private NavigableMap<Integer, WorldModel<? extends Entity>> keyFrames;
    private Map<Integer, Map<EntityID, Long>> perceptionIndices;
    private Map<Integer, Long> updatesIndices;
    private Map<Integer, Long> commandsIndices;
    private Config config;

    public FileLogReader(String name, Registry registry) throws IOException, LogException {
        this(new File(name), registry);
    }

    public FileLogReader(File file, Registry registry) throws IOException, LogException {
        super(registry);
        Logger.info("Reading file log: " + file.getAbsolutePath());
        this.file = new RandomAccessFile(file, "r");
        this.index();
    }

    @Override
    public Config getConfig() {
        return this.config;
    }

    @Override
    public int getMaxTimestep() throws LogException {
        return this.maxTime;
    }

    @Override
    public WorldModel<? extends Entity> getWorldModel(int time) throws LogException {
        Logger.debug("Getting world model at time " + time);
        DefaultWorldModel<Entity> result = new DefaultWorldModel<Entity>(Entity.class);
        Map.Entry<Integer, WorldModel<? extends Entity>> entry = this.keyFrames.floorEntry(time);
        int startTime = entry.getKey();
        Logger.trace("Found key frame " + startTime);
        Logger.trace("Cloning initial conditions");
        for (Entity entity : entry.getValue()) {
            result.addEntity(entity.copy());
        }
        for (int i = startTime + 1; i <= time; ++i) {
            ChangeSet changeSet = this.getUpdates(time).getChangeSet();
            Logger.trace("Merging " + changeSet.getChangedEntities().size() + " updates for timestep " + i);
            result.merge(changeSet);
        }
        Logger.trace("Done");
        this.removeStaleKeyFrames();
        this.keyFrames.put(time, result);
        return result;
    }

    @Override
    public Set<EntityID> getEntitiesWithUpdates(int time) throws LogException {
        Map<EntityID, Long> timestepMap = this.perceptionIndices.get(time);
        if (timestepMap == null) {
            return new HashSet<EntityID>();
        }
        return timestepMap.keySet();
    }

    @Override
    public PerceptionRecord getPerception(int time, EntityID entity) throws LogException {
        Map<EntityID, Long> timestepMap = this.perceptionIndices.get(time);
        if (timestepMap == null) {
            return null;
        }
        Long l = timestepMap.get(entity);
        if (l == null) {
            return null;
        }
        try {
            this.file.seek(l);
            int size = EncodingTools.readInt32(this.file);
            byte[] bytes = EncodingTools.readBytes(size, this.file);
            return new PerceptionRecord(new ByteArrayInputStream(bytes));
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public CommandsRecord getCommands(int time) throws LogException {
        Long index = this.commandsIndices.get(time);
        if (index == null) {
            return null;
        }
        try {
            this.file.seek(index);
            int size = EncodingTools.readInt32(this.file);
            byte[] bytes = EncodingTools.readBytes(size, this.file);
            return new CommandsRecord(new ByteArrayInputStream(bytes));
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    @Override
    public UpdatesRecord getUpdates(int time) throws LogException {
        Long index = this.updatesIndices.get(time);
        if (index == null) {
            return null;
        }
        try {
            this.file.seek(index);
            int size = EncodingTools.readInt32(this.file);
            byte[] bytes = EncodingTools.readBytes(size, this.file);
            return new UpdatesRecord(new ByteArrayInputStream(bytes));
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    private void index() throws LogException {
        try {
            RecordType type;
            Registry.setCurrentRegistry(this.registry);
            this.keyFrames = new TreeMap<Integer, WorldModel<? extends Entity>>();
            this.perceptionIndices = new HashMap<Integer, Map<EntityID, Long>>();
            this.updatesIndices = new HashMap<Integer, Long>();
            this.commandsIndices = new HashMap<Integer, Long>();
            this.file.seek(0L);
            boolean startFound = false;
            do {
                int id = EncodingTools.readInt32(this.file);
                type = RecordType.fromID(id);
                if (!startFound) {
                    if (!RecordType.START_OF_LOG.equals((Object)type)) {
                        throw new LogException("Log does not start with correct magic number");
                    }
                    startFound = true;
                }
                this.indexRecord(type);
            } while (!RecordType.END_OF_LOG.equals((Object)type));
        }
        catch (EOFException e) {
            Logger.debug("EOF found");
        }
        catch (IOException e) {
            throw new LogException(e);
        }
    }

    private void indexRecord(RecordType type) throws IOException, LogException {
        switch (type) {
            case START_OF_LOG: {
                this.indexStart();
                break;
            }
            case INITIAL_CONDITIONS: {
                this.indexInitialConditions();
                break;
            }
            case PERCEPTION: {
                this.indexPerception();
                break;
            }
            case COMMANDS: {
                this.indexCommands();
                break;
            }
            case UPDATES: {
                this.indexUpdates();
                break;
            }
            case CONFIG: {
                this.indexConfig();
                break;
            }
            case END_OF_LOG: {
                this.indexEnd();
                break;
            }
            default: {
                throw new LogException("Unexpected record type: " + (Object)((Object)type));
            }
        }
    }

    private void indexStart() throws IOException {
        int size = EncodingTools.readInt32(this.file);
        EncodingTools.reallySkip(this.file, size);
    }

    private void indexEnd() throws IOException {
        int size = EncodingTools.readInt32(this.file);
        EncodingTools.reallySkip(this.file, size);
    }

    private void indexInitialConditions() throws IOException, LogException {
        int size = EncodingTools.readInt32(this.file);
        if (size < 0) {
            throw new LogException("Invalid initial conditions size: " + size);
        }
        byte[] bytes = EncodingTools.readBytes(size, this.file);
        InitialConditionsRecord record = new InitialConditionsRecord(new ByteArrayInputStream(bytes));
        this.keyFrames.put(0, record.getWorldModel());
    }

    private void indexPerception() throws IOException, LogException {
        long position = this.file.getFilePointer();
        int size = EncodingTools.readInt32(this.file);
        byte[] bytes = EncodingTools.readBytes(size, this.file);
        PerceptionRecord record = new PerceptionRecord(new ByteArrayInputStream(bytes));
        int time = record.getTime();
        EntityID agentID = record.getEntityID();
        Map<EntityID, Long> timestepMap = this.perceptionIndices.get(time);
        if (timestepMap == null) {
            timestepMap = new HashMap<EntityID, Long>();
            this.perceptionIndices.put(time, timestepMap);
        }
        timestepMap.put(agentID, position);
    }

    private void indexCommands() throws IOException, LogException {
        long position = this.file.getFilePointer();
        int size = EncodingTools.readInt32(this.file);
        byte[] bytes = EncodingTools.readBytes(size, this.file);
        CommandsRecord record = new CommandsRecord(new ByteArrayInputStream(bytes));
        int time = record.getTime();
        this.commandsIndices.put(time, position);
        this.maxTime = Math.max(time, this.maxTime);
    }

    private void indexUpdates() throws IOException, LogException {
        long position = this.file.getFilePointer();
        int size = EncodingTools.readInt32(this.file);
        byte[] bytes = EncodingTools.readBytes(size, this.file);
        UpdatesRecord record = new UpdatesRecord(new ByteArrayInputStream(bytes));
        int time = record.getTime();
        this.updatesIndices.put(time, position);
        this.maxTime = Math.max(time, this.maxTime);
    }

    private void indexConfig() throws IOException, LogException {
        int size = EncodingTools.readInt32(this.file);
        byte[] bytes = EncodingTools.readBytes(size, this.file);
        ConfigRecord record = new ConfigRecord(new ByteArrayInputStream(bytes));
        this.config = record.getConfig();
    }

    private void removeStaleKeyFrames() {
        Logger.trace("Removing stale key frames");
        int size = this.keyFrames.size();
        if (size < 10) {
            Logger.trace("Key frame buffer is not full: " + size + (size == 1 ? " entry" : " entries"));
            return;
        }
        int window = this.maxTime / 10;
        for (int i = 0; i < this.maxTime; i += window) {
            NavigableMap<Integer, WorldModel<? extends Entity>> next = this.keyFrames.subMap(i, false, i + window, true);
            Logger.trace("Window " + i + " -> " + (i + window) + " has " + next.size() + " entries");
            if (next.size() <= 1) continue;
            Map.Entry<Integer, WorldModel<? extends Entity>> last = next.lastEntry();
            next.clear();
            next.put(last.getKey(), last.getValue());
            Logger.trace("Retained entry " + last);
        }
        Logger.trace("New key frame set: " + this.keyFrames);
    }
}

