/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.model;

import java.util.List;
import java.util.UUID;
import java.util.Vector;

import com.nexaweb.xml.Document;
import com.nexaweb.xml.Element;
import com.nexaweb.xml.xpath.XPathFactory;

/**
 * <p>[概 要] </p>
 * 処理の結果、新規xalを画面挿入する可能性の有るモデルが継承するクラスです。
 * 
 * <p>[詳 細] </p>
 * xalを画面にレンダリングした後、、ウィンドウ識別を容易にする為の機能を提供します。<br>
 * 継承クラスは{@link #addIdentifierToWindows(Document)}を呼び出すことによって、<br>
 * 取得したxal内ウィンドウレベルエレメントに対して「parentId」、「communicateId」属性を付与します。<br>
 * 
 * <p>[備 考] </p>
 * parentId、communicateIdについては{@link #addIdentifierToWindows(Document)}を参照して下さい。<p>
 * 
 * 
 * <b>使用例)</b><br>
 * ・同じxalを複数読み込むが、読み込んだ後にユニークに特定する
 * 
 * <pre class="samplecode">
 * public class MapXalFetchAction extends BaseAction {
 *     &#064;Override
 *    protected void reserveModels(List<Class<? extends Model>> models) {
 *        models.add(HTTPRequestCore.class);
 *    }
 *
 *    &#064;Override
 *   public boolean nextModel(int index, ModelProcessEvent prev, Model next) throws Exception {
 *        switch(index){
 *            case 0:
 *                // 都道府県選択コンボボックスから選択中の都道府県を取得
 *                String prefecture = getElementValueByName("comboPrefecture").getElementValue(0).getValue();
 *
 *                // ＊サーバロジックは任意
 *                ((HTTPRequestCore)next).setRequestUrl("webcontroller");
 *                ((HTTPRequestCore)next).addUrlParameters("forward.page", "/pages/map.xal");
 *                
 *                // 取得予定のxal内ウィンドウcommunicateIdに都道府県名を設定
 *                ((HTTPRequestCore)next).setCommunicateId(prefecture);
 *                break;
 *        }
 *        return true;
 *    }
 * }   
 * </pre>
 *
 * 結果、ui Documentには以下のように挿入され、communicateIdで同一xalがユニーク識別可能になります。
 * <pre class="samplecode">
 *     &lt;ui&gt;
 *         &lt;window name="map" communicateId="hokkaido"&gt;
 *                 :
 *         &lt;/window&gt;
 * 
 *         &lt;window name="map" communicateId="tokyo"&gt;
 *                 :
 *         &lt;/window&gt;
 *     &lt;/ui&gt;
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public abstract class XalReturnPossibilityModel extends BaseModel {

	/** このモデルを起動したコンポーネントが属するウィンドウレベルエレメントです。 */
	private Element eventSourceWindow;
	
	/** 通信識別IDです。任意に設定されない場合は「UUID.randomUUID()」がデフォルトになります。*/
	private String communicateId = UUID.randomUUID().toString();
	
	/**
	 * <p>[概 要] </p>
	 * このモデルを起動したコンポーネントが属するウィンドウレベルエレメントを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * eventSourceWindowフィールド値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return このモデルを起動したコンポーネントが属するウィンドウレベルエレメント
	 */
	public Element getEventSourceWindow() {
		return eventSourceWindow;
	}

	/**
	 * <p>[概 要] </p>
	 * このモデルを起動したコンポーネントが属するウィンドウレベルエレメントを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * eventSourceWindowフィールド値を、引数eventSourceWindowで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventSourceWindow このモデルを起動したコンポーネントが属するウィンドウレベルエレメント
	 */
	public void setEventSourceWindow(Element eventSourceWindow) {
		this.eventSourceWindow = eventSourceWindow;
	}

	/**
	 * <p>[概 要] </p>
	 * 通信識別IDを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * communicateIdフィールド値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * {@link #setCommunicateId(String)}によって任意設定されていない場合、<br>
	 * UUID.randomUUID().toString()<br>
	 * が返却されます。
	 * 
	 * @return 通信識別ID
	 */
	public String getCommunicateId() {
		return this.communicateId;
	}

	/**
	 * <p>[概 要] </p>
	 * 通信識別IDを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * communicateIdフィールド値を、引数communicateIdで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param communicateId 通信識別ID
	 */
	public void setCommunicateId(String communicateId) {
		this.communicateId = communicateId;
	}

	/**
	 * <p>[概 要] </p>
	 * 主処理です。
	 * 
	 * <p>[詳 細] </p>
	 * アクションから譲渡された、イベント発生元ウィンドウレベルエレメントを、
	 * フィールドに保存します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 保存されたイベント発生元ウィンドウレベルエレメントのid属性値が、
	 * 新たに取得したウィンドウレベルエレメントのparentId属性値として設定されます。
	 * 
	 */
	@Override
	protected void mainproc() throws Exception {
		setEventSourceWindow(getParameterMapping().getEventSourceWindow());
	}
	
	/**
	 * <p>[概 要] </p>
	 * 通信の結果取得したウィンドウレベルエレメントに識別子parentId、communicateIdを付与します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数Documentに含まれる、window、dialogエレメントを取得し、
	 * privateオーバーロードメソッドに処理委譲します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * <b>「parentId」について</b><p>
	 * 
	 * 通信の結果新しく取得したウィンドウレベルエレメントの親を識別する為のIDです。<br>
	 * 「親」はDOM構造的な親では無く、画面の発生元を表します。<br>
	 * Aウィンドウ内アクション発生　→　XalReturnPossibilityModel通信発生、の結果として
	 * Bウィンドウが画面表示された場合、<br>
	 * Aウィンドウ(id="123")はBウィンドウ(id="456", parentId="123")の親となります。<p>
	 * 
	 * <b>「communicateId」について</b><p>
	 * 
	 * 1通信を識別する為のIDです。<br>
	 * 同じ画面（同じxal）が複数クライアント上に存在するような場合にA.xalとA´.xalを
	 * 識別する為に使用します。<br>
	 * 
	 * <pre class="samplecode">
	 *     &lt;ui&gt;
	 *         // A通信の結果取得したA.xal
	 *         &lt;window name="abc" id="nx-12345" communicateId="com_A"&gt;
	 *             &lt;button name="btn"&gt;
	 *         &lt;/window&gt;
	 * 
	 *         // B通信の結果取得したA.xal
	 *         &lt;window name="abc" id="nx-67890" communicateId="com_B"&gt;
	 *             &lt;button name="btn"&gt;
	 *         &lt;/window&gt;
	 *     &lt;/ui&gt;
	 * </pre>
	 * 上記の場合、abcウィンドウはcommunicateIdを判断することによってユニークに
	 * 識別出来ます。<br>
	 * 又、idとは違う為、Document内でユニークである必要は有りません。
	 * 
	 * @param document 通信の結果取得したXML Document
	 * @return documentに存在するウィンドウレベルエレメント群
	 */
	@SuppressWarnings("unchecked")
	protected Vector<Element> addIdentifierToWindows(Document document) {
		Vector<Element> windowsInResponseXal = new Vector<Element>();
		
		List<String> windowDefs = 
			getController().getClientConfig().getWindowLevelElementDefinition();
		for(String windowDef : windowDefs) {
			Vector<Element> allWindows = 
				XPathFactory.createXPath("//" + windowDef).evaluate(document);
			windowsInResponseXal.addAll(allWindows);
		}

		addIdentifierToWindows(windowsInResponseXal);
		
		return windowsInResponseXal;  
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数windowsにparentId、communicateId属性値を付与します。
	 * 
	 * <p>[詳 細] </p>
	 * このモデルを起動したコンポーネントが属するウィンドウレベルエレメントのidを、
	 * 通信の結果取得したウィンドウレベルエレメントに対し、parentId属性値として設定します。<br>
	 * 又、{@link #getCommunicateId()}の値をcommunicateId属性値として設定します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param windows 識別idを付与するウィンドウレベルエレメント群
	 */
	private void addIdentifierToWindows(Vector<Element> windows) {
		for(Element window : windows) {
			window.setAttribute("parentId", getEventSourceWindow().getAttribute("id"));
			window.setAttribute("communicateId", getCommunicateId());
		}
	}
}
