/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.rpc {
	
	import flash.events.IEventDispatcher;
	import flash.utils.Dictionary;
	
	import mx.rpc.soap.Operation;
	import mx.rpc.soap.WebService;

	/**
	 * <p>[概 要]</p>
	 * リモートサーバにSOAPリクエストを行う為の機能モデルクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * WSDLが公開されているSOAPベースのWebサービスと通信を行います。
	 * <p/>
	 * 
	 * 提供されているサービスメソッドの粒度が細かい場合、アクションにこのクラスを複数登録することで
	 * １アクション内でマッシュアップが可能です。<br>
	 * <ul>
	 *   <li>
	 *     従業員名をもとに従業員IDを返却するサービスメソッド<br>
	 *     public String getEmployeeId(String employeeName);
	 *     <p/>
	 * 
	 *   </li>
	 *   <li>
	 *     従業員IDをもとにEメールアドレスを返却するサービスメソッド<br>
	 *     public String getEmployeeEmailAddress(String employeeId);
	 *     <p/>
	 * 
	 *   </li>
	 *   <li>
	 *     Eメールを送信するサービスメソッド<br>
	 *     public Boolean sendEmail(String to, String content);
	 *     <p/>
	 * 
	 *   </li>
	 * </ul>
	 * <p/>
	 * 
	 * 上記のように一つ一つの機能が少ないサービスメソッドが用意されている場合、
	 * 従業員名を元にEmailを送信するアクションは以下のようにSOAPリクエストを定義します。
	 * 
	 * <listing version="3.0">
	    public class WebServiceMashUpAction extends BaseAction {
	    
            override protected function reserveModels(models:Array):void {
                // getEmployeeId用
                models.push(Class(WebServiceCore));
                
                // getEmployeeEmailAddress用
                models.push(Class(WebServiceCore));
                
                // sendEmail用
                models.push(Class(WebServiceCore));
            }
            
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
    
                WebServiceCore(next).wsdl = "employeeService?wsdl";
    
                switch(index) {
                    case 0:
                        var employeeName:String = TextInput(getComponentByName("txtEmployeeName")).text;
    
                        WebServiceCore(next).methodName = "getEmployeeId";
                        WebServiceCore(next).addWsParameter(employeeName);
                        break;
                    case 1:
                        var employeeId:String = ResultEvent(prev.cause).result as String;
                        
                        WebServiceCore(next).methodName = "getEmployeeEmailAddress";
                        WebServiceCore(next).addWsParameter(employeeId);
                        break;
                    case 2:
                        var employeeEmailAddress:String = ResultEvent(prev.cause).result as String;
                        WebServiceCore(next).methodName = "sendEmail";
                        WebServiceCore(next).addWsParameter(employeeEmailAddress);
                        WebServiceCore(next).addWsParameter(TextInput(getComponentByName("taEmailContent")).text);
                        break;
                }
                
                return true;
            }
            
            override public function failureForward(index:int, model:BaseModel, faultEvent:Event):Event {
                switch(index) {
                    case 0:
                        trace("従業員名が登録されていません");
                        break;
                    case 1:
                        trace("Emailアドレスが登録されていません");
                        break;
                    case 2:
                        trace("Email送信中にエラーが発生しました");
                        break;
                }
                
                return faultEvent;
            }
            
            override public function complete():void {
                trace("Emailを送信しました");
            }
        }
	 * 
	 * </listing>
	 * 
	 * <p>[備 考]</p>
	 * <ul>
	 *   <li>
	 *     Flexで使用する場合、他ドメインで公開されているWebサービスにアクセスするには、
	 *     Webサービス配置サーバにcrossdomain.xmlを配置するか、
	 *     一度自ドメインサーバを介してアクセスする必要が有ります。
	 *   </li>
	 *   <li>
	 *     JAX-WS2.0ベースのSOAP WebサービスクラスをデプロイするにはTomcatでは無く、フルスタックな
	 *     APサーバが必要です。
	 *   </li>
	 * </ul>
	 * <p/>
	 * 
	 * @example SOAP WebServiceメソッドにリクエストする
	 * <p/>
	 * 
	 * 以下はシンプルなSOAP WebServiceを起動してクライアントからアクセスする例です。
	 * <p/>
	 * 
	 * ・<b>サーバ側</b>
	 * <p/>
	 * 
	 * ・GreenSheet型(POJO)リストを返却するWebService（Java）
	 * <p/>
	 * 
	 * getGreenSheetInfoはcsvからグリーンシートデータリストを取得
	 * してクライアントに返却する単純なサービスメソッドです。<br>
	 * 
	 * <listing version="3.0">
        package test.server;
        
        import java.io.BufferedReader;
        import java.io.FileInputStream;
        import java.io.IOException;
        import java.io.InputStreamReader;
        import java.util.ArrayList;
        import java.util.List;
        import java.util.StringTokenizer;
        
        import javax.annotation.Resource;
        import javax.jws.WebParam;
        import javax.jws.WebResult;
        import javax.jws.WebService;
        import javax.servlet.ServletContext;
        import javax.xml.ws.WebServiceContext;
        import javax.xml.ws.handler.MessageContext;
        
        
        &#64;WebService
        public class DummyService {
            
            &#64;Resource
            private WebServiceContext wsContext;
        
            &#64;WebResult(name="getGreenSheetReturn")
            public List<GreenSheet> getGreenSheetInfo(&#64;WebParam String brandName) throws IOException{
                List<GreenSheet> greenSheetInfo = new ArrayList<GreenSheet>();
                
                MessageContext msgContext = wsContext.getMessageContext();
                ServletContext servletContext = (ServletContext) msgContext
                        .get(MessageContext.SERVLET_CONTEXT);
                String filePath = servletContext.getRealPath("/greensheet.csv");
        
                FileInputStream fis = new FileInputStream(filePath);
                InputStreamReader in = new InputStreamReader(fis,"SJIS");
                BufferedReader br = new BufferedReader(in); 
                
                String line = null;
                while((line = br.readLine()) != null){
                    StringTokenizer tokenizer = new StringTokenizer(line, ",");
    
                    GreenSheet greenSheet = new GreenSheet();
        
                    greenSheet.setRegistDate(tokenizer.nextToken());
                    greenSheet.setMemberName(tokenizer.nextToken());
                    greenSheet.setBrandName(tokenizer.nextToken());
                    greenSheet.setMarket(tokenizer.nextToken());
                    greenSheet.setHandlingBrand(tokenizer.nextToken());
                    greenSheet.setPublicDivision(tokenizer.nextToken());
                    greenSheet.setTradeDate(tokenizer.nextToken());
                    greenSheet.setBuyAndSellDivision(tokenizer.nextToken());
                    greenSheet.setAmount(tokenizer.nextToken());
                    greenSheet.setPrice(tokenizer.nextToken());
                    greenSheet.setOpponent(tokenizer.nextToken());
                    greenSheet.setDivision(tokenizer.nextToken());
                    
                    if("".equals(brandName) || brandName.equals(greenSheet.getBrandName())) {
                        greenSheetInfo.add(greenSheet);
                    }
                }
                
                br.close();
                fis.close();
                
                return greenSheetInfo;
            }
        }
	 * </listing>
	 * 
	 * ・web.xmlにDummyServiceクラスをWebサービス登録
	 * <p/>
	 * 
	 * http://ドメイン/コンテキストルート/dummyservice?wsdlでWSDL公開します。
	 * 
	 * <listing version="3.0">
        &lt;servlet>
            &lt;servlet-name>DummyWS&lt;/servlet-name>
            &lt;servlet-class>test.server.DummyService&lt;/servlet-class>
        &lt;/servlet>
        &lt;servlet-mapping>
            &lt;servlet-name>DummyWS&lt;/servlet-name>
            &lt;url-pattern>/dummyservice&lt;/url-pattern>
        &lt;/servlet-mapping>
	 * </listing>
	 * 
	 * ・<b>クライアント側</b>
	 * <p/>
	 * 
	 * ・dummyserviceにアクセスするアクションクラス
	 * <p/>
	 * 
	 * 公開したWSDLを参照してDummyServiceクラスのgetGreenSheetInfoメソッドを呼び出します。<br>
	 * getGreenSheetInfoメソッドの引数は、画面上のブランド名入力TextInput（txtBrandName）から取得しています。<br>
	 * successForwardで取得するresultEventにはSOAPレスポンスがObjectフォーマットで格納されています。<br>
	 * このレスポンスフォーマットをXMLやE4X形式に変更したい場合は、resultFormatプロパティに
	 * RPCCore.XMLやRPCCore.E4Xを指定してWebServiceCoreを実行します。
	 * 
	 * <listing version="3.0">
        package test.client.action {
            import flash.events.Event;
            
            import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
            import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
            import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
            import jp.co.fujitsu.reffi.client.flex.model.rpc.WebServiceCore;
            
            import mx.collections.ArrayCollection;
            import mx.controls.TextInput;
            import mx.rpc.events.ResultEvent;
        
            public class SOAPRequestAction extends BaseAction {
                
                override protected function reserveModels(models:Array):void {
                    models.push(Class(WebServiceCore));
                }
                
                override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                    if(index == 0) {
                        var brandName:String = TextInput(getComponentByName("txtBrandName")).text;
                        
                        WebServiceCore(next).wsdl = "dummyservice?wsdl";
                        WebServiceCore(next).methodName = "getGreenSheetInfo";
                        WebServiceCore(next).addWsParameter(brandName);
                    }
                    
                    return true;
                }
                
                override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                    var results:ArrayCollection = ResultEvent(resultEvent).result as ArrayCollection;
                    for each(var greenSheet:Object in results) {
                        trace(
                                greenSheet["registDate"] + ", " +
                                greenSheet["memberName"] + ", " +
                                greenSheet["brandName"] + ", " +
                                greenSheet["market"] + ", " +
                                greenSheet["handlingBrand"] + ", " +
                                greenSheet["publicDivision"] + ", " +
                                greenSheet["tradeDate"] + "," +
                                greenSheet["buyAndSellDivision"] + ", " +
                                greenSheet["amount"] + ", " +
                                greenSheet["price"] + ", " +
                                greenSheet["opponent"] + ", " +
                                greenSheet["division"]
                        );
                    }
                }
                
                override public function complete():void {
                    trace("SOAP Request Complete");
                }
            }
        }
	 * </listing>
	 * 
	 * ・実行結果
	 * 
	 * <listing version="3.0">
        2009/3/1, ○○証券代行株式会社, ○○ソフトウェアエンジニアリング, エマージング, 株券, 毎日, 2009/3/1,売, 50, 25500, △△証券株式会社, 委託売買
        2009/3/1, ○○証券代行株式会社, ○○ソフトウェアエンジニアリング, リージョナル, 株券, 毎日, 2009/3/1,買, 50, 25500, ××証券株式会社, 委託売買
            :
            :
        2009/3/1, ○○証券代行株式会社, ○○ソフトウェアエンジニアリング, エマージング, 株券, 毎日, 2009/3/1,売, 13, 25000, △△証券株式会社, 委託売買
        2009/3/1, ○○証券代行株式会社, ○○ソフトウェアエンジニアリング, フェニックス, 株券, 毎日, 2009/3/1,買, 13, 25000, △△証券株式会社, 委託売買
        SOAP Request Complete
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */ 
	public class WebServiceCore extends RPCCore {
		
		// WSDL
		private var _wsdl:String;
		
		// メソッド名
		private var _methodName:String;
		
		// 引数 
		private var _wsParameter:Array = new Array();
		
		// ヘッダオブジェクト配列
		private var _headers:Array = new Array();
		
		// 受信データフォーマット
		private var _resultFormat:String = RPCCore.OBJECT;

		
		/**
		 * <p>[概 要]</p>
		 * 読み込むWSDLのロケーションです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get wsdl():String{
			return this._wsdl;
		}
		public function set wsdl(wsdl:String):void{
			this._wsdl = wsdl;
		}
		
		
		/**
		 * <p>[概 要]</p>
		 * 実行するメソッド名です.
		 * 
		 * <p>[詳 細]</p>
		 * WSDLに記述されているメソッド名を指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get methodName():String{
			return this._methodName;	
		}
		public function set methodName(methodName:String):void{
			this._methodName = methodName;
		}
		
		/**
		 * <p>[概 要]</p>
		 * メソッド引数です.
		 * 
		 * <p>[詳 細]</p>
		 * methodNameで指定したメソッドが持つパラメータです。
		 * 第一引数、第二引数・・・と順番に値が入ります。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get wsParameter():Array{
			return this._wsParameter;
		}
		public function set wsParameter(wsParameter:Array):void{
			this._wsParameter = wsParameter;
		}
		
		/**
		 * <p>[概 要]</p>
		 * メソッド引数を追加します.
		 * 
		 * <p>[詳 細]</p>
		 * methodNameで指定したメソッドのパラメータを設定します。<br>
		 * 呼び出すサービスメソッドが以下のようなシグネチャを持っている場合、
		 * <p/>
		 * 
		 * public int registProduct(int productId, String productName);
		 * <p/>
		 * 
		 * addWsParameter(5);<br>
		 * addWsParameter("ノートPC");<br>
		 * のように追加します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function addWsParameter(value:Object):Object {
			this._wsParameter.push(value);
			
			return this._wsParameter;
		}
		
		/**
		 * <p>[概 要]</p>
		 * SOAPヘッダ情報です.
		 * 
		 * <p>[詳 細]</p>
		 * 既存のSOAPヘッダ情報は全て上書きされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get headers():Array {
			return this._headers;
		}
		public function set headers(headers:Array):void {
			this._headers = headers;
		}
		
		/**
		 * <p>[概 要]</p>
		 * SOAPヘッダ情報を追加します.
		 * 
		 * <p>[詳 細]</p>
		 * 既に登録されているSOAPヘッダ情報に追加登録します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param header SOAPヘッダ情報
		 */ 
		public function addHeader(header:Object):void {
			this._headers.push(header);
		}
		
		/**
		 * <p>[概 要]</p>
		 * 受信データフォーマットです.
		 * 
		 * <p>[詳 細]</p>
		 * 「object」「e4x」「xml」が指定可能です。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default object
		 * @see jp.co.fujitsu.reffi.client.flex.model.RPCCore
		 */ 
		public function get resultFormat():String{
			return this._resultFormat;
		}
		public function set resultFormat(resultFormat:String):void{
			this._resultFormat = resultFormat;
		}
		
		/**
		 * <p>[概 要]</p>
		 * メイン処理メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * WebServiceオブジェクトに対してプロパティ情報を設定して通信を開始します。
		 * 通信確立の為以下の処理が行われます。
		 * <ul>
		 *   <li>WSDLのロード</li>
		 *   <li>SOAPヘッダの追加</li>
		 *   <li>実行メソッドの特定</li>
		 *   <li>レスポンスデータフォーマットの指定</li>
		 * </ul> 
		 * 
		 * <p>[備 考]</p>
		 */ 
		override protected function mainProc() : void {
			super.mainProc();
			
			// 具象通信オブジェクト
			var service : WebService = this.service as WebService;
			// ロードするWSDLのロケーション
			var wsdl:String = this.wsdl;
			// 実行メソッド名
			var methodName:String = this.methodName;
			// 実行メソッド引数
			var wsParameter:Array = this.wsParameter;
			// SOAPヘッダ
			var headers:Array = this.headers;
			// レスポンスデータフォーマット
			var resultFormat:String = this.resultFormat;

			// 指定されたSOAPヘッダを追加 			
			for (var i:int = 0; i < headers.length; i++) {
				service.addHeader(headers[i]);
			}
			
			// WSDLをロード			
			service.loadWSDL(wsdl);
			// 実行メソッド名を元にOperationオブジェクト生成
			var operation : Operation = service.getOperation(methodName) as Operation;
			// 引数指定
			operation.arguments = wsParameter;
			// レスポンスデータフォーマット指定
			operation.resultFormat = resultFormat;
			// SOAPリクエスト送信	
			operation.send();
		}
		
		/**
		 * <p>[概 要]</p>
		 * 具象通信オブジェクトを作成します.
		 * 
		 * <p>[詳 細]</p>
		 * WebServiceオブジェクトを生成して返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return WebServiceオブジェクト
		 */ 
		override protected final function createConcreteService() : IEventDispatcher {
			return new WebService();
		}
	}
}
