/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.controller;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import jp.co.fujitsu.reffi.client.nexaweb.action.Action;
import jp.co.fujitsu.reffi.client.nexaweb.action.BaseAction;

import com.nexaweb.xml.Document;
import com.nexaweb.xml.Element;
import com.nexaweb.xml.Parser;
import com.nexaweb.xml.ParserFactory;
import com.nexaweb.xml.xpath.XPathFactory;

/**
 * <p>[概 要] </p>
 * イベント紐付け情報保持クラスです。
 * 
 * <p>[詳 細] </p>
 * {@link BaseController#bind(EventBinder)}によるコンポーネント、イベントタイプ、アクションクラス
 * の紐付け情報を、bindInfoフィールドMapに保持します。<br>
 * bindInfoオブジェクトは以下のような構造を持ちます。
 * <pre>
 * [
 * コンポーネント名1 = 
 *     [イベントタイプ1 = アクションクラス, 
 *      イベントタイプ2 = アクションクラス],
 * コンポーネント名2 =
 *     [イベントタイプ1 = アクションクラス] 
 * ]
 * </pre>
 * 
 * <pre class="samplecode">
 * eventBinder.addEventBinding("menu.openChat", "onActiveGained", FocusChatAction.class);
 * eventBinder.addEventBinding("menu.openChat", "onCommand", OpenChatAction.class);
 * eventBinder.addEventBinding("menu.openWindowGroupParent", "onCommand", OpenWindowGroupParentAction.class);
 * </pre>
 * 
 * 上記の場合、以下のようになります。
 * <pre>
 * [
 * "menu.openChat" = 
 *     ["onActiveGained" = FocusChatAction.class, 
 *      "onCommand" = OpenChatAction.class]
 * "menu.openWindowGroupParent" =
 *     ["onCommand" = OpenWindowGroupParentAction.class] 
 * ]
 * 
 * </pre>
 * 
 * <p>[備 考] </p>
 * アプリケーション動作中に紐付け情報を追加する場合、以下のようにEventBinderオブジェクト
 * を取得してaddEventBinding、又はaddEventBindingImmediatelyをコールします。
 * 
 * <pre class="samplecode">
 * 	getController().getEventBinder().addEventBinding("newdisplay.btnRegist", "onCommand", RegistAction.class);
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class EventBinder {

	/** マップ初期容量 */
	private static final int INITIAL_CAPACITY = 2500;
	
	/** Element名、eventタイプ、Actionクラスの紐付け情報を保持するオブジェクトです */
	private Map<String, Map<String, Class<? extends Action>>> bindInfo;

	/** コントローラオブジェクトです。 */
	private BaseController controller;

	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報保持オブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return イベント紐付け情報
	 */
	public Map<String, Map<String, Class<? extends Action>>> getBindInfo() {
		return bindInfo;
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報保持オブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドを、引数bindInfoで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param bindInfo イベント紐付け情報
	 */
	public void setBindInfo(
			Map<String, Map<String, Class<? extends Action>>> bindInfo) {
		this.bindInfo = bindInfo;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コントローラオブジェクト
	 */
	public BaseController getController() {
		return controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを引数controllerで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラオブジェクト
	 */
	public void setController(BaseController controller) {
		this.controller = controller;
	}

	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドインスタンスを生成します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public EventBinder(BaseController controller) {
		setBindInfo(new HashMap<String, Map<String, Class<? extends Action>>>(INITIAL_CAPACITY));
		setController(controller);
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数componentNameが既に登録されている場合、componentName用のMapを取り出し、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * componentNameが未登録の場合、componentName用のMapを新規に生成して、
	 * イベントタイプをキーにしてアクションを登録します。<br>
	 * 登録された情報はイベント紐付け情報保持オブジェクト（bindInfoフィールド）に保持され、
	 * コントローラのエレメント挿抜リスナによって参照されます。
	 * 
	 * <p>[備 考] </p>
	 * 登録された紐付け情報が反映されるのは、componentNameをname属性値として持つ
	 * エレメントが、ui Documentに追加されたタイミングです。
	 * 
	 * @param componentName コンポーネント（エレメント）のname属性値
	 * @param eventType Nexawebイベントタイプ
	 * @param actionClass 起動するアクションのクラス型
	 */
	public void addEventBinding(String componentName, String eventType,
			Class<? extends Action> actionClass) {
		Map<String, Class<? extends Action>> typeMap = null;

		if (getBindInfo().containsKey(componentName)) {
			typeMap = (Map<String, Class<? extends Action>>) getBindInfo().get(
					componentName);
		} else {
			typeMap = new HashMap<String, Class<? extends Action>>();
		}

		typeMap.put(eventType, actionClass);

		getBindInfo().put(componentName, typeMap);
	}

	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報追加メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * {@link #addEventBinding(String, String, Class)}オーバーロードメソッドを
	 * 呼び出して、イベント紐付け情報保持オブジェクト（bindInfo）に追加します。<br>
	 * 追加後、ui Documentを取得して、追加された紐付け情報を即座に反映します。
	 * 
	 * <p>[備 考] </p>
	 * {@link #addEventBinding(String, String, Class)}と違い、登録された情報がui Documentに
	 * 反映されるのはこのメソッド呼び出し直後です。
	 * 
	 * @param componentName コンポーネント（エレメント）のname属性値
	 * @param eventType Nexawebイベントタイプ
	 * @param actionClass 起動するアクションのクラス型
	 * @see #addEventBinding(String, String, Class)
	 */
	@SuppressWarnings("unchecked")
	public void addEventBindingImmediately(String componentName,
			String eventType, Class<? extends Action> actionClass) {
		// イベント紐付け情報を追加
		addEventBinding(componentName, eventType, actionClass);

		// ui Documentに反映
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();
		Vector<Element> elements = XPathFactory.createXPath(
				"descendant::*[@name='" + componentName + "']").evaluate(
				document);

		for (Element element : elements) {
			element.setAttribute(eventType, BaseController.HANDLER_ENTRY_POINT);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 引数指定された名前を持つエレメントに登録されているイベントタイプ群を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドから、引数componentNameで紐付け情報を取り出し、
	 * 登録されている全イベントタイプをString配列に変換して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param componentName コンポーネント（エレメント）のname属性値
	 * @return componentNameに登録されている全イベントタイプが入った配列
	 */
	public String[] getEventTypes(String componentName) {
		Map<String, Class<? extends Action>> typeMap = getBindInfo().get(
				componentName);
		if (typeMap == null) {
			return null;
		}

		Set<String> eventTypeSet = typeMap.keySet();
		String[] eventTypes = eventTypeSet.toArray(new String[1]);

		return eventTypes;
	}

	/**
	 * <p>[概 要] </p>
	 * componentNameをname属性値として持つエレメントの、eventTypeイベント発動時の
	 * アクションクラス型を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * bindInfoフィールドから、引数componentNameで紐付け情報を取り出し、
	 * 取り出した[イベントタイプ = アクションクラス]のマップから、eventTypeイベント
	 * のアクションクラスを取得、返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param componentName コンポーネント（エレメント）のname属性値
	 * @param eventType イベントタイプ
	 * @return componentName、eventTypeに対応するアクションクラス型
	 */
	public Class<? extends Action> getActionClass(String componentName,
			String eventType) {
		Map<String, Class<? extends Action>> typeMap = getBindInfo().get(
				componentName);
		if (typeMap == null) {
			return null;
		}

		Class<? extends Action> actionClass = (Class<? extends Action>) typeMap
				.get(eventType);

		return actionClass;
	}

	/**
	 * 
	 * <p>[概 要]</p>
	 * イベント紐付け情報XMLを読込み追加するメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * イベント紐付け情報XMLを読込み追加するメソッドです。
	 * 
	 * 読込みXMLの例：
	 * <pre>
	 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
	 * &lt;reffi-client-config xmlns="urn:jp.co.fujitsu.reffi.client.config"&gt;
	 *     &lt;import resource="reffi-client-config-window.xml" /&gt;
	 *     &lt;!--
	 *         event mapping config.
	 *         name: Name on element
	 *         event: Set provided by the Nexaweb event name (ex. onCommand, onCreate, onContextMenu ...)
	 *         type: Call action class
	 *     --&gt;
	 *     &lt;event-mappings&gt;
	 *         &lt;!-- menuWindowAction --&gt;
	 *         &lt;event
	 *             name="menu.openChat"
	 *             event="onCommand"
	 *             type="demo.client.menu.action.OpenChatAction" /&gt;
	 *     &lt;/event-mappings&gt;
	 * &lt;/reffi-client-config&gt;
	 * </pre>
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param fileName 設定ファイル名
	 */
	@SuppressWarnings("unchecked")
	public void loadConfig(String fileName) {
		try {
			InputStream inputStream = getClass().getClassLoader().getResourceAsStream(fileName);
			Parser parser = ParserFactory.getParser();
			Document document = parser.parseXml(new InputStreamReader(inputStream, "UTF-8"));
			Vector<Element> elements = document.getRootElement().getChildren();
			for (Element element : elements) {
				if ("import".equals(element.getLocalName())) {
					loadConfig(element.getAttribute("resource"));
				} else if ("event-mappings".equals(element.getLocalName())) {
					loadEventConfig(element.getChildren());
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 
	 * <p>[概 要]</p>
	 * 読み込んだXMLで定義されているevent-mappings属性配下の
	 * イベント情報についてActionとの関連付けを行います。
	 * 
	 * <p>[詳 細]</p>
	 * 読み込んだXMLで定義されているevent-mappings属性配下の
	 * イベント情報についてActionとの関連付けを行います。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param eventMappings イベント情報一覧
	 * @throws Exception イベントバインディングエラー
	 */
	private void loadEventConfig(Vector<Element> events) throws Exception {
		for (Element eventElement : events) {
			String name = eventElement.getAttribute("name");
			String event = eventElement.getAttribute("event");
			String type = eventElement.getAttribute("type");
			Class<? extends BaseAction> clazz = Class.forName(type).asSubclass(
					BaseAction.class);
			this.addEventBinding(name, event, clazz);
		}
	}

}
