/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.controller;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import jp.co.fujitsu.reffi.client.nexaweb.action.Action;
import jp.co.fujitsu.reffi.client.nexaweb.model.Model;
import jp.co.fujitsu.reffi.client.nexaweb.validator.CustomValidator;

import com.nexaweb.client.ClientEvent;
import com.nexaweb.client.ClientSession;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.Element;

/**
 * <p>[概 要] </p>
 * MVC各レイヤを伝播するパラメータオブジェクトです。
 * 
 * <p>[詳 細] </p>
 * Nexaweb Runtimeから提供される情報、フレームワークが動作する為の情報、ビジネスロジックから
 * 参照が必要な情報が格納されます。<br>
 * イベント発生時にコントローラによってインスタンス生成され、
 * アクション→コントローラ→モデル→コントローラ→アクション→コントローラ<br>
 * と伝播して、イベントハンドリングが終了した後に破棄されます。<p>
 * 
 * 従来のMCO開発スタイルの場合、イベントソースエレメントを判定するには以下の定型文を記述します。
 * <pre class="samplecode">
 * ClientSession clientSession = getSession();
 * EventHandler eventHandler = clientSession.getEventHandler();
 * ClientEvent clientEvent = eventHandler.getEvent();
 * Element eventSourceElement = clientEvent.getSourceElement();
 * </pre>
 * 
 * Reffiを使用したMVC開発スタイルでは以下のようになります。
 * <pre class="samplecode">
 * Element eventSourceElement = parameterMapping.getEventSourceElement();
 * </pre>
 * 
 * 又、{@link #put(Object, Object)}を使用することで、
 * ビジネスロジック固有のパラメータを格納することが出来ます。<br>
 * MVCの各フックポイントでこのクラスの同一インスタンスを参照することで、
 * 「アクションレイヤで設定したパラメータをモデルレイヤで参照する」
 * といった処理を、広いスコープの変数を用意することなく実装出来ます。
 * 
 * <p>[備 考] </p> 
 * Reffiによって自動的にset系メソッドがコールされ、情報が設定されます。
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ParameterMapping {

	/** 具象イベントハンドラ名マッピングのキー定数です。 */
	public static final String ACTION_CLASS_NAME = "ACTION_CLASS_NAME";
	
	/** Nexawebクライアントセッションマッピングのキー定数です。 */
	public static final String CLIENT_SESSION = "CLIENT_SESSION";
	
	/** Nexawebクライアントイベントマッピングのキー定数です。 */
	public static final String CLIENT_EVENT = "CLIENT_EVENT";
	
	/** UI DOMマッピングのキー定数です。 */
	public static final String UI_DOCUMENT = "UI_DOCUMENT";
	
	/** イベントを起こしたエレメントマッピングのキー定数です。 */
	public static final String EVENT_SOURCE_ELEMENT = "EVENT_SOURCE_ELEMENT";
	
	/** XALの動的ID生成ヘッダ付加キー定数です。 */
	public static final String HEADER_ID_CREATION = "HEADER_ID_CREATION";
	
	/** コントローラによって実行されるモデルクラス群のキー定数です。 */
	public static final String MODEL_CLASSES = "MODEL_CLASSES";

	/** モデルの登録数を示す定数です。 */
	public static final String MODEL_RESERVED_NUM = "MODEL_RESERVED_NUM";
	
	/** モデルの終了数を示す定数です。 */
	public static final String MODEL_FINISHED_NUM = "MODEL_FINISHED_NUM";
	
	/** 実行されるバリデータ群のキー定数です。 */
	public static final String VALIDATORS = "VALIDATORS";

	/** 実行されるアクションのインスタンスを示すキー定数です。 */
	public static final String ACTION_INSTANCE = "ACTION_INSTANCE";
	
	/** イベントを発生させたエレメントが所属するウィンドウレベルエレメントを示すキー定数です。 */
	public static final String EVENT_SOURCE_WINDOW = "EVENT_SOURCE_WINDOW";

	/** モデル連続実行時、一モデルの終了を待つかどうかのフラグを示すキー定数です。 */
	public static final String RUN_MODELS_AND_NO_WAIT = "RUN_MODELS_AND_NO_WAIT";

	/** 起動されたイベントが持つパラメータ複製を示すキー定数です。 */
	public static final String CLIENT_EVENT_PARAMETERS = "CLIENT_EVENT_PARAMETERS";
	
	/** 起動されたイベントのクエリ文字列を示すキー定数です。 */
	public static final String QUERY_STRING = "QUERY_STRING";
	
	/** 起動されたイベントタイプを示すキー定数です。 */
	public static final String EVENT_TYPE = "EVENT_TYPE";
	
	/** イベントを起動したエレメントの名前を示すキー定数です。 */
	public static final String EVENT_SOURCE_ELEMENT_NAME = "EVENT_SOURCE_ELEMENT_NAME";
	
	/** イベントを起動したエレメントのIDを示すキー定数です。 */
	public static final String EVENT_SOURCE_ELEMENT_ID = "EVENT_SOURCE_ELEMENT_ID";
	
	/** このクラスが保持するデータが全て入ったMapオブジェクトです。 */
	private Map<Object, Object> parameters;
	
	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * parametersフィールドMap領域を初期化します。
	 * 
	 * <p>[備 考] </p>
	 */
	public ParameterMapping(){
		this.parameters = new HashMap<Object, Object>();
		
		this.parameters.put(HEADER_ID_CREATION, true);
	}
	
	/**
	 * <p>[概 要] </p>
	 * クライアントMVCレイヤ内で自由に使用出来るパラメータを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * ParameterMappingオブジェクトに対して、key=valueの形式で値を登録します。
	 * 
	 * <p>[備 考] </p>
	 * 一イベントハンドリング間のビジネスロジックに必要な、任意のパラメータ
	 * を格納する為に使用します。
	 * 
	 * @param key マッピングキー
	 * @param value マッピング値
	 */
	public void put(Object key, Object value){
		this.parameters.put(key, value);
	}
	
	/**
	 * <p>[概 要] </p>
	 * クライアントMVCレイヤ内で追加されたパラメータを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param key マッピングキー
	 * @return マッピング値
	 */
	public Object get(Object key){
		return this.parameters.get(key);
	}

	/**
	 * <p>[概 要] </p>
	 * Nexawebクライアントセッションオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientSession Nexawebクライアントセッションオブジェクト
	 */
	public void setClientSession(ClientSession clientSession){
		this.parameters.put(CLIENT_SESSION, clientSession);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexawebクライアントセッションオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return Nexawebクライアントセッションオブジェクト
	 */
	public ClientSession getClientSession(){
		return (ClientSession)this.parameters.get(CLIENT_SESSION);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexawebクライアントイベントオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientEvent Nexawebクライアントイベントオブジェクト
	 */
	public void setClientEvent(ClientEvent clientEvent) {
		this.parameters.put(CLIENT_EVENT, clientEvent);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexawebクライアントイベントオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * ClientEventを取得できるようにするには、{@link ClientConfig#setDuplicateActionInvoke(boolean)}にて
	 * にてtrueを設定する必要があります。デフォルトではこのフラグはfalseと設定されており、falseの場合nullが返ります。
	 * <br />
	 * これはNexaweb4.5.20以降、任意起動スレッドからClientEventが取得できない実装となっており、
	 * {@link ClientConfig#setDuplicateActionInvoke(boolean)}をfalseと設定している場合、
	 * アクションの多重起動を抑制するため別スレッドにてアクションを実行します。そのためClientEventを
	 * 取得することができません。
	 * 
	 * @return Nexawebクライアントイベントオブジェクト
	 */
	public ClientEvent getClientEvent() {
		return (ClientEvent)this.parameters.get(CLIENT_EVENT);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexaweb画面にレンダリングされているDOM Documentを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * このDOMオブジェクトは/uiから始まります。
	 * 
	 * @param uiDocument 画面レンダリングされているDOM Document
	 */
	public void setUiDocument(Document uiDocument){
		this.parameters.put(UI_DOCUMENT, uiDocument);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexaweb画面にレンダリングされているDOM Documentを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * このDOMオブジェクトは/uiから始まります。
	 * 
	 * @return 画面レンダリングされているDOM Document
	 */
	public Document getUiDocument(){
		return (Document)this.parameters.get(UI_DOCUMENT);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベントを起動したソースエレメントを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventSourceElement イベントソースエレメント
	 */
	public void setEventSourceElement(Element eventSourceElement){
		this.parameters.put(EVENT_SOURCE_ELEMENT, eventSourceElement);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベントを起動したソースエレメントを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return イベントソースエレメント
	 */
	public Element getEventSourceElement(){
		return (Element)this.parameters.get(EVENT_SOURCE_ELEMENT);
	}
	
	/**
	 * <p>[概 要] </p>
	 * XAL要求で動的ID付与の実施可否フラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param idCreation 実施可否 (trueで実施)
	 */
	public void setIdCreation(boolean idCreation) {
		this.parameters.put(HEADER_ID_CREATION, idCreation);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 現在設定されているXAL要求での動的ID付与の実施可否フラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 設定済み実施可否フラグ
	 */
	public boolean getIdCreation() {
		return ((Boolean)this.parameters.get(HEADER_ID_CREATION)).booleanValue();
	}
	
	/**
	 * <p>[概 要]</p>
	 * モデルクラス登録
	 * 
	 * <p>[詳 細]</p>
	 * 実行するモデルクラスを登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param modelClasses 実行するモデルクラス
	 */
	public void setModelClasses(List<Class<? extends Model>> modelClasses) {
		this.parameters.put(MODEL_CLASSES, modelClasses);
	}
	
	/**
	 * <p>[概 要]</p>
	 * モデルクラス一覧取得
	 * 
	 * <p>[詳 細]</p>
	 * 登録されているモデルクラス一覧を取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return 登録されているモデルクラス一覧
	 */
	@SuppressWarnings("unchecked")
	public List<Class<? extends Model>> getModelClasses() {
		return (List<Class<? extends Model>>)this.parameters.get(MODEL_CLASSES);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 登録モデル数設定
	 * 
	 * <p>[詳 細]</p>
	 * 登録したモデル数を設定します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param modelReservedNum 登録するモデル数
	 */
	public void setModelReservedNum(int modelReservedNum) {
		this.parameters.put(MODEL_RESERVED_NUM, modelReservedNum);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 登録モデル数取得
	 * 
	 * <p>[詳 細]</p>
	 * 登録したモデル数を取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return 登録したモデル数
	 */
	public int getModelReservedNum() {
		return ((Integer)this.parameters.get(MODEL_RESERVED_NUM)).intValue();
	}
	
	/**
	 * <p>[概 要]</p>
	 * 終了モデル数設定
	 * 
	 * <p>[詳 細]</p>
	 * 終了したモデル数を設定します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param modelFinishedNum 終了したモデル数
	 */
	public void setModelFinishedNum(int modelFinishedNum) {
		this.parameters.put(MODEL_FINISHED_NUM, modelFinishedNum);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 終了モデル数取得
	 * 
	 * <p>[詳 細]</p>
	 * 終了したモデル数を取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return 終了したモデル数
	 */
	public int getModelFinishedNum() {
		return ((Integer)this.parameters.get(MODEL_FINISHED_NUM)).intValue();
	}
	
	/**
	 * <p>[概 要]</p>
	 * バリデータ登録
	 * 
	 * <p>[詳 細]</p>
	 * 実行するバリデータ一覧を登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param validators 実行するバリデータ一覧
	 */
	public void setValidators(List<CustomValidator> validators) {
		this.parameters.put(VALIDATORS, validators);
	}
	
	/**
	 * <p>[概 要]</p>
	 * バリデータ一覧取得
	 * 
	 * <p>[詳 細]</p>
	 * 登録されているバリデータ一覧を取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return 登録されているバリデータ一覧
	 */
	@SuppressWarnings("unchecked")
	public List<CustomValidator> getValidators() {
		return (List<CustomValidator>)this.parameters.get(VALIDATORS);
	}

	/**
	 * <p>[概 要]</p>
	 * アクション登録
	 * 
	 * <p>[詳 細]</p>
	 * 実行するアクションを登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param action 実行するアクション
	 */
	public void setActionInstance(Action action) {
		this.parameters.put(ACTION_INSTANCE, action);
	}
	
	/**
	 * <p>[概 要]</p>
	 * アクション取得
	 * 
	 * <p>[詳 細]</p>
	 * 登録されているアクションを取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param 登録されているアクション
	 */
	public Action getActionInstance() {
		return (Action)this.parameters.get(ACTION_INSTANCE);
	}
	
	/**
	 * <p>[概 要]</p>
	 * イベントソースウィンドウ登録
	 * 
	 * <p>[詳 細]</p>
	 * イベントが実行されたエレメントが所属するウィンドウを登録します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param window イベント実行されたエレメントが所属するウィンドウ
	 */
	public void setEventSourceWindow(Element eventSourceWindow) {
		this.parameters.put(EVENT_SOURCE_WINDOW, eventSourceWindow);
	}
	
	/**
	 * <p>[概 要]</p>
	 * イベントソースウィンドウ取得
	 * 
	 * <p>[詳 細]</p>
	 * 登録されているウィンドウを取得します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return イベント実行されたエレメントが所属するウィンドウ
	 */
	public Element getEventSourceWindow() {
		return (Element)this.parameters.get(EVENT_SOURCE_WINDOW);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 複数予約されたモデルの実行動作を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * モデル群がコントローラに連続実行される際、モデルの処理結果を待たずに
	 * 次のモデルを実行するかどうかのフラグを取得します。
	 * 
	 * <p>[備 考] </p>
	 * モデルをシーケンシャルに実行しない場合、モデルは登録された順序で実行されますが
	 * レスポンスを待たずに次のモデルが実行されます。
	 * 
	 * @return シーケンシャルに実行しない場合はtrue、それ以外はfalse
	 */
	public boolean isRunModelsAndNoWait(){
		return ((Boolean)this.parameters.get(RUN_MODELS_AND_NO_WAIT)).booleanValue();
	}
	
	/**
	 * <p>[概 要] </p>
	 * 複数予約されたモデルの実行動作を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * モデル群がコントローラに実行される際、モデルの処理結果を待たずに
	 * 次モデルを実行するかどうかのフラグを設定します。
	 * 
	 * <p>[備 考] </p>
	 * モデルをシーケンシャルに実行しない場合、モデルは登録された順序で実行されますが
	 * レスポンスを待たずに次のモデルが実行されます。
	 * 
	 * @param runModelsAndNoWait シーケンシャルに実行しない場合はtrue、それ以外はfalse
	 */
	public void setRunModelsAndNoWait(boolean runModelsAndNoWait) {
		this.parameters.put(RUN_MODELS_AND_NO_WAIT, runModelsAndNoWait);
	}
	
	/**
	 * <p>[概 要] </p>
	 * クライアントイベントパラメータマップを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * ClientEvent#getParameter(String key)が対象とするリソースと同義のマップを返却します。<br>
	 * 取得するMapは以下のような形式になります。<br>
	 * {id=nx:8886-1045, event=onActiveLost, oppositeSubcomponent=, active=false, oppositeId=nx:8886-1035}
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @return クライアントイベントパラメータマップ
	 */
	@SuppressWarnings("unchecked")
	public Map<String, String> getClientEventParameters() {
		return (Map<String, String>)this.parameters.get(CLIENT_EVENT_PARAMETERS);
	}
	
	/**
	 * <p>[概 要] </p>
	 * クライアントイベントパラメータマップを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * ClientEvent#getParameter(String key)が対象とするリソースと同義のマップを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param clientEventParameters クライアントイベントパラメータマップ
	 */
	public void setClientEventParameters(Map<String, String> clientEventParameters) {
		this.parameters.put(CLIENT_EVENT_PARAMETERS, clientEventParameters);
	}
	
	/**
	 * <p>[概 要] </p>
	 * クライアントイベントパラメータマップから、引数keyに対応する値を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link ClientEvent#getParameter(String)}と同様の機能を提供します。<br>
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @param key クライアントイベントパラメータマップのキー
	 * @return 引数keyに対する値
	 */
	public String getClientEventParameter(String key) {
		return getClientEventParameters().get(key);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント発生時のクエリ文字列を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link ClientEvent#getQueryString()}と同様の処理を行います。<br>
	 * 取得するクエリ文字列は以下のような形式になります。<br>
	 * oppositeSubcomponent=&event=onActiveLost&oppositeId=nx%3a8886-1035&active=false&id=nx%3a8886-1045
	 *
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @return イベント発生時のクエリ文字列
	 */
	public String getQueryString() {
		return (String)this.parameters.get(QUERY_STRING);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント発生時のクエリ文字列を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param queryString イベント発生時のクエリ文字列
	 */
	public void setQueryString(String queryString) {
		this.parameters.put(QUERY_STRING, queryString);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント発生時のイベントタイプ文字列を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * ClientEventパラメータのeventキーに対する値を返却します。
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @return イベント発生時のイベントタイプ文字列
	 */
	public String getEventType() {
		return (String)this.parameters.get(EVENT_TYPE);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベント発生時のイベントタイプ文字列を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * ClientEventパラメータのeventキーに対する値を設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventType イベント発生時のイベントタイプ文字列
	 */
	public void setEventType(String eventType) {
		this.parameters.put(EVENT_TYPE, eventType);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベントを発生させたエレメントの名前を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link ClientEvent#getSource()}.getAttribute("name")と同等の処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @return イベントを発生させたエレメントのname属性値
	 */
	public String getEventSourceElementName() {
		return (String)this.parameters.get(EVENT_SOURCE_ELEMENT_NAME);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベントを発生させたエレメントの名前を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventSourceElementName イベントを発生させたエレメントのname属性値
	 */
	public void setEventSourceElementName(String eventSourceElementName) {
		this.parameters.put(EVENT_SOURCE_ELEMENT_NAME, eventSourceElementName);
	}

	/**
	 * <p>[概 要] </p>
	 * イベントを発生させたエレメントのIDを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * {@link ClientEvent#getSource()}.getAttribute("id")と同等の処理を行います。
	 * 
	 * <p>[備 考] </p>
	 * Nexaweb 4.5.20から、パフォーマンス向上の為、任意のスレッドから
	 * ClientEventオブジェクトを参照することが出来ません（設定で可）。<br>
	 * このメソッドを使用することで、同様の情報を取得することが出来ます。
	 * 
	 * @return イベントを発生させたエレメントのid属性値
	 */
	public String getEventSourceElementId() {
		return (String)this.parameters.get(EVENT_SOURCE_ELEMENT_ID);
	}
	
	/**
	 * <p>[概 要] </p>
	 * イベントを発生させたエレメントのIDを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param eventSourceElementId イベントを発生させたエレメントのid属性値
	 */
	public void setEventSourceElementId(String eventSourceElementId) {
		this.parameters.put(EVENT_SOURCE_ELEMENT_ID, eventSourceElementId);
	}

	/**
	 * <p>[概 要] </p>
	 * このクラスオブジェクトの複製を返却します。
	 * 
	 * <p>[詳 細] </p>
	 * ParameterMappingオブジェクト、及び保持するプロパティは完全なコピーを作りますが、
	 * parametersプロパティ内で保持する定数キー要素値のアドレス参照はコピーしません。
	 * <br />
	 * parameterMapping#parametersと、<br>
	 * clonedParameterMapping#parameters<br>
	 * は異なるアドレスで保持されますが、
	 * <br />
	 * parameterMapping#getEventSourceElementと、<br>
	 * clonedParameterMapping#getEventSourceElement<br>
	 * は同じアドレスを返却します。
	 * 
	 * <p>[備 考] </p>
	 * このメソッドはアクションからコントローラに別アクション実行を委譲する場合等に使用します。<br>
	 * <pre class="samplecode">
     *    protected boolean prepare(ParameterMapping parameterMapping) throws Exception {
     *        this.getController().invoke(OtherAction.class, parameterMapping.clone());
     *        return true;
     *    }
     * </pre>
     * コントローラはコールバック（successForwardやfailureForward）先アクションアドレスを
     * ParameterMapping#getActionInstanceで判別しますが、invokeが呼ばれると左記のアドレスを指定
     * された新規アクションのアドレスで上書きします。<br>
     * 上記のようにinvokeを呼び出すことで、コントローラが認識するアクションアドレスの上書き
     * を抑止することが出来ます。
	 * 
	 * @return 複製されたParameterMappingオブジェクト
	 */
	public ParameterMapping clone() {
		ParameterMapping pm = new ParameterMapping();
		for (Iterator<Object> itr = this.parameters.keySet().iterator(); itr.hasNext(); ) {
			Object key = itr.next();
			pm.put(key, this.parameters.get(key));
		}
		return pm;
	}
}
