/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.logconsumer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;

import jp.co.fujitsu.reffi.client.nexaweb.controller.ClientConfig;

import com.nexaweb.util.LogEvent;

/**
 * <p>[概 要] </p>
 * クライアントログイベントをハンドリングしてローカルファイルに書き込むクラスです。
 * 
 * <p>[詳 細] </p>
 * logイベント受信時にファイルにログメッセージ内容を書き込みます。<br>
 * ログファイルのファイル名パターンは「prefix-YYYYMMDD-suffix」です。<br>
 * initメソッドに渡されるClientConfigオブジェクトの以下設定項目によって動作設定が行われます。()内はデフォルト値です。<br>
 * <ul>
 *     <li>ログファイルを格納するディレクトリ名（reff-client-log）</li>
 *     <li>ログファイルのキャラセット(UTF-8)</li>
 *     <li>ディレクトリ内最大ログファイル数(100)</li>
 *     <li>１ログファイルの最大ファイルサイズ(10485760)</li>
 *     <li>サイズ超過時の最大バックアップ数(10)</li>
 *     <li>ログファイル名の接頭語(reff-client-)</li>
 *     <li>ログファイル名の接尾語(.log)</li>
 * </ul>
 * これらの値を変更する場合は、
 * {@link jp.co.fujitsu.reffi.client.nexaweb.controller.BaseController#initialize()}
 * を参照して下さい。
 * 
 * <p>[備 考] </p>
 * このLogConsumerはローカルファイルにアクセスする為、アプレット起動時はデジタル署名が必要です。<p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class FileLogConsumer extends LogConsumerAdapter {

	/** ログファイルが格納されるディレクトリパスです。 */
	private String directoryPath;

	/** ディレクトリ内に存在するログファイル上限です。 */
	private int maxFileExist;

	/** 一ログファイルの最大サイズです。 */
	private long maxFileSize; 
	
	/** サイズローテーション後のバックアップ最大数です。.nの形で接尾されます。 */
	private int maxBackupIndex;
	
	/** ログファイル名の接頭文字です。 */
	private String prefix;
	
	/** ログファイル名の拡張子です。 */
	private String suffix;
	
	/** ログ出力文字セット（デフォルトはUTF-8） */
	private String encoding = "UTF-8";

	/** ログファイルへの書き込みストリームです。 */
	private Writer writer;
	

	/**
	 * <p>[概 要] </p>
	 * ログファイルが格納されるディレクトリパスを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * directoryPathフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 *
	 * @return ログファイルが格納されるディレクトリパス
	 */
	public String getDirectoryPath(){
		return this.directoryPath;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルが格納されるディレクトリ名を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * directoryPathフィールドに引数directoryPathを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param directoryPath ログファイルが格納されるディレクトリパス
	 */
	public void setDirectoryPath(String directoryPath){
		this.directoryPath = directoryPath;
	}

	/**
	 * <p>[概 要] </p>
	 * ディレクトリ内に存在するログファイル上限を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * maxFileExistフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ディレクトリ内に存在するログファイル上限
	 */
	public int getMaxFileExist() {
		return maxFileExist;
	}

	/**
	 * <p>[概 要] </p>
	 * ディレクトリ内に存在するログファイル上限を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * maxFileExistフィールドを引数maxFileExistで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param maxFileExist ディレクトリ内に存在するログファイル上限
	 */
	public void setMaxFileExist(int maxFileExist) {
		this.maxFileExist = maxFileExist;
	}

	/**
	 * <p>[概 要] </p>
	 * 一ログファイルの最大サイズを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * maxFileSizeフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 一ログファイルの最大サイズ
	 */
	public long getMaxFileSize() {
		return maxFileSize;
	}

	/**
	 * <p>[概 要] </p>
	 * 一ログファイルの最大サイズを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * maxFileSizeフィールドに引数maxFileSizeを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param maxFileSize 一ログファイルの最大サイズ
	 */
	public void setMaxFileSize(long maxFileSize) {
		this.maxFileSize = maxFileSize;
	}

	/**
	 * <p>[概 要] </p>
	 * サイズローテーション後のバックアップ最大数を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * maxBackupIndexフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return サイズローテーション後のバックアップ最大数
	 */
	public int getMaxBackupIndex() {
		return maxBackupIndex;
	}

	/**
	 * <p>[概 要] </p>
	 * サイズローテーション後のバックアップ最大数を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * maxBackupIndexフィールドに引数maxBackupIndexを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param maxBackupIndex サイズローテーション後のバックアップ最大数
	 */
	public void setMaxBackupIndex(int maxBackupIndex) {
		this.maxBackupIndex = maxBackupIndex;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接頭文字を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * prefixフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイル名の接頭文字
	 */
	public String getPrefix() {
		return prefix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接頭文字を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * prefixフィールドに引数prefixを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param prefix ログファイル名の接頭文字
	 */
	public void setPrefix(String prefix) {
		this.prefix = prefix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の拡張子を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * suffixフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイル名の拡張子
	 */
	public String getSuffix() {
		return suffix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の拡張子を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * suffixフィールドに引数suffixを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param suffix ログファイル名の拡張子
	 */
	public void setSuffix(String suffix) {
		this.suffix = suffix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログ出力キャラセットを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * encodingフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログ出力キャラセット
	 */
	public String getEncoding(){
		return this.encoding;
	}

	/**
	 * <p>[概 要] </p>
	 * ログ出力キャラセットを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * encodingフィールドに引数encodingを設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param encoding ログ出力キャラセット
	 */
	public void setEncoding(String encoding){
		this.encoding = encoding;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルへの書き込みストリームを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * writerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイルへの書き込みストリーム
	 */
	public Writer getWriter() {
		return writer;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルへの書き込みストリームを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * writerフィールドを引数writerで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param writer ログファイルへの書き込みストリーム
	 */
	public void setWriter(Writer writer) {
		this.writer = writer;
	}

	/**
	 * <p>[概 要] </p>
	 * このLogConsumerを初期化します。
	 * 
	 * <p>[詳 細] </p>
	 * ClientConfig初期化オブジェクトから以下を取得してフィールドに設定します。
	 * <ul>
	 *     <li>getLogFileDir()：ログファイルを格納するディレクトリ名</li>
	 *     <li>getLogFileCharset()：ログファイルのキャラセット</li>
	 *     <li>getLogFileMaxExist()：ディレクトリ内最大ログファイル数</li>
	 *     <li>getLogFileMaxSize()：１ログファイルの最大ファイルサイズ</li>
	 *     <li>getLogFileMaxBackupIndex()：サイズ超過時の最大バックアップ数</li>
	 *     <li>getLogFileNamePrefix()：ログファイル名の接頭語</li>
	 *     <li>getLogFileNameSuffix()：ログファイル名の接尾語</li>
	 * </ul>
	 * 
	 * ログファイル格納ディレクトリに関しては以下のように加工して設定されます。<br>
	 * System.getProperties("user.home") / 指定されたディレクトリ名<br>
	 * このディレクトリが存在しない場合、新規に作成されます。
	 * 
	 * <p>[備 考] </p>
	 * Properties以外のobjは初期化パラメータとして認識されません。
	 * 
	 * @param obj 初期化パラメータ
	 */
	@Override
	public void init(Object obj) {
		super.init(obj);
		
		if(obj instanceof ClientConfig) {
			ClientConfig config = (ClientConfig)obj;

			// ログファイルを保存するディレクトリ名を取得、生成
			String directory = config.getLogFileDir();
			String directoryPath = System.getProperty("user.home") + File.separator + directory;
			if(!new File(directoryPath).isDirectory()){
				new File(directoryPath).mkdirs();
			}
			setDirectoryPath(directoryPath);
			
			// ログファイルの文字セットを取得
			String encoding = config.getLogFileCharset();
			setEncoding(encoding);
			
			// ディレクトリ内に格納する最大ログファイル数を取得
			int maxFileExist = config.getLogFileMaxExist();
			setMaxFileExist(maxFileExist);
			
			// １ログファイルの最大サイズを取得
			long maxFileSize = config.getLogFileMaxSize();
			setMaxFileSize(maxFileSize);

			// 最大サイズを超えたログファイルを何個保持するかの値を取得
			int maxBackupIndex = config.getLogFileMaxBackupIndex();
			setMaxBackupIndex(maxBackupIndex);
			
			// ログファイル名のプレフィックスを取得
			String prefix = config.getLogFileNamePrefix();
			setPrefix(prefix);
			
			// ログファイル名の拡張子を取得
			String suffix = config.getLogFileNameSuffix();
			setSuffix(suffix);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * ログイベント受信ハンドラです。
	 * 
	 * <p>[詳 細] </p>
	 * LogConsumer#log(LogEvent)を実装します。
	 * 初期化された設定情報を元にファイルにログを追記、ローテーションします。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param evt Logクラスが発行したログイベント
	 */
	@Override
	public void log(LogEvent evt) {
		try{
			// 対象ログファイルインスタンスを生成
			File logFile = new File(logFilePath());
			
			// Maxファイルサイズを超過、インデックスによるローテーション
			if(logFile.length() > getMaxFileSize()) {
				rotateBySize(logFile);
			}
			
			if(!logFile.exists()){
				if(getWriter() != null) {
					// ファイルは無いがストリームは有る場合
					// 日付変更によるファイル切り替え時
					// 前日のファイルへのストリームを閉じる
					closeWriter();
				}
				logFile.createNewFile();
				// ファイルが増えたので最大ファイル数チェック
				rotateByMaxExist();
				Writer writer = createWriter(logFile);
				setWriter(writer);
			}else if(getWriter() == null) {
				// ファイルは有るがストリームが無い場合
				// 当日二回目以降起動時
				Writer writer = createWriter(logFile);
				setWriter(writer);
			}

			
			// 出力するログを整形
			String log = 
				createLogString(evt) + System.getProperty("line.separator");
			// 書き込み
			getWriter().write(log, 0, log.length());
			getWriter().flush();
		}catch(IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * 指定最大byteを超えた書き込み対象ログファイルのバックアップをとります。
	 * 
	 * <p>[詳 細] </p>
	 * getMaxBackupIndex()の返却値が0以上の場合、ファイル名に.1を付与します。<br>
	 * ファイル名.1が既に存在している場合、ファイル名.1はファイル名.2にリネームされます。<br>
	 * .nがgetMaxBackupIndex()を超えた場合、最も古いバックアップファイルは削除されます。<br>
	 * 
	 * <p>[備 考] </p>
	 * getMaxBackupIndex()の返却値が0の場合はサイズバックアップは行われません。
	 * 
	 * @param logFile 書き込み中のログファイルインスタンス
	 * @throws IOException
	 */
	protected void rotateBySize(File logFile) throws IOException {
		// 最大保存インデックス（～.n）を取得
		int maxBackupIndex = getMaxBackupIndex();

		if(maxBackupIndex > 0) {
		    File target;
			File file;
			
			// 書き込みストリームを閉じる
			closeWriter();
			
			// 最大保存インデックスのログファイルが存在する場合は削除
			file = new File(logFilePath() + '.' + maxBackupIndex);
			if (file.exists()) {
				file.delete();
			}
			
			// 保存インデックスの古い順に降順ローテーション
			for (int i = maxBackupIndex - 1; i >= 1; i--) {
				file = new File(logFilePath() + "." + i);
				if (file.exists()) {
					target = new File(logFilePath() + '.' + (i + 1));
					file.renameTo(target);
				}
			}
	
			// 直前まで対象だったログファイルに最新インデックスを付与
			target = new File(logFilePath() + "." + 1);
			file = new File(logFilePath());
			file.renameTo(target);
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * ディレクトリ内のログファイルが指定最大数を超えた場合、古いログファイルを消去します。
	 * 
	 * <p>[詳 細] </p>
	 * getDirectoryPath()で求められるディレクトリから全ログファイルを取得、
	 * ファイル数を求めます。<br>
	 * （この際、ログファイルの識別にgetPrefix()が使用されます。）<br>
	 * このファイル数がgetMaxFileExist()で得られる値を超過していた場合、
	 * 超過数分のログファイルを古い順に削除します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void rotateByMaxExist() {
		// 格納ディレクトリ内のログファイルをリストする
		File logDir = new File(getDirectoryPath());
		File[] logFiles = logDir.listFiles(new FilenameFilter(){
			public boolean accept(File dir, String filename) {
				boolean ret = false;
				if(filename.startsWith(getPrefix())) {
					ret = true;
				}
				return ret;
			}
		});
		
		// 超過分を割り出す
		int over = logFiles.length - getMaxFileExist(); 
		if(over > 0){
			// 古い順に並び替え
			Arrays.sort(logFiles, new Comparator<File>(){
				public int compare(File o1, File o2){
					Long l1 = new Long(((File)o1).lastModified());
					Long l2 = new Long(((File)o2).lastModified());
	
					return l1.compareTo(l2);
				}
			});
	
			// 超過分削除
			for(int i=0; i<over; i++){
				logFiles[i].delete();
			}
		}
	}
	
	/**
	 * <p>[概 要] </p>
	 * ログファイルへの書き込みストリームを生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数logFileへのOutputStreamWriterを、{@link #getEncoding()}エンコーディング
	 * で作成、返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFile 書き込み対象にするログファイル
	 * @return ログファイルへの書き込みストリーム
	 * @throws IOException
	 */
	protected OutputStreamWriter createWriter(File logFile) throws IOException {
		FileOutputStream fos = new FileOutputStream(logFile, true);
		OutputStreamWriter osw = new OutputStreamWriter(fos, getEncoding());
		
		return osw;
	}
	
	/**
	 * <p>[概 要] </p>
	 * ログファイルへの書き込みストリームを閉じます。
	 * 
	 * <p>[詳 細] </p>
	 * getWriter()がnullでは無い場合、getWriter().close()をコールします。<br>
	 * ストリームが閉じた後、setWriter(null)を行います。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @throws IOException
	 */
	protected void closeWriter() throws IOException {
		if(getWriter() == null) return;
		getWriter().close();
		setWriter(null);
	}
	
	/**
	 * <p>[概 要] </p>
	 * ログファイル名を生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 「プレフィックス + YYYYMMDD + サフィックス」のパターンでファイル名を生成します。<br>
	 * プレフィックスとサフィックスにはClientConfigオブジェクトの以下の値を使用します。<br>
	 * <ul>
	 *     <li>ClientConfig#getLogFileNamePrefix()</li>
	 *     <li>ClientConfig#getLogFileNameSuffix()</li>
	 * </ul>
	 * 
	 * <p>[備 考] </p>
	 * プレフィックスとサフィックスを変更する場合は
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.controller.BaseController#initialize()}
	 * を参照して下さい。
	 * 
	 * @return ログファイル名
	 */
	protected String logFileName() {
		Date date = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMdd");
		String yyyymmdd = sdf.format(date);
		String filename = getPrefix() + yyyymmdd + getSuffix();

		return filename;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 生成されるログファイルのフルパスを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * System.getProperties("user.home") / 指定されたディレクトリ名 / ログファイル名<br>
	 * を返却します。<br>
	 * 「指定されたディレクトリ名」にはClientConfig#getLogFileDir()の返却値を使用します。
	 * 
	 * <p>[備 考] </p>
	 * ディレクトリ名を変更する場合は
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.controller.BaseController#initialize()}
	 * を参照して下さい。
	 * 
	 * @return ログファイルのフルパス
	 */
	protected String logFilePath() {
		return getDirectoryPath() + File.separator + logFileName();
	}
}
