/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.model;

import java.io.Serializable;

import jp.co.fujitsu.reffi.client.nexaweb.event.ModelProcessEvent;

import com.nexaweb.util.XmlSerializable;

/**
 * <p>[概 要] </p>
 * トピックに発行する為の機能モデルクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b>
 * <pre class="samplecode">
 * 	public class RemarkSendAction extends BaseAction {
 *
 *		&#064;Override
 *		protected void reserveModels(List<Class<? extends Model>> models) {
 *			// publishモデルを実行予約
 *			models.add(PublishCore.class);
 *		}
 *
 *		&#064;Override
 *		public void nextModel(int index, Model prev, Model next) throws Exception {
 *			switch(index){
 *				case 0:
 *					String channelName = 
 *						getElementValueByName("channelName").getElementValue(0).getValue();
 *					String chatRemark = 
 *						getElementValueByName("chatRemark").getElementValue(0).getValue();
 *					String handleName = 
 *						getElementValueByName("handleName").getElementValue(0).getValue(); 
 *
 *					// publish対象のトピック名を設定
 *					((PublishCore)next).setTopic(channelName);
 *					// publishするオブジェクトを設定
 *					((PublishCore)next).setPublishObject("<" + handleName + "> " + chatRemark);
 *					break;
 *			}
 *		}
 *	}
 *
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class PublishCore extends BaseModel {

	/** 発行対象トピックです。 */
	private String topic;

	/** 発行するオブジェクトです。 */
	private Object publishObject;

	/**
	 * <p>[概 要] </p>
	 * 発行対象トピックを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * topicフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 発行対象トピック
	 */
	public String getTopic() {
		return this.topic;
	}

	/**
	 * <p>[概 要] </p>
	 * 発行対象トピックを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * topicフィールドを引数topicで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param topic 発行対象トピック
	 */
	public void setTopic(String topic) {
		this.topic = topic;
	}

	/**
	 * <p>[概 要] </p>
	 * 発行するオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * publishObjectフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 発行するオブジェクト
	 */
	public Object getPublishObject() {
		return publishObject;
	}

	/**
	 * <p>[概 要] </p>
	 * 発行するオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * publishObjectフィールドを引数publishObjectで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param publishObject 発行するオブジェクト
	 */
	public void setPublishObject(Object publishObject) {
		this.publishObject = publishObject;
	}

	/**
	 * <p>[概 要] </p>
	 * publishを行います。
	 * 
	 * <p>[詳 細] </p>
	 * getPublishObject()で取得したオブジェクトを、getTopic()で取得したトピックに発行します。<br>
	 * 発行した時点で例外が発生しなかった場合、モデル処理成功イベントを発火します。<br>
	 * その際ModelProcessEventオブジェクトが生成されますが、resultにはnullが入ります。<br>
	 * これは一方的なpublishである為、戻り値が無い為です。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	@Override
	protected void mainproc() throws Exception {
		// フィールドに設定されたpublishするオブジェクトを取得
		Object publishObject = getPublishObject();
		
		// 取得したpublishオブジェクトの型毎にnexawebのpublish APIをコール
		if(publishObject instanceof String){
			getController().getSession().getMessagingService().publish(topic, (String)publishObject);
		}else if(publishObject instanceof XmlSerializable){
			getController().getSession().getMessagingService().publish(topic, (XmlSerializable)publishObject);
		}else if(publishObject instanceof Serializable){
			getController().getSession().getMessagingService().publish(topic, (Serializable)publishObject);
		}
		
		// publishするだけのモデルなので、この時点で例外が発生しなければ処理成功と見做す
		// Model処理成功イベントを生成
		ModelProcessEvent evt = new ModelProcessEvent(this);
		// 一方的なPublishの為、モデル処理結果は無し
		evt.setResult(null);
		// 発火
		fireModelSuccess(evt);
		
		fireModelFinished(new ModelProcessEvent(this));
	}
}
