/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.action {

	import flash.events.Event;
	
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	import jp.co.fujitsu.reffi.client.flex.validator.ValidateErrors;
	
	/**
	 * <p>[概 要] </p>
	 * イベント発生時、コントローラによって起動されるActionの基底クラスです.
	 * 
	 * <p>[詳 細] </p>
	 * BaseController#bind(EventBinder)によって登録したイベントに対応して
	 * このクラスの継承アクションクラスが起動されます。<br>
	 * 実行される具象アクションクラスには、テンプレートコールされる以下の処理を実装します。<br>
	 * 
	 * <ul>
	 *     <li>前処理、クライアント完結処理 : prepare(ParameterMapping)</li>
	 *     <li>画面入力値のチェック : validators(Array),　validationFault(ValidateErrors)</li>
	 *     <li>実行モデルの予約、設定 : reserveModels(Array),　nextModel(int, ModelProcessEvent, Model)</li>
	 *     <li>実行モデルの成功時ハンドリング : successForward(int, Model, Object)</li>
	 *     <li>実行モデルの失敗時ハンドリング : failureForward(int, Model, Exception)</li>
	 *     <li>登録した全モデルの完了時ハンドリング : complete()</li>
	 * </ul>
	 * これらは全て必須では無く、任意での実装になります。<br>
	 * これらのメソッドはBaseActionのrunメソッド内コールフロー、
	 * 及びBaseControllerからコールバックされる為、任意での呼び出しは不要です。<br>
	 * 
	 * <p>[備 考] </p>
	 * 上記のようにこの基底クラスは、処理フローを実装して開発手順の型決めを行っています。<br>
	 * 全てのモデル処理結果をsucessForward、failureForwardでハンドリングするのでは無い場合はDispatchActionを、<br>
	 * reserveModel、nextModelによるモデル実行制御を行わない場合は、FlexibleActionを基底クラスとして選択して下さい。
	 * <p>
	 * 
	 * @example 典型的なBaseAction継承アクション
	 * <listing>
        package demo.client.pdf.action{
            
            import flash.events.Event;
            
            import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
            import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
            import jp.co.fujitsu.reffi.client.flex.model.events.ModelProcessEvent;
            import jp.co.fujitsu.reffi.client.flex.model.rpc.RpcControllerClientCore;
            
            import mx.controls.Alert;
            import mx.controls.ComboBox;
            import mx.controls.DataGrid;
            import mx.rpc.events.ResultEvent;
        
            public class CreateTableAction extends BaseAction{
                
                // ① RPC呼び出しモデルを予約します（このケースではサーバ側プログラムとしてRPCControllerを使用）
                override protected function reserveModels(models:Array):void{
                    models.push(Class(RPCControllerClientCore));
                }
                
                // ② 生成されたRPCControllerClientCoreインスタンスにパラメータを設定
                override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                    // index 0としてRPCControllerClientCoreインスタンスが引数nextに入っています
                    if (index == 0) {
                        var dataCount:String = ComboBox(getComponentById("dataCount")).text;
                        // サーバ側実行モデルをFQCN指定
                        RpcControllerClientCore(next).modelFQCN = "demo.rpc.model.PDFTableModel";
                        RpcControllerClientCore(next).addRpcParameter("dataCount", dataCount);
                    }
                    return true;
                }
                
                // ③ RPC正常終了処理を記述します
                override public function successForward(index:int, model:BaseModel, resultEvent:Event):void{
                    var event:ResultEvent = resultEvent as ResultEvent;
                    DataGrid(getComponentByName("pdfWindowTable")).dataProvider = event.result;
                }
                
                // ④ RPC異常終了処理を記述します
                override public function failureForward(index:int, model:BaseModel, faultEvent:Event):Event{
                    Alert.show("テーブル作成に失敗しました。");
                    return faultEvent;
                }
            }
        }
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class BaseAction extends AbstractAction {

		/**
		 * <p>[概 要]</p>
		 * コントローラにコールされるアクションの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 継承クラスから情報を収集してコントローラに返却します。
		 * 以下の順にテンプレートメソッドがコールされます。
		 * 	<ol>
		 * 		<li>prepare(ParameterMapping)　：　前準備の実装、クライアント完結コードの実装</li>
		 * 		<li>validators(Array)　：　バリデータオブジェクトの登録</li>
		 * 		<li>validate(Array)　：　バリデーションの実行</li>
		 * 		<li>validationFault(ValidateErrors)　：　バリデーションエラーハンドリング</li>
		 * 		<li>reserveModels(Array)　：　実行モデルの登録</li>
		 * 	</ol>
		 * <p>
		 * 上記のコールフロー終了後、ParameterMappingオブジェクトがコントローラに返却されます。
		 * <p>
		 * prepare実装メソッドでfalseを返却した場合、又はvalidateメソッドが一つでも
		 * ValidateErrorオブジェクトを返却した場合、<br>
		 * コントローラに返却されるParameterMappingオブジェクトはnullになります。<br>
		 * nullのParameterMappingを受け取ったコントローラは以降の処理を中止します。
		 * 
		 * <p>[備 考]</p>
		 * このメソッドを実装する必要は有りません。
		 * <p>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @return 継承Actionから収集した、コントローラに返却される制御情報
		 */
        override public function run(parameterMapping:ParameterMapping):ParameterMapping{
        	try{
        		super.run(parameterMapping);
				
				// コントローラ委譲前の前処理メソッドをテンプレートコール
				// 戻り値がfalseであればその後の処理は行わない
				if(!prepare(parameterMapping)){
					return null;
				}
	
				// 予約されたバリデータを取得
				var vs:Array = new Array();
				validators(vs);
				
				// バリデーション実行
				var validateErrors:ValidateErrors = validate(vs);
				// バリデーションエラーが発生した場合、validationFaultをテンプレートコール
				if(validateErrors.hasError()) {
					validationFault(validateErrors);
					return null;
				}
							
				// コントローラにシーケンシャル実行させる業務モデル群をオーバーライドメソッドから取得
				var models:Array = new Array();
				reserveModels(models);
				
				
				// 全モデル終了までwaitするかどうかのフラグをオーバーライドメソッドから取得
				var syncFlg:Boolean = isWaitModelsDone();
	
				// モデル群実行時、１モデルの結果を待たずに次のモデルを実行するかどうかのフラグを取得
				var isRunModelsAndNoWait:Boolean = isRunModelsAndNoWait();
	
				// パラメータにマッピング
				parameterMapping.models = models;
				parameterMapping.validators = vs;
				parameterMapping.waitModelsDone = syncFlg;
				parameterMapping.waitModelsMessage = waitModelsMessage();
				parameterMapping.isRunModelsAndNoWait = isRunModelsAndNoWait;
        	}catch(e:Error) {
        		e = trap(e);
        		if(e) {
        			throw e;
        		}
        	}
        	
			return parameterMapping;
        }

		/**
		 * <p>[概 要] </p>
		 * BaseAction内で最初にテンプレートコールされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * 単純なDOM操作等、クライアント内で完結するようなイベント処理や、
		 * 条件を判断して以降の処理を実行しない、<br>
		 * 等の処理を実装する場合は、このメソッドをオーバーライドして処理を記述して下さい。<br>
		 * オーバーライド先でfalseを返却すると、それ以降のアクション処理は行われません。
		 *  
		 * <p>[備 考] </p>
		 *
		 * @example
		 * 条件によって処理を中止する（発言欄が空の状態で送信ボタンが押下された場合は送信しない） 
		 * 
		 * <listing version="3.0">
             override protected prepare(parameterMapping:ParameterMapping):ParameterMapping {
                 String chatRemark = 
                     TextInput(getComponentByName("chat.chatRemark")).text;
             
                 if(chatRemark.length == 0){
                     return false;
                 }else{
                     return true;
                 }
             }
		 * </listing>
		 * 
		 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
		 * @return 以降の処理を継続するかどうかのフラグ
		 */
		protected function prepare(parameterMapping:ParameterMapping):Boolean{
			return true;
		}

		/**
		 * <p>[概 要]</p>
		 * このイベントアクションフローで必要なバリデーション情報を指定追加します.
		 * 
		 * <p>[詳 細]</p>
		 * 具象アクションでこのメソッドをオーバーライドして、引数validatorsに
		 * Validatorインスタンスを追加することでアクションの処理フロー中に
		 * バリデーションが掛かるようになります。<br>
		 * 
		 * <p>[備 考]</p>
		 * Flex、AIRのValidatorはsourceプロパティとしてコンポーネントが追加されると
		 * それ以降ValueCommitイベントで逐次バリデーションを行いますが、<br>
		 * このメソッドで登録したValidatorはアクションが稼動した契機でのみ、
		 * バリデーションが行われます。
		 * <p>
		 * 
		 * @example
		 * TextInputに対して桁数チェックを行うValidatorを登録
		 * <listing version="3.0">
            override protected function validators(validators:Array):void {
                var rangeValidator:StringValidator = new StringValidator();
                rangeValidator.source = getComponentById("fax1");
                rangeValidator.property = "text";
                rangeValidator.maxLength = 4;
                rangeValidator.minLength = 3;
                rangeValidator.tooLongError = "FAX番号は3～4桁の数字で入力してください。";
                rangeValidator.tooShortError = "FAX番号は3～4桁の数字で入力してください。";
                rangeValidator.required = true;
                rangeValidator.requiredFieldError = "FAX番号は必須です。";
                rangeValidator.enabled = true;
                validators.push(rangeValidator);
            }
		 * </listing>
		 * 
		 * @param validators mx.validators.Validator継承オブジェクトを格納するリスト
		 */
		protected function validators(validators:Array):void{
		}

		/**
		 * <p>[概 要]</p>
		 * validatorsで指定したValidatorがエラー検出した場合、テンプレートコールされます.
		 * 
		 * <p>[詳 細]</p>
		 * 通常のバリデータのエラー処理の他、独自のエラー通知表現を実装する場合にオーバーライドします。<br>
		 * validateメソッドによるValidateErrors返却後、BaseAction#run(ParameterMapping)は
		 * BaseAction#validationFault(ValidateErrors)メソッドをテンプレートコールします。<br>
		 * ValidateErrorをハンドリングする場合は、validationFaultメソッドをオーバーライドして下さい。
		 * <p/>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @example
		 * <listing version="3.0">
            override public function validationFault(errors:ValidateErrors):void {
                var error:ValidateError = errors.getError(0);
                Label(getComponentById("faxError")).text = error.message;
            } 
		 * </listing>
		 * 
		 * @param faults validatorsで予約されたValidator配列が出したエラーイベントが入ったリスト
		 */
		public function validationFault(faults:ValidateErrors):void {
		}
					
		/**
		 * <p>[概 要] </p>
		 * Action処理終了後、コントローラに実行させるモデルクラス群の予約を行います.
		 * 
		 * <p>[詳 細] </p>
		 * BaseModel基底クラスを継承して自作モデルクラスを作成し、登録することが出来ますが、
		 * HTTPリクエスト送信、Pub/Sub実装といった使用頻度の高い機能モデルについては
		 * Reffi F/Wでも提供しています。詳しくはmodelパッケージを参照して下さい。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * 以下のように実行するモデルクラスを指定して下さい。
		 * <listing version="3.0">
             package demo.client.action.login {
          
                 import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
                 import demo.client.model.login.LoginLogic;
          
                 public class LoginAction extends BaseAction {
                     override protected function reserveModels(models:Array):void{
                         ma.push(Class(LoginLogic)); 
                     }
                 }
             }
		 * </listing>
		 * @param models 実行するモデルクラスを格納するリストオブジェクト
		 */
		protected function reserveModels(models:Array):void {
		}
		
		/**
		 * <p>[概 要] </p>
		 * 予約モデルがコントローラによって実行される直前にコールバックされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * reserveModelsで登録したモデル数＋１回、コントローラによって呼び出されます。<br>
		 * (1モデル登録の場合でも、実行前、実行後と2回呼び出されます。)<br>
		 * 引数prevには前回モデルの処理結果（初回null）が、<br>
		 * 引数nextには次回モデルインスタンス（最終回null）が渡されます。<br>
		 * 前回モデルインスタンスを取得する場合はprev.targetで取得することが出来ます。
		 * </p>
		 * 
		 * BaseAction#isRunModelsAndNoWait()がfalseの場合、次回モデルの実行が
		 * 前回モデルの終了後であることが確約される為、prevの結果を元にnextの動作設定をすることが出来ます。<p>
		 * 
		 * <p>[備 考] </p>
		 * BaseAction#isRunModelsAndNoWait()がtrueの場合、コントローラはモデルの実行を
		 * シーケンシャルには行いません。<br>
		 * この場合、引数prevにはnullが渡されます。
		 * <p>
		 * 
		 * @param index 実行インデックス（0 ～ reserveModelsによるモデル登録数）
		 * @param prev 前インデックスで実行されたモデル処理結果イベント
		 * @param next 次インデックスで実行される予定のモデルインスタンス
		 * @return 次のモデルを実行するかどうかのフラグ
		 */
		public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean{
			return true;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 登録した機能モデルの正常終了処理を記述するメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * reserveModelsによって予約された１モデル処理成功時にコントローラにコールバックされます。<br>
		 * 機能モデルがModelProcessEvent.SUCCESSイベントを発行し、コントローラにハンドリング
		 * された時点でコールバックされます。
		 * </p>
		 * 
		 * 実行された機能モデルの種類によって第三引数resultEventの型が変化します。<br>
		 * （HTTPServiceCoreの場合はResultEvent、ConsumerCoreの場合はMessageEvent等）<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param index 結果を返却したモデルの実行インデックス
		 * @param model 結果を返却したモデルインスタンス
		 * @param resultEvent モデル処理結果オブジェクト
		 */
		public function successForward(index:int, model:BaseModel, resultEvent:Event) : void {
		}
		
		/**
		 * <p>[概 要]</p>
		 * 登録した機能モデルの異常終了処理を記述するメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * reserveModelsによって予約されたどれか一つのモデルが失敗した場合にコールバックされます。<br>
		 * 機能モデルがModelProcessEvent.FAILUREイベントを発行し、コントローラにハンドリング
		 * された時点でコールバックされます。
		 *  
		 * <p>[備 考]</p>
		 * 
		 * @param index 結果を返却したモデルの実行インデックス
		 * @param model 結果を返却したモデルインスタンス
		 * @param faultEvent モデル異常終了時のイベント
		 * @return 引数faultEvent、若しくはオーバーライドメソッドで変換されたイベント
		 */
		public function failureForward(index:int, model:BaseModel, faultEvent:Event):Event{
			return faultEvent;
		}

		/**
		 * <p>[概 要]</p>
		 * reserveModelsで登録されたモデル群が全て終了した契機でコールされます.
		 * 
		 * <p>[詳 細]</p>
		 * reserveModelsで登録された全モデルがModelProcessEvent.FINISHEDイベントを発行
		 * した契機でコントローラによってコールバックされます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function complete():void {
		}


		/**
		 * <p>[概 要]</p>
		 * Action内で発生した同期エラーをハンドリングします.
		 * 
		 * <p>[詳 細]</p>
		 * デフォルト処理は有りません。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param e アクション内で発生した例外
		 * @return 引数e、若しくはオーバーライド先で生成した例外
		 */
		protected function trap(e:Error):Error{
			return e;
		}
		
		/**
		 * <p>[概 要]</p>
		 * reserveModelsによって登録された全モデルが終了するまでモーダル状態にするかどうかのフラグを返却します.
		 * 
		 * <p>[詳 細]</p>
		 * このメソッドをオーバーライドしてtrueを返却することで、登録された
		 * 全ての機能モデルがModelProcessEvent.FINISHEDイベントを発行するまで、
		 * プログレスバー付きのモーダルダイアログが表示されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return reserveModelsで予約された全モデルが実行終了するまで画面をモーダルにするかどうかのフラグ
		 */
		protected function isWaitModelsDone():Boolean{
			return false;
		}
		
		/**
		 * <p>[概 要]</p>
		 * モデル終了までのモーダル状態時にプログレスバーに表示するメッセージを取得します。
		 * 
		 * <p>[詳 細]</p>
		 * このメソッドをオーバーライドしなかった場合はプログレスバーのデフォルトメッセージが
		 * 表示されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return プログレスバーに表示するメッセージ
		 */
		protected function waitModelsMessage():String {
			return null;
		}
		
		/**
		 * <p>[概 要] </p>
		 * モデル群実行を非シーケンシャルに行うかどうかを設定するメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルトではfalseを返却します。<br>
		 * BaseAction#reserveModels(Arrray)によって予約されたモデル群を
		 * コントローラが実行する時、1モデルの結果取得を待ってから次モデルを
		 * 実行するかどうかの設定値として解釈されます。<br>
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @return true : モデルの処理結果取得を待たずに次モデル実行
		 */
		protected function isRunModelsAndNoWait():Boolean {
			return false;
		}
	}
}
