/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.messaging {
	
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.manager.ConsumerCoreManager;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	import mx.messaging.ChannelSet;
	import mx.messaging.Consumer;
	import mx.messaging.events.MessageEvent;
	import mx.messaging.events.MessageFaultEvent;
	
	/**
	 * <p>[概 要] </p>
	 * 発行されているサーバメッセージデスティネーションをクライアントで購読する為の機能モデルクラスです.
	 * 
	 * <p>[詳 細] </p>
	 * サーバ側（LCDS及びBlazeDS）のmessaging-config.xmlで定義されているdestinationを購読開始、停止します。<br>
	 * subscribeプロパティがtrueの場合は購読開始、falseの場合は停止します。<p>
	 * 
	 * この機能モデルを拡張したモデルを作成する場合、onSuccessメソッドとonFailureメソッドをオーバーライド
	 * することで、メッセージ受信成功処理、失敗処理を実装することが出来ます。<br>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @example 
	 * 指定回数subscribeしたらunsubscribeする
	 * <listing version="3.0">
        public class SubscribeStartAction extends BaseAction {

            // ① subscribeを行うConsumerCore機能モデルを登録します    
            override protected function reserveModels(models:Array):void {
                models.push(Class(ConsumerCore));
            }
        
            // ② ConsumerCoreインスタンスの設定を行います。このケースではpublishDemoデスティネーションを購読します 
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch (index) {
                    case 0:
                        ConsumerCore(next).destination = "publishDemo";
                        break;                
                }
                return true;
            }
        
            // ③ メッセージ購読成功時の処理を実装します
            //    ConsumerCoreは継続的にSUCCESSイベントを返却する為、subscribe停止するまで
            //    このメソッドへのコールバックが継続します
            //    ConsumerCoreManagerを使用することで任意のタイミングでsubscribe停止可能です
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                var list:ArrayCollection = getObjectById("serverPushArrayCollection") as ArrayCollection;
                list.list = MessageEvent(resultEvent).message.body as ArrayCollection;
                 
                if(model.successCount == 3) {
                    ConsumerCoreManager.getInstance().unsubscribe("publishDemo");
                }
            }
            
            // ④ 登録モデルが完全に完了した時に呼ばれるメソッドを実装します
            //    継続的に値を返却するモデルの完了タイミングをハンドルします 
            override public function complete():void {
                trace("subscribe is done.");
            }
        }
	 * </listing>
	 * 
	 * 既に実行中のdestination購読を停止する
	 * <listing version="3.0">
        public class SubscribeStopAction extends BaseAction {

            // ① subscribe停止を行うConsumerCore機能モデルを登録します    
            override protected function reserveModels(models:Array):void {
                models.push(Class(ConsumerCore));
            }
            
            // ② subscribeモードをfalseに設定したConsumerCoreが実行されることでsubscribe停止します
            //    同一destinationをsubscribeしているConsumerCoreが複数存在するケースで、任意のConsumerCore
            //    だけsubscribeを停止する場合、開始時に設定しておいたidentifierを指定することが出来ます
            //    identifierが指定されなかった場合は同一destinationをsubscribe中の全ConsumerCoreがsubscribe停止します    
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch (index) {
                    case 0:                
                        ConsumerCore(next).destination = "publishDemo";
                        ConsumerCore(next).subscribe = false;
                        ConsumerCore(next).identifier = "client1";
                        break;                    
                }
                return true;
            }
        }
	 * </listing>
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * 
	 * @see jp.co.fujitsu.reffi.client.flex.manager.ConsumerCoreManager
	 * @author Project Reffi 
	 */
	public class ConsumerCore extends BaseModel {
		
		// メッセージ受信先
		private var _destination:String;
		
		// 購読の開始／終了フラグ
		private var _subscribe:Boolean = true;
		
		// トピックを受信中のSubscribeCoreを識別する為の識別子
		private var _identifier:String;
		
		// 接続チャンネルセット
		private var _channelSet:ChannelSet;
		
		
		/**
		 * <p>[概 要]</p>
		 * メッセージ受信を行う対象のdestinationです.
		 * 
		 * <p>[詳 細]</p>
		 * LCDS及びBlazeDSのmessaging-config.xmlで定義されているdestinationを設定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get destination():String{
			return this._destination;
		}
		public function set destination(destination:String):void{
			this._destination = destination;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 購読開始/購読停止を示すフラグです.
		 * 
		 * <p>[詳 細]</p>
		 * true:購読開始<br>
		 * false:購読停止<br>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default true
		 */
		public function get subscribe():Boolean{
			return this._subscribe;
		}
		public function set subscribe(subscribe:Boolean):void{
			this._subscribe = subscribe;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 購読中のConsumerCoreを特定する為の識別子です.
		 * 
		 * <p>[詳 細]</p>
		 * 同一destinationを購読するConsumerCoreが複数有る場合、
		 * 任意のConsumerCoreを特定する為に使用するプロパティです。<br>
		 * subscribe開始時に指定することが出来ます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @example
		 * 同一チャンネルに入っているチャットクライアントの内、任意のクライアントを終了する<p>
		 * 
		 * 開始
		 * <listing version="3.0">
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch (index) {
                    case 0:
                        ConsumerCore(next).destination = "chatRoom";
                        ConsumerCore(next).identifier = "yamada";
                        break;                
                }
                return true;
            }
		 * </listing>
		 * 
		 * 停止
		 * <listing version="3.0">
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                switch (index) {
                    case 0:
                        ConsumerCore(next).destination = "chatRoom";
                        ConsumerCore(next).subscribe = false;
                        ConsumerCore(next).identifier = "yamada";
                        break;                
                }
                return true;
            }
		 * </listing>
		 */
		public function get identifier():String {
			return this._identifier;
		}
		public function set identifier(identifier:String):void {
			this._identifier = identifier;
		}
				
		/**
		 * <p>[概 要]</p>
		 * 任意設定可能な通信チャンネルセットです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get channelSet():ChannelSet {
			return this._channelSet;
		}
		public function set channelSet(channelSet:ChannelSet):void {
			this._channelSet = channelSet;
		}

		/**
		 * <p>[概 要] </p>
		 * サーバに対して購読の開始、停止処理を行います.
		 * 
		 * <p>[詳 細] </p>
		 * subscribe管理オブジェクト（ConsumerCoreManager）を取得し、購読の開始、中止を委譲します。<br>
		 * 対象となるdestination名は「destination」プロパティから、購読開始、中止の判断を行うフラグは
		 * 「subscribe」プロパティから取得します。<p>
		 * 
		 * 購読中止（subscribeプロパティ == false）時、ConsumerCoreManager#unsubscribeメソッド処理が
		 * 例外をスローしなかった場合は、その時点でModel処理成功イベントを発火します。<br>
		 * 
		 * <p>[備 考] </p>
		 * 購読中止時のModel処理成功イベント内結果オブジェクトはnullです。
		 * 利用することは出来ません。
		 * <p/>
		 * 
		 */
		override protected function mainProc() : void {
			super.mainProc();
			
			var destination:String = this.destination;
			var subscribe:Boolean = this.subscribe;
			var channelSet:ChannelSet = this.channelSet;
			if (subscribe) {
				ConsumerCoreManager.getInstance().subscribe(destination, this, channelSet);
			} else {
				ConsumerCoreManager.getInstance().unsubscribe(destination, this.identifier);

				// モデル成功イベントを生成			
				var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
				dispatchModelSuccess(modelSuccessEvent);

				// モデル完了イベントを生成			
				var modelFinishedEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
				dispatchModelFinished(modelFinishedEvent);
			}
		}
		
		/**
		 * <p>[概 要] </p>
		 * subscribe受信時にコールバックされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * ConsumerCoreManagerに管理されているConsumerのMessageEvent.MESSAGEイベントハンドラ
		 * からコールバックされます。<br>
		 * subscribe受信データ取得用オーバーライドメソッドであるonSuccessを
		 * テンプレートコールします。<br>
		 * onSuccessメソッドが正常終了した後、Model処理成功イベントを発火します。<br>
		 * 
		 * <p>[備 考] </p>
		 * このメソッドはオーバーライド出来ません。<br>
		 * subscribe受信データをモデルレイヤで取得したい場合は、
		 * onSuccessメソッドをオーバーライドして下さい。
		 * <p/>
		 * 
		 * @param event メッセージ受信イベント
		 */
		public final function resultHandler(event:MessageEvent):void{
			onSuccess(event);

			// モデル成功イベントを生成			
			var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
			modelSuccessEvent.cause = event;
			dispatchModelSuccess(modelSuccessEvent);
		}

		/**
		 * <p>[概 要]</p>
		 * subscribe結果受信オーバーライド用メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event メッセージ受信イベント
		 */
		protected function onSuccess(event:MessageEvent):void{
		}

		/**
		 * <p>[概 要]</p>
		 * subscribe受信失敗時にコールバックされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * ConsumerCoreManagerに管理されているConsumerのMessageFaultEvent.FAULTイベントハンドラ
		 * からコールバックされます。<br>
		 * subscribe受信失敗処理用オーバーライドメソッドであるonFailureを
		 * テンプレートコールします。<br>
		 * onFailureメソッドが正常終了した後、Model処理失敗イベントを発火します。<br>
		 * 
		 * <p>[備 考] </p>
		 * このメソッドはオーバーライド出来ません。<br/>
		 * メッセージ受信失敗処理を記述する場合はonFailureメソッドをオーバーライドして下さい。
		 * <p/>
		 * 
		 * @param event メッセージ受信失敗イベント
		 */
		public function faultHandler(event:MessageFaultEvent):void{
			onFailure(event);
			
			// モデル終了イベントを生成
			var modelFailureEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
			modelFailureEvent.cause = event;
			dispatchModelFailure(modelFailureEvent);
		}
		
		/**
		 * <p>[概 要]</p>
		 * subscribe受信失敗オーバーライド用メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event メッセージ受信失敗イベント
		 */
		protected function onFailure(event:MessageFaultEvent):void{
		}
	}
}
