/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.parser;

import java.util.HashMap;
import java.util.Map;

import jp.co.fujitsu.reffi.common.exception.CoreLogicException;

import com.nexaweb.xml.Element;

/**
 * <p>[概 要] </p>
 * 各種エレメントが持つ値を包括的に取得する為のパーサです。
 * 
 * <p>[詳 細] </p>
 * 各種エレメントは値の保持の仕方が異なります。<br>
 * （textFieldエレメントはtext属性値を値として保持し、listBoxエレメントは
 * selected属性がついているlistItem子エレメントのtext属性値を値として保持する等）<p>
 * 
 * このクラスを使用してパースすることで、統一的な値へのアクセスと、
 * 共通のオブジェクトによる値の保持が行われます。
 * 
 * <p>[備 考] </p>
 * 
 * このクラスのperseメソッドはElementの種類を意識せず、
 * 単一の方法で値を取る為の手法を提供します。
 * <P>
 * 
 * <B>NexawebエレメントとReffiが提供するValuePaserの対応表</B>
 * 	<TABLE border="1">
 * 		<TR>
 * 			<TD>エレメント種</TD>
 * 			<TD>対応する値取得パーサクラス</TD>
 * 			<TD>値取得の対象</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>button</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.ButtonValueParser</TD>
 * 			<TD>text属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>label</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.LabelValueParser</TD>
 * 			<TD>text属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>passwordField</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.PasswordFieldValueParser</TD>
 * 			<TD>text属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>textField</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.TextFieldValueParser</TD>
 * 			<TD>text属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>checkBox</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.CheckBoxValueParser</TD>
 * 			<TD>selected="true"属性値があれば"true"、なければ"false"</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>radioButton</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.RadioButtonValueParser</TD>
 * 			<TD>selected="true"属性値があれば"true"、なければ"false"</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>comboBox</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.ComboBoxValueParser</TD>
 * 			<TD>selected="true"属性値がついたlistBox/listItemエレメントのvalue属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>listBox</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.ListBoxValueParser</TD>
 * 			<TD>selected="true"属性値がついたlistItemエレメントのvalue属性値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>horizontalSlider</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.HorizontalSliderValueParser</TD>
 * 			<TD>position属性値、なければ"0"</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>verticalSlider</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.VerticalSliderValueParser</TD>
 * 			<TD>position属性値、なければ"0"</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>textArea</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.TextAreaValueParser</TD>
 * 			<TD>テキストノード値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>textView</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.TextViewValueParser</TD>
 * 			<TD>テキストノード値</TD>
 * 		</TR>
 * 		<TR>
 * 			<TD>table</TD>
 * 			<TD>jp.co.fujitsu.reffi.client.nexaweb.parser.TableValueParser</TD>
 * 			<TD>row/cellエレメントのtext属性値</TD>
 * 		</TR>
 * 	</TABLE>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ElementValueParser implements Parser{

	/** エレメント種類毎に用意されたパーサ群をマッピングします。 */
	public static final Map<String, Class<? extends Parser>> PARSERS;

	static{
		// PARSERS static領域を初期化。フレームワークが提供するValuePaser群を登録。
		PARSERS = new HashMap<String, Class<? extends Parser>>();
		PARSERS.put("button", ButtonValueParser.class);
		PARSERS.put("label", LabelValueParser.class);
		PARSERS.put("passwordField", PasswordFieldValueParser.class);		
		PARSERS.put("textField", TextFieldValueParser.class);
		PARSERS.put("checkBox", CheckBoxValueParser.class);
		PARSERS.put("radioButton", RadioButtonValueParser.class);
		PARSERS.put("comboBox", ComboBoxValueParser.class);
		PARSERS.put("listBox", ListBoxValueParser.class);
		PARSERS.put("horizontalSlider", HorizontalSliderValueParser.class);
		PARSERS.put("verticalSlider", VerticalSliderValueParser.class);
		PARSERS.put("textArea", TextAreaValueParser.class);
		PARSERS.put("textView", TextViewValueParser.class);
		PARSERS.put("table", TableValueParser.class);
	}
	
	/**
	 * <p>[概 要] </p>
	 * Nexaweb提供エレメントの値解析を行うメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数elementのローカル名を元にPARSERSからParserクラスを取り出し
	 * インスタンス化、parseメソッドを引数elementで呼び出します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param element 解析対象エレメント
	 */
	public ElementValues parse(Element element) throws Exception{
		ElementValues ret = null;
		try{
			String localName = element.getLocalName();
			Parser parser = null;
			if(PARSERS.containsKey(localName)){
				parser = (Parser)((Class<? extends Parser>)PARSERS.get(localName)).newInstance();
			}else{
				return null;
			}

			ret = (ElementValues)parser.parse(element);
		}catch(IllegalAccessException e){
			throw new CoreLogicException("EFC0011");
		}catch(InstantiationException e){
			throw new CoreLogicException("EFC0012");
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 任意のParserを追加します。
	 * 
	 * <p>[詳 細] </p>
	 * 各エレメント名に対応する、値パーサマップに新規、上書き追加します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param elementName 解析対象エレメント名
	 * @param parser 解析オブジェクト
	 */
	public static void addParser(String elementName, Class<? extends Parser> parser) {
		PARSERS.put(elementName, parser);
	}
}
