/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.listener;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.fujitsu.reffi.client.nexaweb.model.SubscribeCore;
import jp.co.fujitsu.reffi.common.exception.CoreLogicException;

import com.nexaweb.client.netservice.NetServiceException;
import com.nexaweb.messaging.MessageListener;
import com.nexaweb.util.XmlSerializable;

/**
 * <p>[概 要]</p>
 * 購読管理クラス
 * 
 * <p>[詳 細]</p>
 * トピックをキーとして、複数のアクションに対して購読予約したメッセージを反映させます。
 * 
 * <p>[備 考]</p>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public enum SubscribeManager {
	/** インスタンス */
	instance;
	/** 購読予約管理MAP */
	private Map<String, List<SubscribeCore>> _subscriberMap = new HashMap<String, List<SubscribeCore>>();

	/**
	 * <p>[概 要]</p>
	 * 現在購読予約中のトピック名リストを返却します。
	 * 
	 * <p>[詳 細]</p>
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @return 登録されているトピック名一覧
	 */
	public List<String> getSubscribingTopicNames() {
		List<String> topicNames = new ArrayList<String>();
		Set<String> topics = this._subscriberMap.keySet();
		for(Iterator<String> it = topics.iterator(); it.hasNext(); ){
			topicNames.add(it.next());
		}
		return topicNames;
	}

	/**
	 * <p>[概 要]</p>
	 * 購読予約判定
	 * 
	 * <p>[詳 細]</p>
	 * 指定したトピック名で既に購読予約されているか判定します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic トピック名
	 * @return 登録されている場合はtrue、それ以外はfalse
	 */
	public boolean isSubscribing(String topic) {
		return getSubscribingTopicNames().contains(topic);
	}
	
	/**
	 * <p>[概 要]</p>
	 * 購読予約
	 * 
	 * <p>[詳 細]</p>
	 * トピックでの購読予約を行います。
	 * 指定したトピックで既にリスナーが起動している場合は新たなリスナーの起動は行いません。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic トピック名
	 * @param SubscribeCore 購読モデル
	 * @throws CoreLogicException 購読開始例外
	 */
	public void subscribe(String topic, SubscribeCore subscribeCore) throws CoreLogicException {
		List<SubscribeCore> subscribeCoreList = null;
		
		if (!this._subscriberMap.containsKey(topic)) {
			subscribeCoreList = new ArrayList<SubscribeCore>();
			startListening(topic, subscribeCore);
		} else {
			subscribeCoreList = this._subscriberMap.get(topic);
		}
		subscribeCoreList.add(subscribeCore);
		this._subscriberMap.put(topic, subscribeCoreList);
	}

	/**
	 * <p>[概 要]</p>
	 * 購読停止
	 * 
	 * <p>[詳 細]</p>
	 * 指定したトピックでの購読停止を行います。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic トピック名
	 * @throws CoreLogicException 購読停止例外
	 */
	public void unsubscribe(String topic) throws CoreLogicException {
		unsubscribe(topic, null);
	}
	
	/**
	 * <p>[概 要] 
	 * 購読停止
	 * 
	 * <p>[詳 細] 
	 * 指定したトピックでの購読停止を行います。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic トピック名
	 * @param identifier モデル識別子
	 * @throws CoreLogicException 購読停止例外
	 */
	public void unsubscribe(String topic, String identifier) throws CoreLogicException {
		if (!this._subscriberMap.containsKey(topic)) {
			return;
		}
		try {
			if(identifier == null) {
				SubscribeCore subscribeCore = (SubscribeCore)this._subscriberMap.get(topic).get(0);
				subscribeCore.getController().getSession().getMessagingService().unsubscribe(topic);
				subscribeCore.unsubscribeCompleted();
				this._subscriberMap.remove(topic);
			}else{
				List<SubscribeCore> subscribeCoreList = this._subscriberMap.get(topic);
				for (Iterator<SubscribeCore> itr = subscribeCoreList.iterator(); itr.hasNext();) {
					SubscribeCore subscribeCore = itr.next();
					if(identifier.equals(subscribeCore.getIdentifier())) {
						subscribeCore.unsubscribeCompleted();
						itr.remove();
					}
				}
			}
		} catch (NetServiceException e) {
			throw new CoreLogicException("EFC1005", e);
		}
	}

	/**
	 * <p>[概 要]</p>
	 * 購読予約開始
	 * 
	 * <p>[詳 細]</p>
	 * 指定したトピックにて購読予約を開始します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic　トピック
	 * @param subscribeCore 購読モデル
	 * @throws CoreLogicException　購読開始例外
	 */
	private void startListening(String topic, SubscribeCore subscribeCore)
			throws CoreLogicException {
		try {
			// リスナー管理オブジェクトと連携が必要
			MessageListener messageListener = new MessageListener() {
				public void onMessage(String topic, Serializable obj) {
					subscribeMessage(topic, obj);
				}

				public void onMessage(String topic, XmlSerializable obj) {
					subscribeMessage(topic, obj);
				}

				public void onMessage(String topic, String string) {
					subscribeMessage(topic, string);
				}
			};
			subscribeCore.getController().getSession().getMessagingService().subscribe(topic, messageListener);
		} catch (NetServiceException e) {
			throw new CoreLogicException("EFC1005", e);
		}
	}

	/**
	 * <p>[概 要]</p>
	 * 購読予約したメッセージを受信するハンドラメソッドです。
	 * 
	 * <p>[詳 細]</p>
	 * トピックにて対象となる購読モデルを取得し、取得した購読モデルのsubscribeCompletedをコールします。 
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param topic トピック名
	 * @param result 受信したデータ
	 */
	protected void subscribeMessage(String topic, Object result) {
		List<SubscribeCore> subscribeCoreList = this._subscriberMap.get(topic);
		for (Iterator<SubscribeCore> itr = subscribeCoreList.iterator(); itr.hasNext();) {
			itr.next().subscribeCompleted(result);
		}
	}
}
