/*
 * Copyright (c) 2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.swing.util;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.Window;

import jp.co.fujitsu.reffi.client.swing.listener.MessageDialogListener;

/**
 * <p>[概 要] </p>
 * メッセージダイアログを表示するユーティリティクラスです。
 * 
 * <p>[詳 細] </p>
 * JOptionPaneが保持する機能の他、モードレスでのダイアログ表示をサポートします。
 * 
 * 
 * <p>[備 考] </p>
 * 
 * <b>使用例）</b><br>
 * <pre>
 * 	MessageDialogUtil.showModelessMessageDialog(c, 
 * 		"アカウントが有りません", 
 * 		"エラー", 
 * 		MessageDialogUtil.ERROR_MESSAGE, 
 * 		new MessageDialogListener()
 *	{
 * 		&#064;Override
 *		public void dialogClosed(MessageDialogEvent e) {
 *			System.out.println("モードレスダイアログの結果 = " + e.getReturnValue());
 *		}
 *	});
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class MessageDialogUtil {
	
	/** 「はい」が押されたことを示す定数です。 */
	public static final int YES_OPTION = 0;

	/** 「いいえ」が押されたことを示す定数です。 */
	public static final int NO_OPTION = 1;

	/** 「キャンセル」が押されたことを示す定数です。 */
	public static final int CANCEL_OPTION = 2;
	
	/** 「OK」が押されたことを示す定数です。 */
	public static final int OK_OPTION = 0;
	
	/** 「×」が押されたことを示す定数です。 */
	public static final int CLOSED_OPTION = -1;

	
	/** エラーメッセージダイアログであることを示す定数です。 */
	public static final int ERROR_MESSAGE = 0;

	/** 情報メッセージダイアログであることを示す定数です。 */
	public static final int INFORMATION_MESSAGE = 1;

	/** 警告メッセージダイアログであることを示す定数です。 */
	public static final int WARNING_MESSAGE = 2;

	/** 質問メッセージダイアログであることを示す定数です。 */
	public static final int QUESTION_MESSAGE = 3;

	/** 通常メッセージダイアログであることを示す定数です。 */
	public static final int PLAIN_MESSAGE = -1;

	
	/** 通常ボタン構成であることを示す定数です。 */
	public static final int DEFAULT_OPTION = -1;

	/** 「はい」、「いいえ」ボタン構成であることを示す定数です。 */
	public static final int YES_NO_OPTION = 0;

	/** 「はい」、「いいえ」、「キャンセル」ボタン構成であることを示す定数です。 */
	public static final int YES_NO_CANCEL_OPTION = 1;

	/** 「OK」、「キャンセル」ボタン構成であることを示す定数です。 */
	public static final int OK_CANCEL_OPTION = 2;
	
	
	/**
	 * <p>[概 要] </p>
	 * メッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * messageType : INFORMATION_MESSAGE
	 * buttonOption : DEFAULT_OPTION
	 * で
	 * {@link #showMessageDialog(Component, String, String, int, int)}
	 * メソッドをコールします。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parent 表示基底になる親コンポーネント
	 * @param message 表示するメッセージ
	 * @param title ダイアログタイトル
	 * @return 押下されたボタンの識別子
	 */
    public static int showMessageDialog(Component parent, String message, String title) {
    	return showMessageDialog(parent, message, title, INFORMATION_MESSAGE, DEFAULT_OPTION);
    }

    /**
	 * <p>[概 要] </p>
	 * メッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
     * 
	 * @param parent 表示基底になる親コンポーネント
	 * @param message 表示するメッセージ
	 * @param title ダイアログタイトル
     * @param messageType
     * @return 押下されたボタンの識別子
     */
	public static int showMessageDialog(Component parent, String message, String title, int messageType) {
    	return showMessageDialog(parent, message, title, messageType, DEFAULT_OPTION);
    }

	/**
	 * <p>[概 要] </p>
	 * メッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parent 表示基底になる親コンポーネント
	 * @param message 表示するメッセージ
	 * @param title ダイアログタイトル
	 * @param messageType
	 * @param buttonOption
	 * @return 押下されたボタンの識別子
	 */
	public static int showMessageDialog(Component parent, 
											String message, 
											String title, 
											int messageType, 
											int buttonOption) 
    {
		MessageDialog dialog = 
			createDialog((Window)parent, message, title, messageType, buttonOption);
		return dialog.showDialog();
	}
	
	/**
	 * <p>[概 要] </p>
	 * メッセージダイアログを生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param owner
	 * @param message
	 * @param title
	 * @param messageType
	 * @param buttonOption
	 * @return 生成されたモーダルメッセージダイアログ
	 */
	protected static MessageDialog createDialog(Window owner, 
											String message, 
											String title, 
											int messageType, 
											int buttonOption) 
	{
		MessageDialog dialog = 
			new MessageDialog(owner,
					Dialog.ModalityType.APPLICATION_MODAL,
					message,
					title,
					messageType,
					buttonOption);

		return dialog;
	}

	/**
	 * <p>[概 要] </p>
	 * モードレスメッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parent
	 * @param message
	 * @param title
	 * @param listener
	 */
    public static void showModelessMessageDialog(Component parent, 
    											String message, 
    											String title, 
    											MessageDialogListener listener) 
    {
    	showModelessMessageDialog(parent, message, title, INFORMATION_MESSAGE, DEFAULT_OPTION, listener);
    }

    /**
	 * <p>[概 要] </p>
	 * モードレスメッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
     * 
     * @param parent
     * @param message
     * @param title
     * @param messageType
     * @param listener
     */
    public static void showModelessMessageDialog(Component parent, 
    											String message, 
    											String title, 
    											int messageType, 
    											MessageDialogListener listener) 
    {
    	showModelessMessageDialog(parent, message, title, messageType, DEFAULT_OPTION, listener);
    }

    /**
	 * <p>[概 要] </p>
	 * モードレスメッセージダイアログを表示します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
     * 
     * @param parent
     * @param message
     * @param title
     * @param messageType
     * @param buttonOption
     * @param listener
     */
	public static void showModelessMessageDialog(Component parent, 
											String message, 
											String title, 
											int messageType, 
											int buttonOption,
											MessageDialogListener listener)
	{
		MessageDialog dialog = 
			createModelessDialog((Window)parent, message, title, messageType, buttonOption, listener);
		dialog.showDialog();
	}

	/**
	 * <p>[概 要] </p>
	 * モードレスメッセージダイアログを生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param owner
	 * @param message
	 * @param title
	 * @param messageType
	 * @param buttonOption
	 * @param listener
	 * @return 生成されたモードレスダイアログ
	 */
	protected static MessageDialog createModelessDialog(Window owner, 
											String message, 
											String title, 
											int messageType, 
											int buttonOption,
											MessageDialogListener listener) 
	{
		MessageDialog dialog = 
			new MessageDialog(owner,
					Dialog.ModalityType.MODELESS,
					message,
					title,
					messageType,
					buttonOption);
		
		dialog.addMessageDialogListener(listener);

		return dialog;
	}
}
