/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.util;

import java.beans.XMLEncoder;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import jp.co.fujitsu.reffi.common.exception.CoreLogicException;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.configuration.MutableConfiguration;
import org.apache.batik.transcoder.Transcoder;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.commons.io.IOUtils;
import org.apache.fop.apps.FOUserAgent;
import org.apache.fop.apps.Fop;
import org.apache.fop.apps.FopFactory;
import org.apache.fop.apps.MimeConstants;
import org.apache.fop.svg.PDFTranscoder;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * <p>[概 要] </p>
 * Apache FOPを使用するためのユーティリティクラスです。
 * <p>[詳 細] </p>
 * Apache FOPを使用するためメソッドを提供します。
 * <p>[備 考] </p>
 * 
 * @author Project Reffi 
 */
public enum FopUtil {
	/** インスタンス */
	instance;

	/**
	 * <p>[概 要] </p>
	 * XSL-FOファイルより、PDFを作成します。(XSL-FO -> PDF)
	 * <p>[詳 細] </p>
	 * XSL-FOファイルより、PDFを作成します。(XSL-FO -> PDF)</br>

	 * （呼び出しサンプル）
	 * <pre class="samplecode">
	 * public static void main(String[] args) {
	 *     try {
	 *         File baseDir = new File("bin");
	 *         File outDir = new File(baseDir, "out");
	 *         outDir.mkdirs();
	 * 
	 *         File fofile = new File(baseDir, "作成元FOファイル.fo");
	 *         File pdffile = new File(outDir, "出力PDFファイル.pdf");
	 * 
	 *         FopUtil.getInstance().convertFO2PDF(fofile, pdffile, "FOP設定ファイルパス");
	 *     } catch (Exception e) {
	 *         e.printStackTrace(System.err);
	 *     }
	 * }
	 * 
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param fo XSL-FOファイル
	 * @param pdf PDFファイル
	 * @param configPath 設定ファイルパス
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertFO2PDF(File fo, File pdf, String configPath) throws CoreLogicException {
		FileOutputStream out = null;
		try {

			// 設定ファイルの指定
			FopFactory fopFactory = FopFactory.newInstance();
			fopFactory.setUserConfig(convertConfiguration(configPath));

			FOUserAgent foUserAgent = fopFactory.newFOUserAgent();

			// 出力タイプをPDFに指定
			out = new FileOutputStream(pdf);
			Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, foUserAgent,
					out);

			// XSL-FOファイルをPDFで出力する
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
			Source src = new StreamSource(fo);
			Result res = new SAXResult(fop.getDefaultHandler());
			transformer.transform(src, res);
		} catch (FileNotFoundException e) {
			throw new CoreLogicException("EFC9002", e);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * XML + XSLTファイルより、XSL-FOファイルを作成します。(XML + XSLT -> XSL-FO)
	 * 
	 * <p>[詳 細] </p>
	 * XML + XSLTファイルより、XSL-FOファイルを作成します。(XML + XSLT -> XSL-FO)</br>
	 * 
	 * （呼び出しサンプル）
	 * <pre class="samplecode">
	 * 	public static void main(String[] args) {
	 * 		try {
	 * 			File baseDir = new File("bin");
	 * 			File outDir = new File(baseDir, "out");
	 * 			outDir.mkdirs();
	 * 
	 * 			File xmlfile = new File(baseDir, "作成元XMLファイル.xml");
	 * 			File xsltfile = new File(baseDir, "作成元XSLTファイル.xsl");
	 * 			File fofile = new File(outDir, "出力FOファイル.fo");
	 *	
	 *	 		FopUtil.getInstance().convertXML2FO(xmlfile, xsltfile, fofile);
	 * 		} catch (Exception e) {
	 *	 		e.printStackTrace(System.err);
	 * 		}
	 * 	}
	 * 
	 * </pre> 
	 *  
	 * <p>[備 考] </p>
	 * @param xml XMLファイル
	 * @param xslt XSLTファイル
	 * @param fo FOファイル
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertXML2FO(File xml, File xslt, File fo) throws CoreLogicException {
		FileOutputStream out = null;
		try {
			// 出力ファイルにFOファイルを指定
			out = new FileOutputStream(fo);
			// XML + XSLT からXSL-FOファイルを生成する
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer(new StreamSource(xslt));
			Source src = new StreamSource(xml);
			Result res = new StreamResult(out);
			transformer.transform(src, res);
		} catch (FileNotFoundException e) {
			throw new CoreLogicException("EFC9002", e);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * フォントファイル・フォントメトリクスファイルの配置先を修正・設定します。
	 * <p>[詳 細] </p>
	 * フォントファイル・フォントメトリクスファイルがDeploy先に配置される場合、設定ファイル中に固定のパスを明記できない問題を回避。</br>
	 * コンフィグファイルと同じディレクトリ内に置かれているものとします。
	 * <p>[備 考] </p>
	 * @param configPath コンフィグファイルの場所
	 * @return 読み込んだコンフィグファイルを修正したコンフィギュレーション
	 * @throws ConfigurationException 
	 * @throws SAXException
	 * @throws IOException
	 */
	private Configuration convertConfiguration(String configPath) throws ConfigurationException, SAXException, IOException {
		File configFile = new File(configPath);
		// コンフィグファイルと同じ場所にフォントファイル・フォントメトリクスファイルがあるものとします。
		String configRootPath = configFile.getParentFile().toURI().toString();

		MutableConfiguration userConfig = new DefaultConfiguration(new DefaultConfigurationBuilder().buildFromFile(configFile));
		MutableConfiguration[] renderers = userConfig.getMutableChild("renderers").getMutableChildren();
		for (int r = 0; r < renderers.length; r++) {
			MutableConfiguration[] fonts = renderers[r].getMutableChild("fonts").getMutableChildren("font");
			for (int i = 0; i < fonts.length; i++) {
				String metrics = configRootPath + "/" + fonts[i].getAttribute("metrics-url");
				fonts[i].setAttribute("metrics-url", metrics);
				String embed = configRootPath + "/" + fonts[i].getAttribute("embed-url");
				fonts[i].setAttribute("embed-url", embed);
			}
		}
		return userConfig;
	}

	/**
	 * <p>[概 要] </p>
	 * XML + XSLTファイルより、PDFを作成します。(XML -> XSL-FO -> PDF)
	 * 
	 * <p>[詳 細] </p>
	 * XML + XSLTファイルより、PDFを作成します。(XML -> XSL-FO -> PDF)</br>
	 * 
	 * （呼び出しサンプル）
	 * <pre class="samplecode">
	 * 	public static void main(String[] args) {
	 * 		try {
	 * 			File baseDir = new File("bin");
	 * 			File outDir = new File(baseDir, "out");
	 * 			outDir.mkdirs();
	 * 
	 * 			File xmlfile = new File(baseDir, "作成元XMLファイル.xml");
	 * 			File xsltfile = new File(baseDir, "作成元XSLTファイル.xsl");
	 * 			File pdffile = new File(outDir, "出力PDFファイル.pdf");
	 *	
	 *	 		FopUtil.getInstance().convertXML2PDF(xmlfile, xsltfile, pdffile, "FOP設定ファイルパス");
	 * 		} catch (Exception e) {
	 *	 		e.printStackTrace(System.err);
	 * 		}
	 * 	}
	 * 
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * @param xml XMLファイル
	 * @param xslt XSLTファイル
	 * @param pdf PDFファイル
	 * @param configPath 設定ファイルパス
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertXML2PDF(File xml, File xslt, File pdf, String configPath) throws CoreLogicException {
		FileOutputStream out = null;
		try {
			// 設定ファイルの指定
			FopFactory fopFactory = FopFactory.newInstance();
			FOUserAgent foUserAgent = fopFactory.newFOUserAgent();
			fopFactory.setUserConfig(convertConfiguration(configPath));

			// 出力タイプをPDFに指定
			out = new FileOutputStream(pdf);
			Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, foUserAgent, out);

			// XML + XSLT からPDFファイルを生成する
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer(new StreamSource(xslt));
			Source src = new StreamSource(xml);
			Result res = new SAXResult(fop.getDefaultHandler());
			transformer.transform(src, res);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * Objectより、XMLを作成します。(Object -> XML)
	 * 
	 * <p>[詳 細] </p>
	 * Objectより、DOMを作成します。(Object -> XML)</br>
	 * XMLEncoderを使用して、オブジェクトをXMLに出力します。
	 *  
	 * <p>[備 考] </p>
	 * @param obj Object
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertObj2XML(Object obj, File xml) throws CoreLogicException {
		FileOutputStream out = null;
		try {
			// Obj -> XML(メモリ上)
			out = new FileOutputStream(xml);
			XMLEncoder encoder = new XMLEncoder(new BufferedOutputStream(out));
			encoder.writeObject(obj);
			encoder.close();
		} catch (IOException e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * Objectより、PDFを作成します。(Object -> XML + XSLT -> PDF)
	 * 
	 * <p>[詳 細] </p>
	 * Objectより、PDFを作成します。(Object -> XML + XSLT -> PDF)</br>
	 * XMLEncoderを使用して、オブジェクトをPDFに出力します。
	 *  
	 * <p>[備 考] </p>
	 * @param obj Object
	 * @param xslt XSLファイル
	 * @param pdf PDFファイル
	 * @param configPath 設定ファイルパス
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertObj2PDF(Object obj, File xslt, File pdf, String configPath) throws CoreLogicException {
		ByteArrayOutputStream byteArrayOutputStream = null;
		FileOutputStream out = null;
		try {
			// Obj -> XML(メモリ上)
			byteArrayOutputStream = new ByteArrayOutputStream();
			XMLEncoder encoder = new XMLEncoder(new BufferedOutputStream(byteArrayOutputStream));
			encoder.writeObject(obj);
			encoder.close();

			// 設定ファイルの指定
			FopFactory fopFactory = FopFactory.newInstance();
			FOUserAgent foUserAgent = fopFactory.newFOUserAgent();
			fopFactory.setUserConfig(convertConfiguration(configPath));

			// 出力タイプをPDFに指定
			out = new FileOutputStream(pdf);
			Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, foUserAgent, out);

			// XML + XSLT からPDFファイルを生成する
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer(new StreamSource(xslt));
			Source src = new StreamSource(new ByteArrayInputStream(byteArrayOutputStream.toByteArray()));
			Result res = new SAXResult(fop.getDefaultHandler());
			transformer.transform(src, res);
		} catch (FileNotFoundException e) {
			throw new CoreLogicException("EFC9002", e);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(byteArrayOutputStream);
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * DOM（XSL-FOフォーマット）より、PDFを作成します。(DOM -> PDF)
	 * 
	 * <p>[詳 細] </p>
	 * DOM（XSL-FOフォーマット）より、PDFを作成します。(DOM -> PDF)</br>
	 *
	 * （呼び出しサンプル）
	 * <pre class="samplecode">
	 * 	private static Document buildDOMDocument() throws ParserConfigurationException {
	 * 		Document foDoc = null;
	 * 		Element root = null, ele1 = null, ele2 = null, ele3 = null;
	 * 
	 * 		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
	 * 		dbf.setNamespaceAware(true);
	 * 		DocumentBuilder db = dbf.newDocumentBuilder();
	 * 		foDoc = db.newDocument();
	 * 
	 * 		root = foDoc.createElementNS(foNS, "fo:root");
	 * 		foDoc.appendChild(root);
	 * 
	 * 		ele1 = foDoc.createElementNS(foNS, "fo:layout-master-set");
	 * 		root.appendChild(ele1);
	 * 		ele2 = foDoc.createElementNS(foNS, "fo:simple-page-master");
	 * 		ele1.appendChild(ele2);
	 * 		ele2.setAttributeNS(null, "master-name", "letter");
	 * 		ele2.setAttributeNS(null, "page-height", "11in");
	 * 		ele2.setAttributeNS(null, "page-width", "8.5in");
	 * 		ele2.setAttributeNS(null, "margin-top", "1in");
	 * 		ele2.setAttributeNS(null, "margin-bottom", "1in");
	 * 		ele2.setAttributeNS(null, "margin-left", "1in");
	 * 		ele2.setAttributeNS(null, "margin-right", "1in");
	 * 		ele3 = foDoc.createElementNS(foNS, "fo:region-body");
	 * 		ele2.appendChild(ele3);
	 * 		ele1 = foDoc.createElementNS(foNS, "fo:page-sequence");
	 * 		root.appendChild(ele1);
	 * 		ele1.setAttributeNS(null, "master-reference", "letter");
	 * 		ele2 = foDoc.createElementNS(foNS, "fo:flow");
	 * 		ele1.appendChild(ele2);
	 * 		ele2.setAttributeNS(null, "flow-name", "xsl-region-body");
	 * 		addElement(ele2, "fo:block", "Hello World!");
	 * 		return foDoc;
	 * 	}
	 * 
	 * 	protected static void addElement(Node parent, String newNodeName,
	 * 								String textVal) {
	 * 		if (textVal == null) {
	 * 			return;
	 * 		}  // use only with text nodes
	 * 		Element newElement = parent.getOwnerDocument().createElementNS(
	 * 										foNS, newNodeName);
	 * 		Text elementText = parent.getOwnerDocument().createTextNode(textVal);
	 * 		newElement.appendChild(elementText);
	 * 		parent.appendChild(newElement);
	 * 	}
	 *  
	 * 	public static void main(String[] args) {
	 * 		try {
	 * 			File baseDir = new File("bin");
	 * 			File outDir = new File(baseDir, "out");
	 * 			outDir.mkdirs();
	 * 
	 * 			File pdffile = new File(outDir, "出力PDF.pdf");
	 * 
	 * 			Document foDoc = buildDOMDocument();
	 * 			FopUtil.getInstance().convertDOM2PDF(foDoc, pdffile, "FOP設定ファイルパス");
	 * 		} catch (Exception e) {
	 * 			e.printStackTrace(System.err);
	 * 		}
	 * 	}
	 * 
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * @param foDom XSL-FO形式のDOM
	 * @param pdf PDFファイル
	 * @param configPath 設定ファイルパス
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertDOM2PDF(Document foDom, File pdf, String configPath) throws CoreLogicException {
		FileOutputStream out = null;
		try {
			// 設定ファイルの指定
			FopFactory fopFactory = FopFactory.newInstance();
			fopFactory.setUserConfig(convertConfiguration(configPath));

			// 出力タイプをPDFに指定
			FOUserAgent foUserAgent = fopFactory.newFOUserAgent();
			out = new FileOutputStream(pdf);
			Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, foUserAgent, out);

			// DOMからPDFファイルを生成する
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
			Source src = new DOMSource(foDom);
			Result res = new SAXResult(fop.getDefaultHandler());
			transformer.transform(src, res);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * SVGより、PDFを作成します。(SVG -> PDF)
	 * 
	 * <p>[詳 細] </p>
	 * SVGより、PDFを作成します。(SVG -> PDF)</br>
	 *
	 * （呼び出しサンプル）
	 * 
	 * <pre class="samplecode">
	 * 	public static void main(String[] args) {
	 * 		try {
	 * 			File baseDir = new File("bin");
	 * 			File outDir = new File(baseDir, "out");
	 * 			outDir.mkdirs();
	 * 
	 * 			File svgfile = new File(baseDir, "作成元SVG.svg");
	 * 			File pdffile = new File(outDir, "出力PDF.pdf");
	 * 
	 * 			FopUtil.getInstance().convertSVG2PDF(svgfile, pdffile);
	 * 		} catch (Exception e) {
	 * 			e.printStackTrace(System.err);
	 * 		}
	 * 	}
	 * 
	 * </pre>
	 * 
	 * <p>[備 考] </p>
	 * @param svg SVGファイル
	 * @param pdf PDFファイル
	 * @throws CoreLogicException FOP処理例外
	 */
	public void convertSVG2PDF(File svg, File pdf) throws CoreLogicException {
		FileInputStream in = null;
		BufferedOutputStream out = null;
		try {
			// 入出力ファイル設定
			in = new FileInputStream(svg);
			out = new BufferedOutputStream(new FileOutputStream(pdf));

			// SVGファイルからPDFを生成する
			TranscoderInput input = new TranscoderInput(in);
			Transcoder transcoder = new PDFTranscoder();
			TranscoderOutput output = new TranscoderOutput(out);
			transcoder.transcode(input, output);
		} catch (FileNotFoundException e) {
			throw new CoreLogicException("EFC9002", e);
		} catch (Exception e) {
			throw new CoreLogicException("EFC9001", e);
		} finally {
			IOUtils.closeQuietly(in);
			IOUtils.closeQuietly(out);
		}
	}
}
