package jp.co.fujitsu.reffi.client.android.manager;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import jp.co.fujitsu.reffi.client.android.event.ModelProcessEvent;
import jp.co.fujitsu.reffi.client.android.model.TimerProcessCore;

/**
 * <p>[概 要] </p>
 * TimerProcessCore機能モデルを管理するマネージャクラスです。
 * 
 * <p>[詳 細] </p>
 * タイマーを開始したTimerProcessCoreをtimerIdをキーにして管理します。<br>
 * タイマーを停止する際、指定された任意のtimerIdを持つTimerProcessCoreに
 * タイマー停止を命じます。<br>
 * 
 * 
 * <p>[備 考] </p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class TimerProcessCoreManager {
	
	/** このマネージャのシングルトンインスタンスです。 */
	private static TimerProcessCoreManager instance;
	
	/** TimerProcessCoreを管理するマップオブジェクトです。 */
	private Map<String, TimerProcessCore> timers = new HashMap<String, TimerProcessCore>();
	
	
	/**
	 * <p>[概 要] </p>
	 * TimerProcessCoreを管理するマップオブジェクトを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * timersフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return TimerProcessCoreを管理するマップオブジェクト
	 */
	public Map<String, TimerProcessCore> getTimers() {
		return timers;
	}

	/**
	 * <p>[概 要] </p>
	 * TimerProcessCoreを管理するマップオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * timersフィールドを引数timersで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timers TimerProcessCoreを管理するマップオブジェクト
	 */
	public void setTimers(Map<String, TimerProcessCore> timers) {
		this.timers = timers;
	}

	/**
	 * <p>[概 要] </p>
	 * プライベートコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * インスタンス生成には{@link #getInstance()}を使用します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	private TimerProcessCoreManager() {
	}
	
	/**
	 * <p>[概 要] </p>
	 * シングルトンインスタンス取得メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * TimerProcessCoreManagerのインスタンスはJVM内でユニークです。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return マネージャのシングルトンインスタンス
	 */
	public static TimerProcessCoreManager getInstance() {
		if(instance == null) {
			instance = new TimerProcessCoreManager();
		}
		
		return instance;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数timerIdを持つTimerProcessCoreが既に管理されているか調べます。
	 * 
	 * <p>[詳 細] </p>
	 * timers管理マップに引数timerIdがキーとして登録されているかどうか
	 * 調べて返却します。<br>
	 * trueの場合、タイマーをstartしているTimerProcessCoreが既に存在します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerId タイマーの識別子
	 * @return true : 既登録、false : 未登録
	 */
	public boolean isRegist(String timerId) {
		return getTimers().containsKey(timerId);
	}
	
	/**
	 * <p>[概 要] </p>
	 * タイマー処理を開始します。
	 * 
	 * <p>[詳 細] </p>
	 * 既に同名のtimerIdを持つtimerProcessCoreが存在する場合はfalseを返却します。<br>
	 * 
	 * ScheduledThreadPoolExecuterに定期処理を委譲します。<br>
	 * Executerの実行にはTimerProcessCoreに設定されている情報が使用されます。<br>
	 * 
	 * <ol>
	 *   <li>初期遅延：{@link TimerProcessCore#getInitialDelay()}</li>
	 *   <li>実行間隔：{@link TimerProcessCore#getPeriod()}</li>
	 *   <li>終了時間：{@link TimerProcessCore#getStopLater()}（0の場合は自動停止無し）</li>
	 *   <li>実行処理：{@link TimerProcessCore#getIntervalAction()} （nullの場合は成功イベント発行のみ）</li>
	 * </ol>
	 *
	 * タイマー処理を開始したTimerProcessCoreは、timerIdをキーにtimersフィールドで管理開始されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerProcessCore 開始するタイマー情報を持った機能モデルインスタンス
	 * @return true : タイマー開始成功、 false : 失敗
	 */
	public boolean start(final TimerProcessCore timerProcessCore) {
		if(isRegist(timerProcessCore.getTimerId())) {
			return false;
		}
		
		ScheduledThreadPoolExecutor executorService = 
			(ScheduledThreadPoolExecutor)Executors.newScheduledThreadPool(1);
		executorService.setContinueExistingPeriodicTasksAfterShutdownPolicy(true);
		executorService.setExecuteExistingDelayedTasksAfterShutdownPolicy(true);
		
		timerProcessCore.setExecutorService(executorService);	
		
		ScheduledFuture<?> future = executorService.scheduleAtFixedRate(new Runnable() {
			public void run() {
				if(timerProcessCore.getIntervalAction() != null) {
					timerProcessCore.getController().invoke(timerProcessCore.getIntervalAction(), 
							timerProcessCore.getParameterMapping().clone());
				}else{
					timerProcessCore.timerCompleted();
				}
			}
		}, timerProcessCore.getInitialDelay(), timerProcessCore.getPeriod(), TimeUnit.MILLISECONDS);
		
		if(timerProcessCore.getStopLater() > 0) {
			executorService.schedule(new Runnable() {
				public void run() {
					stop(timerProcessCore.getTimerId());
				}
			}, timerProcessCore.getStopLater(), TimeUnit.MILLISECONDS);
		}
		timerProcessCore.setFuture(future);
		getTimers().put(timerProcessCore.getTimerId(), timerProcessCore);
		
		return true;
	}
	
	/**
	 * <p>[概 要] </p>
	 * タイマー処理を停止します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数timerIdを持つTimerProcessCoreに対してタイマー処理の停止を命じます。<br>
	 * timerIdを持つTimerProcessCoreが管理されていない場合はfalseを返却します。<br>
	 * タイマー停止が行われた後、timersフィールドからtimerIdがキーのTimerProcessCoreを削除します。<br>
	 * <p>
	 * 
	 * これらの処理が正常に行われた後、タイマー処理を停止したTimerProcessCoreに対して
	 * ModelProcessEvent.FINISHEDイベントを発行され、BaseAction#completeがコールバックされます。<br>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param timerId タイマーの識別子
	 * @return true : 停止成功、false : 停止失敗
	 */
	public boolean stop(String timerId) {
		if(!isRegist(timerId)) {
			return false;
		}
		
		TimerProcessCore timerProcessCore = getTimers().get(timerId);
		timerProcessCore.getFuture().cancel(true);
		timerProcessCore.getExecutorService().shutdown();
		getTimers().remove(timerId);
		
		ModelProcessEvent finishedEvent = new ModelProcessEvent(timerProcessCore);
		timerProcessCore.fireModelFinished(finishedEvent);

		return true;
	}
}
