/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.controller{
	
	import flash.display.DisplayObject;
	import flash.utils.Dictionary;
	
	import jp.co.fujitsu.reffi.client.flex.util.CompositionSearchUtil;
	
	/**
	 * <p>[概 要] </p>
	 * イベント紐付け情報保持クラスです.
	 * 
	 * <p>[詳 細] </p>
	 * 具象コントローラ#bind(EventBinder)実装によるコンポーネント、イベントタイプ、アクションクラス
	 * の紐付け情報を、bindInfoプロパティに保持します。<br />
	 * bindInfoプロパティはコントローラレイヤ内の以下の二箇所から参照されます。
	 * <ol>
	 *   <li>
	 *     <b>Application、SystemManagerに対するコンポーネント追加、削除リスナ</b><br />
	 *     画面表示リストにコンポーネントが追加された際、コンポーネント名を調べ、bindInfoに登録されている名前であれば、<br />
	 *     登録されているイベントタイプでリスナを追加し、汎用イベントハンドラを登録します。<br />
	 *     addedComponent.addEventListener("登録されているイベントタイプ", BaseController.handerFacade);
	 *   </li>
	 *   <li>
	 *     <b>汎用イベントハンドラ（BaseController#handlerFacade()）</b><br />
	 *     1でイベントリスナ登録されたコンポーネントがイベントディスパッチした際に呼ばれます。<br />
	 *     発生したイベントタイプとイベントを発生させたコンポーネント名から、紐付けられたアクションを検出します。<br />
	 * 	</li>
	 * </ol>
	 * 
	 * bindInfoプロパティは以下のような構造を持ちます。
	 * 
	 * <listing version="3.0">
	 * bindInfo:Dictionary
	 *     "component name1" = typeMap:Dictionary
	 *                             "event type name1" = actionInfoList:Array
	 *                                                     [0] = actionInfo:Dictionary
	 *                                                               "actionClass" = Class(Action)
	 *                                                               "useCapture" = false
	 *                                                               "priority" = 0
	 *                                                               "useWeakReference" = false
	 *                                                     [1] = actionInfo:Dictionary
	 *                                                               "actionClass" = Class(Action)
	 *                                                               "useCapture" = false
	 *                                                               "priority" = 0
	 *                                                               "useWeakReference" = false
	 *                             "event type name2" = actionInfoList:Array
	 *                                                     [0] = actionInfo:Dictionary
	 *                                                               "actionClass" = Class(Action)
	 *                                                               "useCapture" = false
	 *                                                               "priority" = 0
	 *                                                               "useWeakReference" = false
	 *     "component name2" = typeMap:Dictionary
	 *                             "event type name1" = actionInfoList:Array
	 *                                                     [0] = actionInfo:Dictionary
	 *                                                               "actionClass" = Class(Action)
	 *                                                               "useCapture" = false
	 *                                                               "priority" = 0
	 *                                                               "useWeakReference" = false
	 * </listing>
	 * 
	 * 具象コントローラ#bindメソッドで以下のようなイベント定義を記述した場合、<p>
	 * 
	 * eventBinder.addEventBinding("menu.openChat", FocusEvent.FOCUS_IN, Class(FocusChatAction));<br />
	 * eventBinder.addEventBinding("menu.openChat", MouseEvent.CLICK, Class(OpenChatAction));<br />
	 * eventBinder.addEventBinding("menu.openWindow", MouseEvent.CLICK, Class(OpenWindowGroupParentAction));<p>
	 * 
	 * bindInfoの構造は以下のようになります。
	 * 
	 * <listing version="3.0">
	 * bindInfo:Dictionary
	 *     "menu.openChat"   = typeMap:Dictionary
	 *                             FocusEvent.FOCUS_IN = actionInfoList:Array
	 *                                                       [0] = actionInfo:Dictionary
	 *                                                                 "actionClass" = Class(FocusChatAction)
	 *                                                                 "useCapture" = false
	 *                                                                 "priority" = 0
	 *                                                                 "useWeakReference" = false
	 *                             MouseEvent.CLICK    = actionInfoList:Array
	 *                                                       [0] = actionInfo:Dictionary
	 *                                                                 "actionClass" = Class(OpenChatAction)
	 *                                                                 "useCapture" = false
	 *                                                                 "priority" = 0
	 *                                                                 "useWeakReference" = false
	 *     "menu.openWindow" = typeMap:Dictionary
	 *                             MouseEvent.CLICK    = actionInfoList:Array
	 *                                                       [0] = actionInfo:Dictionary
	 *                                                                 "actionClass" = Class(OpenWindowGroupParentAction)
	 *                                                                 "useCapture" = false
	 *                                                                 "priority" = 0
	 *                                                                 "useWeakReference" = false
	 * </listing>
	 * 
	 * <p>[備 考] </p>
	 * アプリケーション動作中に紐付け情報を動的追加する場合、<br />
	 * 以下のようにEventBinderオブジェクトを取得してaddEventBindingImmediatelyをコールします。<br />
	 * <listing version="3.0">
	 * 	controller.eventBinder.addEventBindingImmediately("newdisplay.btnRegist", MouseEvent.CLICK, Class(RegistAction));
	 * </listing>
	 * 
	 * ＊１コンポーネントに対して同じイベントで同じアクションを複数登録することは出来ません。
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi 
	 */
	public class EventBinder{

		/** コンポーネント名、eventタイプ、Actionクラスの紐付け情報を保持するオブジェクトです */
		private var _bindInfo:Dictionary;
		
		/** コントローラオブジェクトです */
		private var _controller:BaseController;
		
		/**
		 * <p>[概 要]</p>
		 * イベント定義情報を保持するオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * コンポーネントの名前、イベントタイプ、アクションクラスをマッピングします。
		 * 
		 * <p>[備 考]</p>
		 */
		public function get bindInfo():Dictionary{
			return this._bindInfo;
		}
		public function set bindInfo(bindInfo:Dictionary):void{
			this._bindInfo = bindInfo;
		}
		
		/**
		 * <p>[概 要]</p>
		 * このEventBinderを保持しているコントローラオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * イベントハンドリングを委譲されたコントローラは、このEventBinder
		 * オブジェクトを参照して、イベントを起こしたコンポーネントの名前と
		 * イベントタイプを元に、実行するべきアクションクラスを取得します。
		 * 
		 * <p>[備 考]</p>
		 */
		public function get controller():BaseController {
			return this._controller;
		}
		public function set controller(controller:BaseController):void {
			this._controller = controller;
		}
		
		/**
		 * <p>[概 要]</p>
		 * コンストラクタです.
		 * 
		 * <p>[詳 細]</p>
		 * bindInfo領域の生成、引数controllerのプロパティへの設定を行います。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param controller 
		 */
		public function EventBinder(controller:BaseController){
			this.bindInfo = new Dictionary();
			this.controller = controller;
		}
		
		/**
		 * <p>[概 要]</p>
		 * イベント紐付け情報追加メソッドです.
		 *  
		 * <p>[詳 細]</p>
		 * 引数componentNameが既に登録されている場合、componentName用のMapを取り出し、
		 * イベントタイプをキーにしてアクションを登録します。<br />
		 * componentNameが未登録の場合、componentName用のMapを新規に生成して、
		 * イベントタイプをキーにしてアクションを登録します。<br />
		 * 登録された情報はイベント紐付け情報保持オブジェクト（bindInfoプロパティ）に保持され、
		 * コントローラのエレメント挿抜リスナによって参照されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param componentName コンポーネント（エレメント）のname属性値
		 * @param eventType イベントタイプ
		 * @param actionClass 起動するアクションのクラス型
		 * @param useChapture
		 * @param priority
		 * @param useWeakReference
		 */
		public function addEventBinding(componentName:String, 
										  type:String, 
										  actionClass:Class, 
										  useCapture:Boolean=false, 
										  priority:int=0, 
										  useWeakReference:Boolean=false):void
		{
			// 一コンポーネントに対するイベント情報保持オブジェクト
			var typeMap:Dictionary = null;
			// イベント情報紐付けオブジェクトに、指定されたコンポーネント名が登録されている場合
			if(bindInfo.hasOwnProperty(componentName)){
				typeMap = bindInfo[componentName];
			}else{
				typeMap = new Dictionary();
				bindInfo[componentName] = typeMap;
			}
			
			// 1コンポーネント1イベントタイプに対応する実行予定アクション情報リスト
			var actionInfoList:Array = null;
			if(typeMap.hasOwnProperty(type)){
				actionInfoList = typeMap[type];
			}else{
				actionInfoList = new Array();
				typeMap[type] = actionInfoList;
			}

			// 実行予定アクション情報リストに既に同じアクションが登録されている場合は追加しない
			for each(var tmpInfo:Dictionary in actionInfoList) {
				if(tmpInfo["actionClass"] == actionClass) {
					return;
				}
			}
			
			// 実行予定アクション情報を作成、登録
			var actionInfo:Dictionary = new Dictionary();
			actionInfo["actionClass"] = actionClass;
			actionInfo["useCapture"] = useCapture;
			actionInfo["priority"] = 0;
			actionInfo["useWeakReference"] = useWeakReference;
			actionInfoList.push(actionInfo);
		}

		/**
		 * <p>[概 要]</p>
		 * イベント紐付け情報追加メソッドです.
		 *  
		 * <p>[詳 細]</p>
		 * componentNames配列内のコンポーネント名全てに引数typeイベントタイプで
		 * 引数actionClassを紐付けます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param componentNames コンポーネント（エレメント）のname属性値配列
		 * @param eventType イベントタイプ
		 * @param actionClass 起動するアクションのクラス型
		 * @param useChapture
		 * @param priority
		 * @param useWeakReference
		 */
		public function addEventBindings(componentNames:Array, 
										  type:String, 
										  actionClass:Class, 
										  useCapture:Boolean=false, 
										  priority:int=0, 
										  useWeakReference:Boolean=false):void
		{
			for each(var componentName:String in componentNames) {
				addEventBinding(componentName, type, actionClass, useCapture, priority, useWeakReference);
			}
		}

		/**
		 * <p>[概 要] </p>
		 * イベント紐付け情報追加メソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * addEventBinding(String, String, Class)メソッドを
		 * 呼び出して、イベント紐付け情報保持オブジェクト（bindInfo）に追加します。<br />
		 * 追加後、ui Documentを取得して、追加された紐付け情報を即座に反映します。
		 * 
		 * <p>[備 考] </p>
		 * addEventBindingと違い、登録された情報が画面オブジェクトに
		 * 反映されるのはこのメソッド呼び出し直後です。
		 * 
		 * @param componentName コンポーネント（エレメント）のname属性値
		 * @param eventType イベントタイプ
		 * @param actionClass 起動するアクションのクラス型
		 * @see #addEventBinding()
		 */
		public function addEventBindingImmediately(componentName:String, 
										  type:String, 
										  action:Class, 
										  useCapture:Boolean=false, 
										  priority:int=0, 
										  useWeakReference:Boolean=false):void
		{
			addEventBinding(componentName, type, action, useCapture, priority, useWeakReference);
			
			var components:Array = 
				CompositionSearchUtil.searchComponentsByName(controller.application, componentName, CompositionSearchUtil.ALL_RANGE);
			for each(var component:DisplayObject in components) {
				component.addEventListener(type, controller.handlerFacade);
			}
		}
		
		/**
		 * <p>[概 要]</p>
		 * 引数componentNameに対するイベント紐付け情報があるかどうか調べます. 
		 * 
		 * <p>[詳 細]</p>
		 * binInfoプロパティにcomponentNameプロパティが存在するか調べて返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param componentName イベント紐付け情報が登録されているか調べるコンポーネント名
		 */
		public function hasEventBinding(componentName:String):Boolean {
			return bindInfo.hasOwnProperty(componentName);
		}

		/**
		 * <p>[概 要] </p>
		 * 引数指定された名前を持つエレメントに登録されているイベントタイプ群を返却します.
		 * 
		 * <p>[詳 細] </p>
		 * bindInfoフィールドから、引数componentNameで紐付け情報を取り出し、
		 * 登録されている全イベントタイプをString配列に変換して返却します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @param componentName コンポーネントのname属性値
		 * @return componentNameに登録されている全イベントタイプが入った配列
		 */
		public function getEventTypes(componentName:String):Array {
			var ret:Array = new Array();
			
			// イベント紐付け情報に引数componentNameが登録されている場合
			if(hasEventBinding(componentName)) {
				// コンポーネントに登録されているイベントタイプマップを取得
				var typeMap:Dictionary = bindInfo[componentName];
				// イベントタイプマップのキー名を返却用配列に格納
				for(var eventType:String in typeMap) {
					ret.push(eventType);
				}
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要] </p>
		 * componentNameをname属性値として持つエレメントの、eventTypeイベント発動時の
		 * アクションクラス型配列を取得します.
		 * 
		 * <p>[詳 細] </p>
		 * bindInfoフィールドから、引数componentNameで紐付け情報を取り出し、
		 * 取り出した[イベントタイプ = アクションクラス]のマップから、eventTypeイベント
		 * のアクションクラス群を取得、返却します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @param componentName コンポーネントのname属性値
		 * @param eventType イベントタイプ
		 * @return componentName、eventTypeに対応するアクションクラス型
		 */
		public function getActionClasses(componentName:String, eventType:String):Array {
			var ret:Array = new Array();

			// イベント紐付け情報に引数componentNameが登録されている場合
			if(hasEventBinding(componentName)) {
				// コンポーネントに登録されているイベントタイプマップを取得
				var typeMap:Dictionary = bindInfo[componentName];
				var actionInfoList:Array = typeMap[eventType];
				for each(var actionInfo:Dictionary in actionInfoList) {
					ret.push(actionInfo["actionClass"]);
				}
			}
			
			return ret;
		}
		
		/**
		 * <p>[概 要]</p>
		 * イベント紐付け情報削除メソッドです.
		 *  
		 * <p>[詳 細]</p>
		 * 保持されているコンポーネン紐付け情報から、引数componentNameの
		 * 引数typeイベントに対する紐付けを削除します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @param componentName コンポーネントのname属性値
		 * @param type アクション紐付け情報を削除するイベントタイプ
		 */
		public function removeEventBinding(componentName:String, 
										  type:String):void
		{
			
			// 一コンポーネントに対するイベント情報保持オブジェクト
			var componentInfo:Dictionary = null;
			
			// イベント情報紐付けオブジェクトを取得
			if(!bindInfo.hasOwnProperty(componentName)){
				// 指定されたコンポーネント名がまだ登録されていない場合は終了
				return;
			}else{
				componentInfo = bindInfo[componentName];
			}
			
			// コンポーネントから対象のイベントを削除
			if(!componentInfo.hasOwnProperty(type)){
				// 対象のイベントが存在しない時は終了
				return;
			}else{
				delete componentInfo[type];
			}
		}
	}
}
