/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.config.cmd;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.co.fujitsu.reffi.client.config.ReffiClientConfigDocument;
import jp.co.fujitsu.reffi.client.config.ImportDocument.Import;
import jp.co.fujitsu.reffi.client.config.exception.ConfigException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xmlbeans.XmlError;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;

/**
 * <p>[概 要]</p>
 *  設定ファイルチェッククラス.
 * <p>[詳 細]</p>
 *  記述した設定ファイルのフォーマットチェックを行います。
 *  <br />
 *  チェックを実行する際、設定ファイルはクラスパスのルートに含まれている必要があります。
 *  また、ファイル名は「reffi-client-config.xml」とする必要があります。
 *  
 * <p>[備 考]</p>
 *  記述内容について、イベント形式やクラスの存在可否はチェックしません。
 * 
 * @author Project Reffi
 */
public class ConfigCheck {
	/** */
	private static final String DEFAULT_CONFIG_FILE = "reffi-client-config.xml";
	/** */
	private static final Log LOG = LogFactory.getLog(ConfigCheck.class);
	
	/**
	 * 設定ファイルチェック
	 * <br />
	 * 設定ファイルのチェックを行います。
	 * 
	 * @throws ConfigException 設定ファイル記述例外
	 */
	public void run() throws ConfigException {
		try {
			validate(DEFAULT_CONFIG_FILE);
		} catch (XmlException e) {
			LOG.error("XmlException", e);
			throw new ConfigException(e);
		} catch (IOException e) {
			LOG.error("IOException", e);
			throw new ConfigException(e);
		}
	}
	
	private void printErrors(List<XmlError> errors) {
		LOG.info("Erros discovered during validation:");
		for (Iterator<XmlError> itr = errors.iterator(); itr.hasNext(); ) {
			XmlError error = itr.next();
			String errorCode = error.getErrorCode();
			int line = error.getLine();
			int column = error.getColumn();
			String message = error.getMessage();
			LOG.error(">> " + errorCode + ": " + message + " (" + line + ", " + column + ")");
        }
	}
	
	private void validate(String resourceFile) throws XmlException, IOException {
		List<XmlError> errors = new ArrayList<XmlError>();
		XmlOptions xmlOptions = new XmlOptions();
		xmlOptions.setErrorListener(errors);
		InputStream inputStream = getClass().getClassLoader().getResourceAsStream(resourceFile);
		ReffiClientConfigDocument document = ReffiClientConfigDocument.Factory.parse(inputStream);
		boolean ret = document.validate(xmlOptions);
		if (!ret) {
			LOG.error(resourceFile + ": errors found.");
			printErrors(errors);
		} else {
			LOG.info(resourceFile + ": no errors found.");
		}
		Import[] imports = document.getReffiClientConfig().getImportArray();
		for (int i = 0; i < imports.length; i++) {
			validate(imports[i].getResource());
		}
	}
	
	public static void main(String[] argv) throws Exception {
		ConfigCheck configCheck = new ConfigCheck();
		configCheck.run();
	}
}
