/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.localConnection{

	import flash.events.Event;
	
	import jp.co.fujitsu.reffi.client.flex.error.CoreLogicError;
	import jp.co.fujitsu.reffi.client.flex.events.LocalConnectionEvent;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	import jp.co.fujitsu.reffi.client.flex.manager.LocalConnectionManager;
	import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	
	/**
	 * <p>[概 要] </p>
	 * LocalConnectionに接続するためのモデルクラスです.
	 * 
	 * <p>[詳 細] </p>
	 * 任意の接続名称でLocalConnectionに、接続、接続解除を行います。<br />
	 * connectedプロパティがtrueの場合は接続、falseの場合は接続解除します。
	 * <br /> 
	 * この機能モデルを拡張したモデルを作成する場合、onSuccessメソッドとonFailureメソッドをオーバーライド
	 * することで、LocalConnectionのsendメソッド受信成功処理、失敗処理を実装することが出来ます。<br />
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @example 
	 * LocalConnectionへの接続
	 * <listing version="3.0">
        public class LocalConnectionStartAction extends BaseAction {

            // ① 接続を行うLocalConnectionReceiverCore機能モデルを登録します    
            override protected function reserveModels(models:Array):void {
                models.push(Class(LocalConnectionReceiverCore));
            }
        
            // ② LocalConnectionReceiverCoreインスタンスの設定を行います.このケースではlocalConnectionDemoという名称でLocalConnectionに接続します
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                if (index == 0) {
                    LocalConnectionReceiverCore(next).localConnectionName = "localConnectionDemo";
                }
                return true;
            }
        
            // ③ sendメソッド受信成功時の処理を実装します
            //    LocalConnectionReceiverCoreは継続的にSUCCESSイベントを返却する為、localConnectionの接続を解除するまで
            //    このメソッドへのコールバックが継続します
            //    LocalConnectionManagerを使用することで任意のタイミングで接続解除可能です
            override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
                var event:LocalConnectionEvent = resultEvent as LocalConnectionEvent;
                // 引数取得
                var receiveArgs:Array = event.args;
                // 引数を使用した特定の処理を行う.
            }
            
            // ④ 登録モデルが完全に完了した時に呼ばれるメソッドを実装します
            //    継続的に値を返却するモデルの完了タイミングをハンドルします 
            override public function complete():void {
                trace("LocalConnection is done");
            }
        }
	 * </listing>
	 * 
	 * 既に実行中のLocalConnectionの接続を解除する
	 * <listing version="3.0">
        public class LocalConnectionStopAction extends BaseAction {

            // ① LocalConnection接続解除を行うLocalConnectionReceiverCore機能モデルを登録します    
            override protected function reserveModels(models:Array):void {
                models.push(Class(LocalConnectionReceiverCore));
            }
            
            // ② connectedモードをfalseに設定したLocalConnectionReceiverCoreが実行されることでLocalConnectionを停止します
            override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
                if (index == 0){
                    ConsumerCore(next).localConnectionName = "localConnectionDemo";
                    ConsumerCore(next).connected = false;
                }
                return true;
            }
        }
	 * </listing>
	 * <p>Copyright (c) 2008-2010 FUJITSU Japan All rights reserved.</p>
	 * 
	 * @see jp.co.fujitsu.reffi.client.flex.manager.LocalConnectionManager
	 * @author Project Reffi 
	 */
	public class LocalConnectionReceiverCore extends BaseModel{

		// LocalConnection名称
		private var _localConnectionName:String;

		// 接続許可するドメイン
		private var _allowDomain:Array = new Array();

		// HTTPS取得したSWFのLocalConnectionを、HTTP取得したSWFに公開するかどうかのフラグ
		private var _https:Boolean = false;

		// LocalConnection接続フラグ
		private var _connected:Boolean = true;

		/**
		 * <p>[概 要]</p>
		 * LocalConnection接続名称です.
		 * 
		 * <p>[詳 細]</p>
		 * allowDomainに「~~*~」を指定する場合、
		 * localConnectionNameの接頭辞にアンダーバーをつけなければ公開されません。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get localConnectionName():String{
			return this._localConnectionName;
		}
		public function set localConnectionName(localConnectionName:String):void{
			this._localConnectionName = localConnectionName;
		}

		/**
		 * <p>[概 要]</p>
		 * LocalConnectionを許可するドメインの配列です.
		 * 
		 * <p>[詳 細]</p>
		 * 許可するドメイン分、登録してください。
		 * allowDomainに「~~*~」を指定すると、全てのドメインに公開されます。
		 * allowDomainに「~~*~」を指定する場合、
		 * localConnectionNameの接頭辞にアンダーバーをつけなければ公開されません。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get allowDomain():Array{
			return this._allowDomain;
		}
		public function set allowDomain(allowDomain:Array):void{
			this._allowDomain = allowDomain;
		}

		/**
		 * <p>[概 要]</p>
		 * HTTPS取得したSWFのLocalConnectionを、HTTP取得したSWFに公開するかどうかのフラグです.
		 * 
		 * <p>[詳 細]</p>
		 * HTTPSドメインから取得したSWFでLocalConnection接続している場合、
		 * HTTPドメインから取得したSWFに接続許可する場合はtrueを指定してください。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get https():Boolean{
			return this._https;
		}
		public function set https(https:Boolean):void{
			this._https = https;
		}

		/**
		 * <p>[概 要]</p>
		 * LocalConnection接続フラグです.
		 * 
		 * <p>[詳 細]</p>
		 * trueでモデルを実行した場合は、LocalConnectionの接続をします。
		 * falseでモデルを実行した場合は、LocalConnectionの接続を解除します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */
		public function get connected():Boolean{
			return this._connected;
		}
		public function set connected(connected:Boolean):void{
			this._connected = connected;
		}

		/**
		 * <p>[概 要] </p>
		 * LocalConnectionの接続／接続解除を行います.
		 * 
		 * <p>[詳 細] </p>
		 * LocalConnectionの接続／接続解除を行います。
		 * connectedモードがtrueの場合は、LocalConnectionの接続をします。
		 * connectedモードがfalseの場合は、LocalConnectionの接続を解除します。
		 * 
		 */
		override protected function mainProc() : void {
			super.mainProc();
			var localConnectionName:String = this._localConnectionName;
			var connected:Boolean = this._connected;
			if (localConnectionName == null){
				var error:CoreLogicError = new CoreLogicError("EFC1010");
				throw error;
			}
			
			if (connected) {
				LocalConnectionManager.getInstance().connect(localConnectionName, this);
			} else {
				LocalConnectionManager.getInstance().unconnect(localConnectionName);

				// モデル成功イベントを生成			
				var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
				dispatchModelSuccess(modelSuccessEvent);

				// モデル完了イベントを生成			
				var modelFinishedEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
				dispatchModelFinished(modelFinishedEvent);
			}
		}

		/**
		 * <p>[概 要] </p>
		 * sendメソッド成功時に、LocalConnection経由で呼び出されるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * sendメソッド成功時に、LocalConnection経由で呼び出されるメソッドです。
		 * 
		 * <p>[備 考] </p>
		 * このメソッドはオーバーライド出来ません。<br/>
		 * 
		 * @param args sendメソッド実行時の引数
		 */
		public final function receive(args:Array):void{
			var event:LocalConnectionEvent = new LocalConnectionEvent(LocalConnectionEvent.RECEIVE, args);
			resultHandler(event);
		}

		/**
		 * <p>[概 要] </p>
		 * sendメソッド成功時に、呼び出されるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * sendメソッド成功時に、呼び出されるメソッドです。
		 * 
		 * <p>[備 考] </p>
		 * このメソッドはオーバーライド出来ません。<br/>
		 * メッセージ受信失敗処理を記述する場合はonSuccessメソッドをオーバーライドして下さい。
		 * 
		 * @param args LocalConnectionEvent
		 */
		public final function resultHandler(event:LocalConnectionEvent):void{
			onSuccess(event);

			// モデル成功イベントを生成			
			var modelSuccessEvent : ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
			modelSuccessEvent.cause = event;
			dispatchModelSuccess(modelSuccessEvent);
		}

		/**
		 * <p>[概 要]</p>
		 * sendメソッド結果受信オーバーライド用メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event LocalConnectionEvent
		 */
		protected function onSuccess(event:LocalConnectionEvent):void{
		}

		/**
		 * <p>[概 要]</p>
		 * SecurityErrorEvent、AsyncErrorEvent発生時にコールバックされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * 
		 * <p>[備 考] </p>
		 * このメソッドはオーバーライド出来ません。<br/>
		 * メッセージ受信失敗処理を記述する場合はonFailureメソッドをオーバーライドして下さい。
		 * 
		 * @param event Event
		 */
		public final function faultHandler(event:Event):void{
			onFailure(event);
			
			// モデル終了イベントを生成
			var modelFailureEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
			modelFailureEvent.cause = event;
			dispatchModelFailure(modelFailureEvent);
		}
		
		/**
		 * <p>[概 要]</p>
		 * SecurityErrorEvent、AsyncErrorEvent発生時オーバーライド用メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param event Event
		 */
		protected function onFailure(event:Event):void{
		}
	}
}