/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model.rpc {

	import flash.events.IEventDispatcher;
	import flash.net.URLVariables;
	
	import mx.rpc.http.HTTPService;

	/**
	 * <p>[概 要]</p>
	 * HTTPServiceによるサーバアクセスを行う機能モデルクラスです.
	 * 
	 * <p>[詳 細]</p>
	 * アクセス先URL指定、アクセスメソッド指定（デフォルト：POST）、
	 * 返却フォーマット指定（デフォルト：object）、URLパラメータ指定を行い、<br />
	 * HTTPリクエスト送信、レスポンス受信を行います。<br />
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @example リモートサーバからXMLデータを受信する
	 * 
	 * 受信するデータ
	 * <listing version="3.0">
	  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
	  &lt;employees&gt;
	      &lt;employee&gt;
	 	       &lt;name&gt;yamada taro&lt;/name&gt;
	 	       &lt;age&gt;32&lt;/age&gt;
	      &lt;/employee&gt;
	      &lt;employee&gt;
	          &lt;name&gt;suzuki ichiro&lt;/name&gt;
	          &lt;age&gt;35&lt;/age&gt;
	      &lt;/employee&gt;
	  &lt;/employees&gt;
	 * </listing>
	 * 
	 * HTTPServiceCoreを使用して受信処理を実装したアクション
	 * <listing version="3.0">
	  package demo.client.form.action {
	      import flash.events.Event;
	 
	      import jp.co.fujitsu.reffi.client.flex.action.BaseAction;
	      import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	      import jp.co.fujitsu.reffi.client.flex.model.BaseModel;
	      import jp.co.fujitsu.reffi.client.flex.model.events.ModelProcessEvent;
	      import jp.co.fujitsu.reffi.client.flex.model.rpc.HTTPServiceCore;
	      import jp.co.fujitsu.reffi.client.flex.model.rpc.RPCCore;
	 	
	      import mx.rpc.events.ResultEvent;
	 
	 	   public class SendAction extends BaseAction {
	 	
	          // ①HTTP通信の為の機能モデルを登録します
	  	       override protected function reserveModels(models:Array):void {
	 	            models.push(Class(HTTPServiceCore));
	          }
	 
	          // ②コントローラによってインスタンス化された機能モデルの設定を行います（サーバロジックは任意）
	          override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
	              switch(index) {
	                  case 0:
	                      HTTPServiceCore(next).url ="webController";
	                      HTTPServiceCore(next).addUrlParameter("model.fqcn", "demo.rpc.model.DummyRestResponseModel");
	                      HTTPServiceCore(next).addUrlParameter("forward.page", "DummyRestResponse.xml");
	                      HTTPServiceCore(next).resultFormat = RPCCore.E4X;
	                      break;
	              }
	              return true;
	          }
	 	
	          // ③受信結果を取得します
	          override public function successForward(index:int, model:BaseModel, resultEvent:Event):void {
	              trace(ResultEvent(resultEvent).result.employee[0].name);
	          }
	      }
	  }
	 * </listing>
	 * 
	 * 実行結果
	 * <listing version="3.0">
	 * yamada taro
	 * </listing>
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */ 
	public class HTTPServiceCore extends RPCCore {
		
		// アクセス先URL
		private var _url:String;
		
		// アクセスメソッド
		private var _httpMethod:String = "POST";
		
		// 受信データフォーマット
		private var _resultFormat:String = RPCCore.OBJECT;
		
		// URLパラメータ
		private var _urlParameter:Object = new URLVariables();
		
		// リクエスト本文
		private var _request:Object = null;
		
		// コンテンツタイプ
		private var _contentType:String = "application/x-www-form-urlencoded";

		// XMLエンコーダ
		private var _xmlEncode:Function = null;
		
		// XMLデコーダ
		private var _xmlDecode:Function = null;
		
		/**
		 * <p>[概 要]</p>
		 * アクセスを行う対象URLです.
		 * 
		 * <p>[詳 細]</p>
		 * 同コンテキストルートにアクセスする場合、ドメイン指定は不要です。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get url():String{
			return this._url;
		}
		public function set url(url:String):void{
			this._url = url;
		}
		
		/**
		 * <p>[概 要]</p>
		 * HTTPアクセスメソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * POST、GETを文字列指定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default "POST"
		 */ 
		public function get httpMethod():String{
			return this._httpMethod;
		}
		public function set httpMethod(httpMethod:String):void{
			this._httpMethod = httpMethod;
		}

		/**
		 * <p>[概 要]</p>
		 * 受信データフォーマットです.
		 * 
		 * <p>[詳 細]</p>
		 * 「object」「e4x」「xml」が指定可能です。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @default object
		 * @see jp.co.fujitsu.reffi.client.flex.model.rpc.RPCCore
		 */ 
		public function get resultFormat():String{
			return this._resultFormat;
		}
		public function set resultFormat(resultFormat:String):void{
			this._resultFormat = resultFormat;
		}

		/**
		 * <p>[概 要]</p>
		 * リクエストする際のURLパラメータです.
		 * 
		 * <p>[詳 細]</p>
		 * このプロパティに設定されたキー=値のセットが送信時のURLパラメータとして適用されます。
		 * 
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get urlParameter():Object{
			return this._urlParameter;
		}
		public function set urlParameter(urlParamter:Object):void {
			this._urlParameter = urlParamter;
		}

		/**
		 * <p>[概 要]</p>
		 * リクエスト本文です.
		 * 
		 * <p>[詳 細]</p>
		 * null以外が指定された場合、リクエスト本文として送信されます。
		 * 
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get request():Object{
			return this._request;
		}
		public function set request(request:Object):void {
			this._request = request;
		}
		
		/**
		 * <p>[概 要]</p>
		 * コンテンツタイプです.
		 * 
		 * <p>[詳 細]</p>
		 * HTTP通信時のコンテンツタイプを指定します。
		 * 
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get contentType():String{
			return this._contentType;
		}
		public function set contentType(contentType:String):void {
			this._contentType = contentType;
		}
		
		/**
		 * <p>[概 要]</p>
		 * XMLエンコーダです.
		 * 
		 * <p>[詳 細]</p>
		 * コンテンツタイプが application/xml で、渡された要求オブジェクトが Object の場合、
		 * このxmlEncodeを使用してオブジェクトに変換しようとします。
		 * xmlEncode プロパティがnullの場合、デフォルトの XMLEncoder を使用して、
		 * オブジェクトグラフを flash.xml.XMLNode オブジェクトに変換します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get xmlEncode():Function{
			return this._xmlEncode;
		}
		public function set xmlEncode(xmlEncode:Function):void {
			this._xmlEncode = xmlEncode;
		}

		/**
		 * <p>[概 要]</p>
		 * XMLデコーダです.
		 * 
		 * <p>[詳 細]</p>
		 * resultFormat がオブジェクトで、xmlDecode プロパティが設定されている場合、
		 * HTTPService が返す XML を使用してオブジェクトを作成します。
		 * xmlDecodeにnullが設定されている場合は、デフォルトの XMLDecoder を使用してこの処理が実行されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 */ 
		public function get xmlDecode():Function{
			return this._xmlDecode;
		}
		public function set xmlDecode(xmlDecode:Function):void {
			this._xmlDecode = xmlDecode;
		}

		
		/**
		 * <p>[概 要]</p>
		 * URLパラメータにパラメータを追加します.
		 * 
		 * <p>[詳 細]</p>
		 * 引数key = 引数valueの形式でurlParameterプロパティに新規パラメータを追加します。
		 * 
		 * <p>[備 考]</p>
		 * valueが配列の場合は引数key=引数value[0]&amp;引数key=引数value[1]の状態で
		 * パラメータが生成されます。
		 * 
		 * @param key URLパラメータキー
		 * @param value URLパラメータ値
		 */ 
		public function addUrlParameter(key:String, value:Object):Object {
			urlParameter[key] = value;
			
			return urlParameter;
		}

		/**
		 * <p>[概 要]</p>
		 * 具象通信オブジェクトを作成します.
		 * 
		 * <p>[詳 細]</p>
		 * HTTPServiceオブジェクトを生成して返却します。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return HTTPServiceオブジェクト
		 */ 
		override protected final function createConcreteService() : IEventDispatcher {
			return new HTTPService();
		}
		
		/**
		 * <p>[概 要]</p>
		 * キャッシュモード実行時、通信オブジェクトを識別する為のIDを返却します.
		 * 
		 * <p>[詳 細]</p>
		 * urlプロパティを返却します。<br/>
		 * cacheIdはcacheModeプロパティがtrueの場合のみ使用されます。
		 * 
		 * <p>[備 考]</p>
		 *
		 * @return 通信オブジェクトを識別する為のキャッシュID
		 */ 
		override protected function cacheId() : String {
			return this.url;
		}

		/**
		 * <p>[概 要]</p>
		 * HTTPリクエストを送信します.
		 * 
		 * <p>[詳 細]</p>
		 * urlプロパティ値に対して、httpMethodプロパティ値で、urlParameterプロパティ値を
		 * 引数にしてリクエストを送信します。<br />
		 * 受信した値にアクセスする為の型としてresultFormatプロパティ値が適用されます。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return AsyncToken
		 */ 
		override protected function send(dispatcher:IEventDispatcher) : Object{			
			
			// HTTP通信オブジェクトを生成 
			var service : HTTPService = dispatcher as HTTPService;
			
			// アクセスURLとURLパラメータを指定するテンプレートメソッドをコール
			service.url = this.url + "?" + this.urlParameter.toString();

			// アクセスメソッドを指定するテンプレートメソッドをコール
			service.method = this.httpMethod;
			
			// リクエスト本文を設定するテンプレートメソッドをコール
			service.request = this.request;
			
			// 受信データのフォーマットを指定するテンプレートメソッドをコール
			service.resultFormat = this.resultFormat;

			// コンテンツタイプを指定するテンプレートメソッドをコール
			service.contentType = this.contentType;
			
			// XMLエンコードを指定するテンプレートメソッドをコール
			service.xmlEncode = this.xmlEncode;

			// XMLデコードを指定するテンプレートメソッドをコール
			service.xmlDecode = this.xmlDecode;
			
			// HTTPリクエスト送信
			return service.send();
		}
		
		/**
		 * <p>[概 要]</p>
		 * 通信成功、失敗に関わらずコールされます.
		 *  
		 * <p>[詳 細]</p>
		 * 処理は有りません。
		 * 
		 * <p>[備 考]</p>
		 * 通信終了処理実装はonFinishをオーバーライドしてください。
		 */ 
		override protected final function finishConnect() : void {
			super.finishConnect();
		}
	}
}
