/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.httpd;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/09/28
 */
public class HTTPRequestContinuableParser {

	private static enum S { INIT, HEAD, POST, DONE }

	private static Pattern PTN = Pattern.compile(
			"([A-Z]+) +([^ ]+) +HTTP/([01].[0-9])");

	private Map<String, String> headers =
			new LinkedHashMap<String, String>();
	private Map<String, String> parameters =
			new LinkedHashMap<String, String>();

	private URLEncodedInputStream ins;
	private StringBuffer buf = new StringBuffer();
	private Matcher match = null;
	private String key, val;
	private Charset enc;
	private S state = S.INIT;
	private int etat = 0;

	/**
	 * 
	 * @param encoding
	 */
	public HTTPRequestContinuableParser(Charset encoding) {
		ins = new URLEncodedInputStream();
		key = val = null;
		enc = encoding;
	}

	private void completeString() throws IOException {
		String s;

		s = buf.toString();
		s = new String(s.getBytes("ISO-8859-1"), enc);
		if(!(match = PTN.matcher(s)).matches()) {
			throw new IOException();
		}
		buf = new StringBuffer();
		etat = 0;
	}

	boolean readString() throws IOException {
		int c;

		if(ins == null)  throw new IOException();
		for(; true; etat = c) {
			if((c = ins.read()) < 0)  return false;
			switch(etat) {
			case '\r':
				if(c == '\n') {
					completeString();
					state = S.HEAD;
					return true;
				} else {
					buf.append(etat).appendCodePoint(c);
					break;
				}
			default:
				if(c != '\r')  buf.appendCodePoint(c);
				break;
			}
		}
	}

	private boolean completeHeader() throws IOException {
		if(key != null) {
			val = buf.toString().trim();
		} else if(buf.length() > 0) {
			key = buf.toString().trim();
		} else if(match.group(1).equals("POST")) {
			state = S.POST;  etat = 0;
			return true;
		} else {
			state = S.DONE;  etat = 0;
			return true;
		}
		headers.put(key, val);
		key = val = null;
		buf = new StringBuffer();
		return false;
	}

	boolean readHeader() throws IOException {
		int c;

		if(ins == null) {
			state = S.DONE;
			return false;
		}

		for(; true; etat = c) {
			if((c = ins.read()) < 0)  return false;
			switch(etat) {
			case '\r':
				if(c != '\n') {
					buf.appendCodePoint(etat).appendCodePoint(c);
					break;
				} else if(completeHeader()) {
					return true;
				}
				break;
			case ':':
				if(key == null) {
					key = buf.toString().trim();
					buf = new StringBuffer();
				}
				buf.appendCodePoint(c);
				break;
			default:
				if(c != '\r' && !(c == ':' && key == null)) {
					buf.appendCodePoint(c);
				}
				break;
			}
		}
	}

	private boolean completeParameters() throws IOException {
		if(key != null) {
			val = buf.toString().trim();
		} else if(buf.length() > 0) {
			key = buf.toString().trim();
		} else {
			state = S.DONE;  etat = 0;
			return true;
		}
		parameters.put(key, val == null ? "" : val);
		key = val = null;
		buf = new StringBuffer();
		return false;
	}

	boolean readParameters() throws IOException {
		int c;

		for(; true; etat = c) {
			if((c = ins.read()) < 0)  return false;
			switch(etat) {
			case '\r':
				if(c != '\n') {
					buf.appendCodePoint(etat).appendCodePoint(c);
				} else if(completeParameters()) {
					return true;
				}
				break;
			case '=':
				if(key == null) {
					key = buf.toString().trim();
					buf = new StringBuffer();
				}
				buf.appendCodePoint(c);
				break;
			case '&':
				if(key != null) {
					val = buf.toString().trim();
				} else {
					key = buf.toString().trim();
				}
				parameters.put(key, val == null ? "" : val);
				key = val = null;
				buf = new StringBuffer().appendCodePoint(c);
				break;
			default:
				if(c != '\r' && c != '&' && c != '=') {
					buf.appendCodePoint(c);
				}
				break;
			}
		}
	}

	/**
	 * 
	 * @return
	 */
	public boolean isDone() {
		return state.equals(S.DONE);
	}

	/**
	 * 
	 * @return
	 */
	public HTTPRequest getRequest() throws IOException {
		switch(state) {
		case INIT:  completeString();  break;
		case HEAD:  completeHeader();  break;
		case POST:  completeParameters();  break;
		case DONE:  break;
		}
		state = S.DONE;
		return new HTTPRequest(match.group(1), match.group(2),
				match.group(3), headers, parameters);
	}

	/**
	 * 
	 * @param ins
	 * @throws IOException
	 */
	public void readPartial(InputStream in) throws IOException {
		boolean b = true;

		ins.newInputStream(in);
		while(b) {
			switch(state) {
			case INIT:  b = readString();  break;
			case HEAD:  b = readHeader();  break;
			case POST:  b = readParameters();  break;
			case DONE:
				while(ins.read() >= 0);
				return;
			}
		}
	}

}
