<?php
/* ========================================================================
 - [webapp/libs/private/dia_book.php]
 -      日記をPDFファイルにする
 -      Copyright (c) 2006 Yujiro Takahashi
 - ライセンス:
 -      This program is free software; you can redistribute it and/or
 -      modify it under the terms of the GNU General Public License
 -      as published by the Free Software Foundation; either version 2
 -      of the License, or (at your option) any later version.
 -
 -      This program is distributed in the hope that it will be useful,
 -      but WITHOUT ANY WARRANTY; without even the implied warranty of
 -      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 -      GNU General Public License for more details.
 -
 -      You should have received a copy of the GNU General Public License
 -      along with this program; if not, write to the Free Software
 -      Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 - 問い合わせ先：
 -      yujiro@rakuto.net
 -      http://rakuto.net/rktSNS/
 - 更新履歴：
 -      [2006/09/29] プログレスバーの修正
 -      [2006/09/24] 生成ページが無いときの処理を変更
 -      [2006/09/17] webappクラスから呼び出されるように変更
 -      [2006/09/14] SQLITE_TOOBIGの1MBの壁のためファイルとして保存
 -      [2006/09/14] リクエスエラーの緊急避難
 -      [2006/09/12] 画像追加
 -      [2006/09/11] 作成
 - ======================================================================== */

/* =-=- モジュール等の読み込み -=-= */
require_once LIB_DIR.'rkt_db.php';
require_once LIB_DIR.'rkt_blob.php';
require_once LIB_DIR.'rkt_image.php';
require_once LIB_DIR.'tcpdf/tcpdf.php';
require_once WEBAPP_LIB_DIR.'sql_query.php';

/**
 * dia_pdfクラス
 *
 * @author 高橋 裕志郎 <yujiro@rakuto.net>
 * @package enty_import
 * @access public
 * @version 1.0
 */
class dia_pdf extends TCPDF
{
    /**
     * ヘッダーテキストカラー
     * @var array
     */
    var $header_text_color = array();

	/**
 	 * This method is used to render the page header.
 	 * It is automatically called by AddPage() and could be overwritten in your own inherited class.
	 */
	function Header() 
    {
		if (!isset($this->original_lMargin)) {
			$this->original_lMargin = $this->lMargin;
		}
		if (!isset($this->original_rMargin)) {
			$this->original_rMargin = $this->rMargin;
		}
		
		//set current position
		$this->SetXY($this->original_lMargin, $this->header_margin);
		$this->img_rb_y = $this->GetY();

		$cell_height = round((K_CELL_HEIGHT_RATIO * $this->header_font[2]) / $this->k, 2);
		$header_x = $this->original_lMargin + ($this->header_logo_width * 1.05); //set left margin for text data cell

		// テキスト色の設定
        $this->SetTextColor($this->header_text_color['red'],
                            $this->header_text_color['green'],
                            $this->header_text_color['blue']);
		$this->SetLineWidth(0.3);

   		// ヘッダー文字の描画
		$this->SetFont($this->header_font[0], $this->header_font[1], $this->header_font[2]);
		$this->MultiCell($this->header_width, $cell_height, $this->header_title, 'B', 'C', 0);
		// print an ending header line

		//restore position
		$this->SetXY($this->original_lMargin, $this->tMargin);
	}

	/**
 	 * This method is used to render the page footer. 
 	 * It is automatically called by AddPage() and could be overwritten in your own inherited class.
	 */
	function Footer()
    {
		if (!isset($this->original_lMargin)) {
			$this->original_lMargin = $this->lMargin;
		}
		if (!isset($this->original_rMargin)) {
			$this->original_rMargin = $this->rMargin;
		}
		
		//set font
		$this->SetFont($this->footer_font[0], $this->footer_font[1] , $this->footer_font[2]);
		//set style for cell border
		$line_width = 0.3;
		$this->SetLineWidth($line_width);
		
		$footer_height = round((K_CELL_HEIGHT_RATIO * $this->footer_font[2]) / $this->k, 2); //footer height
		//get footer y position
		$footer_y = $this->h - $this->footer_margin - $footer_height;
		//set current position
		$this->SetXY($this->original_lMargin, $footer_y); 
		
		//print document barcode
		if ($this->barcode) {
			$this->Ln();
			$barcode_width = round(($this->w - $this->original_lMargin - $this->original_rMargin)); //max width
			$this->writeBarcode($this->original_lMargin, $footer_y + $line_width, $barcode_width, $footer_height - $line_width, "C128B", false, false, 2, $this->barcode);
		}
		
		$this->SetXY($this->original_lMargin, $footer_y); 
		
		//Print page number
		$this->Cell(0, $footer_height, $this->l['w_page']." ".$this->PageNo().' / {nb}', 'T', 0, 'R'); 
	}

    /**
     * ヘッダーテキスト色の設定
     *
     * @access public
     * @param array $color
     * @return void
     */
    function setHeaderTextColor($color)
    {
        $this->header_text_color = $color;
    }
} // dia_pdf


/**
 * dia_bookクラス
 *
 * @author 高橋 裕志郎 <yujiro@rakuto.net>
 * @package enty_import
 * @access public
 * @version 1.0
 */
class dia_book
{
    /**
     * 本・印刷ID
     * @var integer
     */
    var $id = null;

    /**
     * テーブル名
     * @var string
     */
    var $table = '';

    /**
     * WHERE句
     * @var string
     */
    var $where = '';

    /**
     * 本・印刷情報
     * @var array
     */
    var $book = array();

    /**
     * 画像の高さ
     * @var float
     */
    var $image_height = 0;

    /**
     * DB接続オジェクト
     * @var object
     */
    var $objdb = null;

    /**
     * バイナリファイル管理オジェクト
     * @var object
     */
    var $objblb = null;

    /**
     * 画像処理オジェクト
     * @var object
     */
    var $objimg = null;

    /**
     * PDFオジェクト
     * @var object
     */
    var $objpdf = null;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function dia_book($id)
    {
        set_time_limit (WAP_TIME_LIMIT);
        $this->id = $id;

        $this->objdb = RKT_db::getInstance();
        $this->objblb = RKT_blob::getInstance(BLOB_DRIVER,BLOB_PARAM);
        $this->objimg = RKT_image::factory(IMAGE_DRIVER);

        $this->objpdf = new dia_pdf(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true);
    }

    /**
     * 実行処理
     *
     * @access public
     * @return void
     **/
    function execute()
    {
        if (!$this->set_book()){
            return ;
        }
        $this->set_pdf();
        $this->set_condition();
        if ($this->create_page()){
            return ;
        }
        
        $mime_type = 'application/pdf';
        $name = 'diary_'.$this->book['ref_account'].'.pdf';
        $buffer = $this->objpdf->getPDFData();

        /* 保存 */
        if (WAP_DB_DRIVER == 'sqlite'){
            $filename = K_PATH_CACHE.$name;

    		$handle = fopen($filename,'wb');
            if (is_resource($handle)) {
        		fwrite($handle, $buffer, strlen($buffer));
        		fclose($handle);
            }
        } else {
            $filename = ARCHIVES_MEMBER_PATH.$this->book['ref_account'].'/pdf/'.$name;

            $this->objblb->setUID($this->book['ref_account']);
            $this->objblb->setMimeType($mime_type);
            $this->objblb->setBuffer($buffer);
            $this->objblb->save($filename);
        } // if (WAP_DB_DRIVER == 'sqlite')


        $this->created_book();

        flush();
        ob_flush();
        echo '<script type="text/javascript">setProgress(100,50);</script>'."\r\n";
        echo '<script type="text/javascript">setComplete();</script>'."\r\n";
    }

    /**
     * 本・印刷情報の設定
     *
     * @access private
     * @return boolean
     **/
    function set_book()
    {
        $sql=
            'SELECT '.
                'bok.id as id,'.
                'bok.ref_account as ref_account,'.
                'fnt.font as font,'.
                'sbj.rgb as sbj_rgb,'.
                'bdy.rgb as bdy_rgb,'.
                'lin.rgb as lin_rgb,'.
                'bok.start_date as start_date,'.
                'bok.end_date as end_date,'.
                'bok.comment_flag as comment_flag,'.
                'bok.subject as subject '.
            'FROM '.
                DB_PREFIX.'dia_font fnt,'.
                DB_PREFIX.'dia_color sbj,'.
                DB_PREFIX.'dia_color bdy,'.
                DB_PREFIX.'dia_color lin,'.
                DB_PREFIX.'dia_book bok '.
            'WHERE '.
                'fnt.id = bok.ref_font AND '.
                'sbj.id = bok.ref_subject_color AND '.
                'bdy.id = bok.ref_body_color AND '.
                'lin.id = bok.ref_line_color AND '.
                'bok.created = 0 AND '.
                'bok.id = '.$this->id;
        $stmt = $this->objdb->prepare($sql); 
        $stmt->execute(); 
        $this->book = $stmt->fetch(PDO_FETCH_ASSOC);
        $stmt->closeCursor();
        if (empty($this->book)){
            return false;
        }

        $this->book['sbj_color'] = $this->rgb2array($this->book['sbj_rgb']);
        $this->book['bdy_color'] = $this->rgb2array($this->book['bdy_rgb']);
        $this->book['lin_color'] = $this->rgb2array($this->book['lin_rgb']);

        return true;
    }

    /**
     * PDFオブジェクトの設定
     *
     * @access private
     * @return void
     **/
    function set_pdf()
    {
        // set document information
        $this->objpdf->SetCreator(PDF_CREATOR);
        $this->objpdf->SetAuthor(PDF_AUTHOR);
        $this->objpdf->SetTitle($this->book['subject']);
        $this->objpdf->SetSubject('日記の印刷');
        $this->objpdf->SetKeywords('rktSNS');

        //set margins
        $this->objpdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
        //set auto page breaks
        $this->objpdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
        $this->objpdf->SetHeaderMargin(PDF_MARGIN_HEADER);
        $this->objpdf->SetFooterMargin(PDF_MARGIN_FOOTER);
        $this->objpdf->setImageScale(PDF_IMAGE_SCALE_RATIO); //set image scale factor

        //フォントの追加
        $this->objpdf->AddFont($this->book['font'], '', $this->book['font'].'.php');

        $this->objpdf->SetHeaderData('', 0,  $this->book['subject'], '');
        $this->objpdf->setHeaderFont(Array($this->book['font'], '', PDF_FONT_SIZE_DATA));
        $this->objpdf->setFooterFont(Array($this->book['font'], '', PDF_FONT_SIZE_DATA));

        $this->objpdf->SetDrawColor(
                    $this->book['lin_color']['red'],
                    $this->book['lin_color']['green'],
                    $this->book['lin_color']['blue']);
        $this->objpdf->setHeaderTextColor($this->book['bdy_color']);

        // set language items
        $language = array(
            'a_meta_charset'=>  'UTF-8',    // PAGE META DESCRIPTORS
            'a_meta_dir'=>      'ltr',
            'a_meta_language'=> 'jp',
            'w_page'=>          'page',     // TRANSLATIONS
        );
        $this->objpdf->setLanguageArray($language);

        //initialize document
        $this->objpdf->AliasNbPages();
    }

    /**
     * 色を配列に変換
     *
     * @access private
     * @param string $color
     * @return array
     */
    function rgb2array($color)
    {        
        return array (
            'red'=>   hexdec(substr($color, 0, 2)),
            'green'=> hexdec(substr($color, 2, 2)),
            'blue'=>  hexdec(substr($color, 4, 2)),
        );
    }

    /**
     * 以下のリクエスエラーの緊急避難
     *
     * Requested: 」 U-12301 Displayed: 《 U-12298
     * Requested: 名 U-21517 Displayed: 吊 U-21514
     * Requested: 對 U-23565 Displayed: 尊 U-23562
     * Requested: 服 U-26381 Displayed: ? U-26378
     * 
     * @access private
     * @param string $string
     * @return string
     */
    function srt2mycmap($string)
    {
        $string = mb_ereg_replace ('」','',$string);
        $string = mb_ereg_replace ('名','',$string);
        $string = mb_ereg_replace ('對','',$string);

        return $string;
    }

    /**
     * 記事リストの絞込み
     *
     * @access private
     * @return string   絞込み句
     **/
    function set_condition()
    {
        // 絞込み条件の初期化
        $where = array();

        /* 範囲 開始 */
        $timstamp = strtotime($this->book['start_date']);
        $where[] = "ent.regist_date >= '".date('Y-m-d H:i:s', $timstamp)."'";

        /* 範囲 終了 */
        $timstamp = strtotime($this->book['end_date']);
        $where[] = "ent.regist_date <= '".date('Y-m-d 23:59:59', $timstamp)."'";

        /* カテゴリ */
        $sql=
            'SELECT '.
                'ref_category '.
            'FROM '.
                DB_PREFIX.'dia_bookselect '.
            'WHERE '.
                'ref_book = '.$this->book['id'];
        $stmt = $this->objdb->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetchAll(PDO_FETCH_ASSOC);
        if (!empty($result)){
            $category = array();
            foreach ($result as $value){
                $category[] = $value['ref_category'];
            }
            $this->table = DB_PREFIX.'dia_select sel,';
            $where[] = 'ent.id = sel.ref_entry';
            $where[] = 'sel.ref_category in('.implode(',',$category).')';
        }

        $this->where = 'AND '. implode(' AND ',$where) .' ';
    }

    /**
     * 画像の描画
     *
     * @access private
     * @param array $values
     * @return boolean
     */
    function draw_image($values)
    {
        $this->objblb->load($values['filename']);

        /* ファイルの種類が画像か判定 */
        $mime_type = $this->objblb->getMimeType();
        $format    = $this->objblb->getExtension();

        if (!preg_match('/image/i', $mime_type)){
            return false;
        }

        $this->objimg->setBinary($this->objblb->getBuffer(), $format);
        if (preg_match('/gif/i', $mime_type)){
            $format = 'png';
            $this->objimg->convert('png');
        }

		$tempnam = tempnam(CACHE_IMG_DIR,'blob');
        $this->objimg->save($tempnam);

		$this->objpdf->Image($tempnam,
                             $this->objpdf->GetX(),
                             $this->objpdf->GetY(),
                             $this->objpdf->pixelsToMillimeters($values['width']), 
                             $this->objpdf->pixelsToMillimeters($values['height']),
                             $format);
		unlink($tempnam);

		$this->objpdf->SetX($this->objpdf->img_rb_x + 3.52);
        $this->image_height = ($this->image_height < $this->objpdf->img_rb_y)?
                                    $this->objpdf->img_rb_y:$this->image_height;
        return true;
    }

    /**
     * 添付画像の初期設定
     *
     * @access private
     * @param integer $ref_entry
     * @return float
     */
    function set_picture_init($ref_entry)
    {
        $sql=
            'SELECT '.
                'count(id) as ct,'.
                'sum(t_width) as width '.
            'FROM '.
                DB_PREFIX.'dia_picture '.
            'WHERE '.
                'ref_entry = '.$ref_entry.' '.
            'LIMIT 3';
        $stmt = $this->objdb->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetch(PDO_FETCH_ASSOC);
        $stmt->closeCursor();
        if (empty($result['ct'])){
            return false;
        }

        // ページ幅 180mm 510px
        // マージン 3.52mm  10px        
        $page_width = PDF_BODY_WIDTH_PIXELS;
        $margin = IMAGE_MARGIN_PIXELS * ($result['ct'] - 1);
        $width = $page_width - $margin;

        $scale = round($width / $result['width'],2);
        if ($scale > IMAGE_MAX_SCALE_RATIO){
            $scale = IMAGE_MAX_SCALE_RATIO;
        }
        
        $width = $result['width'] * $scale + $margin;
        $margin_left = intval(($page_width - $width) / 2);

        $x = $this->objpdf->GetX();
        $x += $this->objpdf->pixelsToMillimeters($margin_left);

        $this->objpdf->SetX($x);

        return $scale;
    }

    /**
     * 添付画像の取得
     *
     * @access private
     * @param integer $ref_entry
     * @return void
     */
    function set_picture($ref_entry)
    {
        $scale = $this->set_picture_init($ref_entry);
        if ($scale === false){
            return false;
        }

        $sql=
            'SELECT '.
                'id,'.
                'filename,'.
                't_width as width,'.
                't_height as height '.
            'FROM '.
                DB_PREFIX.'dia_picture '.
            'WHERE '.
                'ref_entry = '.$ref_entry.' '.
            'LIMIT '.
                '3';
        $stmt = $this->objdb->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetchAll(PDO_FETCH_ASSOC);
        
        foreach ($result as $values){
            $values['width'] *= $scale;
            $values['height'] *= $scale;
            $this->draw_image($values);
        }
        if (0 < $this->image_height){
            $this->objpdf->SetY($this->image_height);
        }

        $this->image_height = 0;
        return true;
    }

    /**
     * ページの作成
     *
     * @access private
     * @return boolean
     */
    function create_page()
    {
        $date_format = sql_query::dateFormat('ent.regist_date','%Y年%m月%d日');
        $sql=
            'SELECT '.
                'ent.id as id,'.
                $date_format.' as regist_date,'.
                'ent.subject as subject,'.
                'ent.body as body '.
            'FROM '.
                $this->table.
                DB_PREFIX.'dia_entry ent '.
            'WHERE '.
                'ent.ref_account = '.$this->book['ref_account'].' '.
                $this->where.
            'ORDER BY '.
                'ent.regist_date';
        $stmt = $this->objdb->prepare($sql); 
        $stmt->execute(); 
        $result = $stmt->fetchAll(PDO_FETCH_ASSOC);
        if (empty($result)){
            return false;
        }

        $number = count($result) + 1;
        $number = 50/$number;
        $row = 0;
        $oldstep = 0;

        foreach ($result as $values){
            $values['subject'] = $this->srt2mycmap($values['subject']);
            $values['body']    = $this->srt2mycmap($values['body']);

            $this->objpdf->AddPage();

            $this->objpdf->SetTextColor(
                        $this->book['sbj_color']['red'],
                        $this->book['sbj_color']['green'],
                        $this->book['sbj_color']['blue']);
            $this->objpdf->SetFont($this->book['font'], 'B', 14);
    		$this->objpdf->MultiCell(0, PDF_FONT_SIZE_DATA, '《'.$values['subject'].'》', 0, 'C', 0);

            $this->objpdf->SetFont($this->book['font'], '', PDF_FONT_SIZE_MAIN);
    		$this->objpdf->MultiCell(0, PDF_FONT_SIZE_DATA, $values['regist_date'], 0, 'R', 0);

            $this->objpdf->SetTextColor(
                        $this->book['bdy_color']['red'],
                        $this->book['bdy_color']['green'],
                        $this->book['bdy_color']['blue']);
            $this->set_picture($values['id']);
            $this->objpdf->Write(PDF_FONT_SIZE_DATA, $values['body'], '', 0);

            $row++;
            $progress_id = intval($row * $number);
            $percent = $progress_id * 2;

            if ($oldstep != $step){
                flush();
                ob_flush();
                echo '<script type="text/javascript">setProgress('.$percent.','.$progress_id.');</script>'."\r\n";
                echo '<script type="text/javascript">setEcho("'.$values['subject'].'");</script>'."\r\n";
                $oldstep = $step;
            }
        } // foreach ($result as $values)
        flush();
        ob_flush();
        echo '<script type="text/javascript">setEcho("全体タイトル");</script>'."\r\n";

        return true;
    }

    /**
     * 更新処理
     *
     * @access private
     * @param array $values
     * @return boolean
     */
    function created_book()
    {
        $sql = 
            'UPDATE '.
                DB_PREFIX.'dia_book '.
            'SET '.
                'created = 0 '.
            'WHERE '.
                'ref_account = '.$this->book['ref_account'];
        $this->objdb->exec($sql);

        $sql = 
            'UPDATE '.
                DB_PREFIX.'dia_book '.
            'SET '.
                'created = 1 '.
            'WHERE '.
                'id = '.$this->id;
        return $this->objdb->exec($sql);
    }

} // dia_book
?>
