
package jp.sourceforge.torkenizer.test;

import jp.sourceforge.tokenizer.EofToken;
import jp.sourceforge.tokenizer.Token;
import jp.sourceforge.tokenizer.TokenInfo;
import jp.sourceforge.tokenizer.Tokenizer;
import jp.sourceforge.tokenizer.TokenizerException;
import junit.framework.TestCase;

/**
 * 字句解析のテストを行います。
 * 
 * @author uguu@users.sourceforge.jp
 */
public class TokenizerTest extends TestCase {

    /**
     * インスタンスを初期化します。
     * 
     * @param name
     *            テストの名前。
     */
    public TokenizerTest(String name) {
        super(name);
    }

    /**
     * 正常系テスト1。
     */
    public void testNormal1() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new SkipTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[0];

        Tokenizer tokenizer = new Tokenizer(text);

        assertNull(tokenizer.current());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        Token t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("123.456", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof PlusToken);
        assertEquals("+", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof LeftBracketToken);
        assertEquals("(", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("789.012", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof MinusToken);
        assertEquals("-", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("345.678", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof RightBracketToken);
        assertEquals(")", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof MinusToken);
        assertEquals("-", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof SkipToken);
        assertEquals(" ", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("012.345", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertFalse(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof EofToken);
        assertEquals(null, t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());
    }

    /**
     * 正常系テスト2。
     */
    public void testNormal2() {
        String text = "123.456   +     (789.012-345.678    ) - 012.345        ";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { new SkipTokenInfo(), };

        Tokenizer tokenizer = new Tokenizer(text);

        assertNull(tokenizer.current());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        Token t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("123.456", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof PlusToken);
        assertEquals("+", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof LeftBracketToken);
        assertEquals("(", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("789.012", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof MinusToken);
        assertEquals("-", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("345.678", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof RightBracketToken);
        assertEquals(")", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof MinusToken);
        assertEquals("-", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("012.345", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertFalse(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof EofToken);
        assertEquals(null, t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());
    }

    /**
     * 正常系テスト3。
     */
    public void testNormal3() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { new SkipTokenInfo(), };

        Tokenizer tokenizer = new Tokenizer(text);

        assertNull(tokenizer.current());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        Token t = tokenizer.current();
        assertTrue(t instanceof NumberToken);
        assertEquals("123.456", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof PlusToken);
        assertEquals("+", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());

        assertFalse(tokenizer.read(readTokenInfos, skipTokenInfos));
        t = tokenizer.current();
        assertTrue(t instanceof PlusToken);
        assertEquals("+", t.getToken());
        assertEquals(0, t.getLine());
        assertEquals(0, t.getColumn());
    }

    /**
     * 異常系テスト1。コンストラクタでtextがnullの場合、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testIllegal1() {
        try {
            new Tokenizer(null);
            fail();
        } catch (NullPointerException e) {
            assertEquals("textがnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト2。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}でreadTokenInfosがnullの場合、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testIllegal2() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = null;
        TokenInfo[] skipTokenInfos = new TokenInfo[0];

        Tokenizer tokenizer = new Tokenizer(text);
        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (NullPointerException e) {
            assertEquals("readTokenInfosがnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト3。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}でskipTokenInfosがnullの場合、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testIllegal3() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = null;

        Tokenizer tokenizer = new Tokenizer(text);
        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (NullPointerException e) {
            assertEquals("skipTokenInfosがnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト4。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}でreadTokenInfos[2]がnullの場合、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testIllegal4() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new PlusTokenInfo(), null, new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { new SkipTokenInfo(), };

        Tokenizer tokenizer = new Tokenizer(text);
        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (NullPointerException e) {
            assertEquals("readTokenInfos[2]がnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト5。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}でskipTokenInfos[0]がnullの場合、{@link NullPointerException}例外がスローされることを確認します。
     */
    public void testIllegal5() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { null, };

        Tokenizer tokenizer = new Tokenizer(text);
        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (NullPointerException e) {
            assertEquals("skipTokenInfos[0]がnullです。", e.getMessage());
        }
    }

    /**
     * 異常系テスト6。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}でreadTokenInfosの要素数が0の場合、{@link IllegalArgumentException}例外がスローされることを確認します。
     */
    public void testIllegal6() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[0];
        TokenInfo[] skipTokenInfos = new TokenInfo[0];

        Tokenizer tokenizer = new Tokenizer(text);
        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (IllegalArgumentException e) {
            assertEquals("readTokenInfosの要素数が0です。", e.getMessage());
        }
    }

    /**
     * 異常系テスト7。{@link Tokenizer#read(TokenInfo[], TokenInfo[])}で複数のトークンに一致した場合、{@link TokenizerException}例外がスローされることを確認します。
     */
    public void testIllegal7() {
        String text = "123.456 + (789.012 - 345.678) - 012.345";
        TokenInfo[] readTokenInfos = new TokenInfo[] { new NumberTokenInfo(), new SkipTokenInfo(), new PlusTokenInfo(), new MinusTokenInfo(), new LeftBracketTokenInfo(), new RightBracketTokenInfo(), };
        TokenInfo[] skipTokenInfos = new TokenInfo[] { new SkipTokenInfo(), };

        Tokenizer tokenizer = new Tokenizer(text);

        assertTrue(tokenizer.read(readTokenInfos, skipTokenInfos));

        try {
            tokenizer.read(readTokenInfos, skipTokenInfos);
            fail();
        } catch (TokenizerException e) {
            assertEquals("0行目、0列目で複数のトークンに一致しました。一致したトークンの正規表現は次の通りです。{ ,  }", e.getMessage());
            assertEquals(0, e.getLine());
            assertEquals(0, e.getColumn());
        }
    }
}
