/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.rpn_computer;

/**
 * <p>
 * 数式を解析し、逆ポーランド記法に変換し、計算を行います。数式は、実際はコマンドに分解され、それをスタックマシンとして実行します。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class RpnComputer {

    /**
     * <p>
     * 逆ポーランド記法の順序で並んでいるコマンドの配列を解釈し、計算を行います。
     * </p>
     * 
     * @param commandList
     *            計算内容を表すコマンドの配列。nullの場合、{@link NullPointerException}例外をスローします。
     * @return 計算結果の値。
     */
    public double compute(RpnCommandList commandList) {
        // 引数をチェックします。
        if (commandList == null) {
            throw new NullPointerException("commandListがnullです。");
        }
        // 計算を行います。
        RpnContext ctx = new RpnContext();
        RpnCommand[] commands = commandList.getCommands();
        for (int i = 0; i < commands.length; i++) {
            commands[i].execute(ctx);
        }
        // 結果を返します。
        if (ctx.sizeStack() == 0) {
            throw new ComputeException("計算終了後にスタックに結果の値が追加されていません。");
        }
        if (ctx.sizeStack() > 1) {
            throw new ComputeException("計算終了後にスタックに追加されている数が2つ以上あります。");
        }
        Double resultValue = ctx.popStack();
        return resultValue.doubleValue();
    }

    /**
     * <p>
     * 通常の数式を計算します。
     * </p>
     * 
     * @param expression
     *            通常の数式。
     * @return 計算結果の値。
     */
    public double compute(String expression) {
        RpnParser parser = new RpnParser();
        RpnNode node = parser.parse(expression);

        RpnCompiler compiler = new RpnCompiler();
        RpnCommandList cl = compiler.compile(node);

        return this.compute(cl);
    }
}
