/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.tokenizer;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;

/**
 * <p>
 * 字句解析を行います。入力文字列を解析して、トークンに分解します。
 * </p>
 * <p>
 * 分解するトークンは、{@link TokenInfo}インターフェイスを実装したクラスを作成し、そのインスタンスをコンストラクタで与えてください。 {@link Tokenizer}クラスは、{@link TokenInfo#getTokenPattern()}メソッドが返す正規表現を使用して文字列を解析し、合致する文字列をトークンとして切り出します。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class Tokenizer {

    private String text;

    private Token  current;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param text
     *            解析対象の文字列。nullの場合、{@link NullPointerException}例外をスローします。
     */
    public Tokenizer(String text) {
        // 引数をチェックします。
        if (text == null) {
            throw new NullPointerException("textがnullです。");
        }
        // 初期化します。
        this.text = text;
    }

    /**
     * <p>
     * 現在読み込んでいるトークンを返します。初期化直後のトークンが読み込まれていない状態では、nullを返します。文字列の終端まで解析が完了している場合は{@link EofToken}インスタンスを返します。
     * </p>
     * 
     * @return 現在読み込んでいるトークン、null、または{@link EofToken}インスタンス。
     */
    public Token current() {
        return this.current;
    }

    /**
     * <p>
     * 文字列を解析し、トークンを読み込みます。読み込むトークンはreadTokenInfos引数で指定します。skipTokenInfos引数で指定したトークンは読み込まれず、スキップされます。複数のトークンに一致した場合、{@link TokenizerException}例外をスローします。
     * </p>
     * 
     * @param readTokenInfos
     *            読み込むトークンの配列。nullの場合、配列にnullが含まれる場合、{@link NullPointerException}例外をスローします。配列の要素が0の場合、{@link IllegalArgumentException}例外をスローします。
     * @param skipTokenInfos
     *            スキップするトークンの配列。nullの場合、配列にnullが含まれる場合、{@link NullPointerException}例外をスローします。
     * @return トークンを読み込んだ場合はtrue、トークンの正規表現に一致しなかった場合、文字列の終端まで解析した場合、false。
     */
    public boolean read(TokenInfo[] readTokenInfos, TokenInfo[] skipTokenInfos) {
        // 引数をチェックします。
        if (readTokenInfos == null) {
            throw new NullPointerException("readTokenInfosがnullです。");
        }
        for (int i = 0; i < readTokenInfos.length; i++) {
            if (readTokenInfos[i] == null) {
                throw new NullPointerException("readTokenInfos[" + i + "]がnullです。");
            }
        }
        if (readTokenInfos.length == 0) {
            throw new IllegalArgumentException("readTokenInfosの要素数が0です。");
        }
        if (skipTokenInfos == null) {
            throw new NullPointerException("skipTokenInfosがnullです。");
        }
        for (int i = 0; i < skipTokenInfos.length; i++) {
            if (skipTokenInfos[i] == null) {
                throw new NullPointerException("skipTokenInfos[" + i + "]がnullです。");
            }
        }
        // 状態をチェックします。
        if (this.current instanceof EofToken) {
            return false;
        }
        if (this.text.length() == 0) {
            this.current = new EofToken(0, 0); // TODO: 行、列番号を設定する。
            return false;
        }
        // トークンを読み込みます。
        List hitReadTokenList = new ArrayList();
        String hitText = null;
        for (int i = 0; i < readTokenInfos.length; i++) {
            Matcher m = readTokenInfos[i].getTokenPattern().matcher(this.text);
            if (m.find() && m.start() == 0) {
                hitReadTokenList.add(readTokenInfos[i]);
                hitText = m.group();
            }
        }
        List hitSkipTokenList = new ArrayList();
        for (int i = 0; i < skipTokenInfos.length; i++) {
            Matcher m = skipTokenInfos[i].getTokenPattern().matcher(this.text);
            if (m.find() && m.start() == 0) {
                hitSkipTokenList.add(skipTokenInfos[i]);
                hitText = m.group();
            }
        }
        if (hitReadTokenList.size() == 0 && hitSkipTokenList.size() == 0) {
            return false;
        } else if ((hitReadTokenList.size() + hitSkipTokenList.size()) > 1) {
            hitReadTokenList.addAll(hitSkipTokenList);
            TokenInfo[] tokens = (TokenInfo[]) hitReadTokenList.toArray(new TokenInfo[0]);
            throw new TokenizerException(0, 0, tokens); // TODO: 行、列番号を設定する。
        } else if (hitReadTokenList.size() == 1) {
            TokenInfo tokenInfo = (TokenInfo) hitReadTokenList.get(0);
            this.current = tokenInfo.createToken(hitText, 0, 0); // TODO: 行、列番号を設定する。
            this.text = this.text.substring(hitText.length());
            return true;
        } else {
            this.text = this.text.substring(hitText.length());
            return this.read(readTokenInfos, skipTokenInfos);
        }
    }

}
