/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.tokenizer;

/**
 * <p>
 * 字句解析に失敗したことを表す例外です。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class TokenizerException extends RuntimeException {

    private int line;

    private int column;

    /**
     * <p>
     * インスタンスを初期化します。複数のトークンが一致したことを表すメッセージを構築します。
     * </p>
     * 
     * @param line
     *            例外が発生したときに読み込んでいた行番号。
     * @param column
     *            例外が発生したときに読み込まんでいた列番号。
     * @param tokens
     *            一致した複数のトークンの配列。
     */
    public TokenizerException(int line, int column, TokenInfo[] tokens) {
        super(TokenizerException.buildMessage(line, column, tokens));

        this.line = line;
        this.column = column;
    }

    private static String buildMessage(int line, int column, TokenInfo[] tokens) {
        String msg = line + "行目、" + column + "列目で複数のトークンに一致しました。一致したトークンの正規表現は次の通りです。{";
        for (int i = 0; i < tokens.length; i++) {
            if (i != 0) {
                msg += ", ";
            }
            msg += tokens[i].getTokenPattern().pattern();
        }
        msg += "}";
        return msg;
    }

    /**
     * <p>
     * 例外が発生したときに読み込んでいた行番号を返します。
     * </p>
     * 
     * @return 例外が発生したときに読み込んでいた行番号。
     */
    public int getLine() {
        return this.line;
    }

    /**
     * <p>
     * 例外が発生したときに読み込んでいた列番号を返します。
     * </p>
     * 
     * @return 例外が発生したときに読み込んでいた列番号。
     */
    public int getColumn() {
        return this.column;
    }
}
