/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.node;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.expression_computer.CompileContext;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.command.ConditionalAndCommand;

/**
 * <p>
 * ConditionalAndExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class ConditionalAndExpressionNode extends AbstractNode implements OperandNode {

    private Node[] nodes;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param nodes
     *            項の配列。<br>
     *            nullの場合、配列にnullがある場合、{@link NullPointerException}例外をスローします。<br>
     *            配列の要素数が2未満の場合、{@link IllegalArgumentException}例外をスローします。
     */
    public ConditionalAndExpressionNode(Node[] nodes) {
        if (nodes == null) {
            throw new NullPointerException("nodesがnullです。");
        }
        for (int i = 0; i < nodes.length; i++) {
            if (nodes[i] == null) {
                throw new NullPointerException("nodes[" + i + "]がnullです。");
            }
        }
        if (nodes.length < 2) {
            throw new IllegalArgumentException("nodes.length < 2です。");
        }

        List nodeList = new ArrayList();
        for (int i = 0; i < nodes.length; i++) {
            if (i > 0) {
                nodeList.add(new ConditionalAndExpressionNode.OperatorNode());
            }
            nodeList.add(nodes[i]);
        }

        this.nodes = (Node[]) nodeList.toArray(new Node[0]);
    }

    /**
     * {@inheritDoc}
     */
    public void compile(CompileContext context) {
        this.nodes[0].compile(context);
        for (int i = 2; i < this.nodes.length; i += 2) {
            this.nodes[i].compile(context);
            this.nodes[i - 1].compile(context);
        }
    }

    /**
     * {@inheritDoc}
     */
    public Node[] getChildren() {
        return (Node[]) this.nodes.clone();
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + this.toChildrenString();
    }

    /**
     * <p>
     * ConditionalAndExpressionの"&"演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class OperatorNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            context.add(new ConditionalAndCommand());
        }

        /**
         * {@inheritDoc}
         */
        public Node[] getChildren() {
            return new Node[0];
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

}
