/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.node;

import jp.sourceforge.expression_computer.CompileContext;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.command.ConditionCommand;

/**
 * <p>
 * ConditionalExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class ConditionalExpressionNode extends AbstractNode implements OperandNode {

    private OperandNode                            operand1;

    private ConditionalExpressionNode.QuestionNode question;

    private OperandNode                            operand2;

    private ConditionalExpressionNode.ColonNode    colon;

    private OperandNode                            operand3;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param operand1
     *            第1項。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param operand2
     *            第2項。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param operand3
     *            第3項。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     */
    public ConditionalExpressionNode(OperandNode operand1, OperandNode operand2, OperandNode operand3) {
        if (operand1 == null) {
            throw new NullPointerException("operand1がnullです。");
        }
        if (operand2 == null) {
            throw new NullPointerException("operand2がnullです。");
        }
        if (operand3 == null) {
            throw new NullPointerException("operand3がnullです。");
        }
        this.operand1 = operand1;
        this.question = new ConditionalExpressionNode.QuestionNode();
        this.operand2 = operand2;
        this.colon = new ConditionalExpressionNode.ColonNode();
        this.operand3 = operand3;
    }

    /**
     * {@inheritDoc}
     */
    public void compile(CompileContext context) {
        this.operand1.compile(context);
        if (this.question != null) {
            this.operand2.compile(context);
            this.operand3.compile(context);
            context.add(new ConditionCommand());
        }
    }

    /**
     * {@inheritDoc}
     */
    public Node[] getChildren() {
        return new Node[] { this.operand1, this.question, this.operand2, this.colon, this.operand3 };
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + this.toChildrenString();
    }

    /**
     * <p>
     * ConditionalExpressionの演算子を表す抽象既定ノードです。
     * </p>
     * 
     * @author naono
     */
    public abstract static class OperatorNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public final Node[] getChildren() {
            return new Node[0];
        }

    }

    /**
     * <p>
     * ConditionalExpressionの"?"演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class QuestionNode extends ConditionalExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * ConditionalExpressionの":"演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class ColonNode extends ConditionalExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            throw new UnsupportedOperationException();
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

}
