/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.node;

import jp.sourceforge.expression_computer.CompileContext;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.command.EqualCommand;
import jp.sourceforge.expression_computer.command.NotEqualCommand;

/**
 * <p>
 * EqualityExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class EqualityExpressionNode extends AbstractNode implements OperandNode {

    private Node[] nodes;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param nodes
     *            式の要素。"項 (演算子 項)+"という順序で並んでいる必要があります。<br>
     *            nullの場合、配列にnullがある場合、{@link NullPointerException}例外をスローします。<br>
     *            配列の要素の並び順が間違えている場合、配列の要素数が3未満の場合、配列の要素数が奇数ではない場合、{@link IllegalArgumentException}例外をスローします。<br>
     */
    public EqualityExpressionNode(Node[] nodes) {
        if (nodes == null) {
            throw new NullPointerException("nodesがnullです。");
        }
        for (int i = 0; i < nodes.length; i++) {
            if (nodes[i] == null) {
                throw new NullPointerException("nodes[" + i + "]がnullです。");
            }
            if ((i % 2 == 0) && !(nodes[i] instanceof OperandNode)) {
                throw new IllegalArgumentException("nodes[" + i + "]が項ではありません。");
            } else if ((i % 2 == 1) && !(nodes[i] instanceof EqualityExpressionNode.OperatorNode)) {
                throw new IllegalArgumentException("nodes[" + i + "]が演算子ではありません。");
            }
        }
        if (nodes.length < 3) {
            throw new IllegalArgumentException("nodes.length < 3です。");
        }
        if ((nodes.length % 2) != 1) {
            throw new IllegalArgumentException("nodes.lengthが奇数ではありません。");
        }

        this.nodes = (Node[]) nodes.clone();
    }

    /**
     * {@inheritDoc}
     */
    public void compile(CompileContext context) {
        this.nodes[0].compile(context);
        for (int i = 2; i < this.nodes.length; i += 2) {
            this.nodes[i].compile(context);
            this.nodes[i - 1].compile(context);
        }
    }

    /**
     * {@inheritDoc}
     */
    public Node[] getChildren() {
        return (Node[]) this.nodes.clone();
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + this.toChildrenString();
    }

    /**
     * <p>
     * EqualityExpressionNodeの演算子を表す抽象基底ノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public abstract static class OperatorNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public final Node[] getChildren() {
            return new Node[0];
        }

    }

    /**
     * <p>
     * EqualityExpressionNodeの"=="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class EqualNode extends EqualityExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            context.add(new EqualCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * EqualityExpressionNodeの"!="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class NotEqualNode extends EqualityExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            context.add(new NotEqualCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

}
