
package jp.sourceforge.expression_computer.test;

import jp.sourceforge.expression_computer.Command;
import jp.sourceforge.expression_computer.CommandList;
import jp.sourceforge.expression_computer.Compiler;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.Parser;
import jp.sourceforge.expression_computer.command.AddCommand;
import jp.sourceforge.expression_computer.command.AndCommand;
import jp.sourceforge.expression_computer.command.ArithmeticRightShiftCommand;
import jp.sourceforge.expression_computer.command.BitReversingCommand;
import jp.sourceforge.expression_computer.command.ConditionCommand;
import jp.sourceforge.expression_computer.command.ConditionalAndCommand;
import jp.sourceforge.expression_computer.command.ConditionalOrCommand;
import jp.sourceforge.expression_computer.command.DivideCommand;
import jp.sourceforge.expression_computer.command.EqualCommand;
import jp.sourceforge.expression_computer.command.ExclusiveOrCommand;
import jp.sourceforge.expression_computer.command.FunctionCallCommand;
import jp.sourceforge.expression_computer.command.GetVariableCommand;
import jp.sourceforge.expression_computer.command.GreaterThanCommand;
import jp.sourceforge.expression_computer.command.GreaterThanEqualCommand;
import jp.sourceforge.expression_computer.command.InclusiveOrCommand;
import jp.sourceforge.expression_computer.command.LeftShiftCommand;
import jp.sourceforge.expression_computer.command.LessThanCommand;
import jp.sourceforge.expression_computer.command.LessThanEqualCommand;
import jp.sourceforge.expression_computer.command.LogicalRightShiftCommand;
import jp.sourceforge.expression_computer.command.MultiplyCommand;
import jp.sourceforge.expression_computer.command.NotCommand;
import jp.sourceforge.expression_computer.command.NotEqualCommand;
import jp.sourceforge.expression_computer.command.PostDecrementCommand;
import jp.sourceforge.expression_computer.command.PostIncrementCommand;
import jp.sourceforge.expression_computer.command.PreDecrementCommand;
import jp.sourceforge.expression_computer.command.PreIncrementCommand;
import jp.sourceforge.expression_computer.command.PushStackCommand;
import jp.sourceforge.expression_computer.command.SetVariableCommand;
import jp.sourceforge.expression_computer.command.SignReversingCommand;
import jp.sourceforge.expression_computer.command.SubtractCommand;
import jp.sourceforge.expression_computer.command.SurplusCommand;
import junit.framework.TestCase;

public class RpnCompilerTest extends TestCase {

    public RpnCompilerTest(String name) {
        super(name);
    }

    public void testFail_null1() {
        Compiler compiler = new Compiler();
        try {
            compiler.compile(null);
            fail();
        } catch (NullPointerException e) {
            assertEquals("nodeがnullです。", e.getMessage());
        }
    }

    public void testNormal_IntegerLiteral1() {
        String exp = "1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_IntegerLiteral2() {
        String exp = "123456789";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123456789, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_IntegerLiteral3() {
        String exp = "0xfFff";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(65535, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_IntegerLiteral4() {
        String exp = "07777";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(4095, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_IntegerLiteral5() {
        String exp = "0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FloatingPointLiteral1() {
        String exp = "123.456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123.456, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FloatingPointLiteral2() {
        String exp = "0.123456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0.123456, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FloatingPointLiteral3() {
        String exp = ".123456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0.123456, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FloatingPointLiteral4() {
        String exp = "0.0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FloatingPointLiteral5() {
        String exp = ".0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_FunctionExpression1() {
        String exp = "foo()";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof FunctionCallCommand);
        assertEquals("foo", ((FunctionCallCommand) c).getName());
        assertEquals(0, ((FunctionCallCommand) c).getArgumentNumber());
    }

    public void testNormal_FunctionExpression2() {
        String exp = "foo(1, 2)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof FunctionCallCommand);
        assertEquals("foo", ((FunctionCallCommand) c).getName());
        assertEquals(2, ((FunctionCallCommand) c).getArgumentNumber());
    }

    public void testNormal_FunctionExpression3() {
        String exp = "foo(1 + 2)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AddCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof FunctionCallCommand);
        assertEquals("foo", ((FunctionCallCommand) c).getName());
        assertEquals(1, ((FunctionCallCommand) c).getArgumentNumber());
    }

    public void testNormal_FunctionExpression4() {
        String exp = "foo() + bar()";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof FunctionCallCommand);
        assertEquals("foo", ((FunctionCallCommand) c).getName());
        assertEquals(0, ((FunctionCallCommand) c).getArgumentNumber());

        c = cl.getCommand(1);
        assertTrue(c instanceof FunctionCallCommand);
        assertEquals("bar", ((FunctionCallCommand) c).getName());
        assertEquals(0, ((FunctionCallCommand) c).getArgumentNumber());

        c = cl.getCommand(2);
        assertTrue(c instanceof AddCommand);
    }

    public void testNormal_Variable1() {
        String exp = "x";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());
    }

    public void testNormal_Variable2() {
        String exp = "abc123_";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("abc123_", ((GetVariableCommand) c).getName());
    }

    public void testNormal_Variable3() {
        String exp = "_abc";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("_abc", ((GetVariableCommand) c).getName());
    }

    public void testNormal_Variable4() {
        String exp = "ほげ";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("ほげ", ((GetVariableCommand) c).getName());
    }

    public void testNormal_ParenthesesExpression1() {
        String exp = "(1 + 2)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AddCommand);
    }

    public void testNormal_ParenthesesExpression2() {
        String exp = "(1 + (2 - 3))";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(3);
        assertTrue(c instanceof SubtractCommand);

        c = cl.getCommand(4);
        assertTrue(c instanceof AddCommand);
    }

    public void testNormal_PreIncrementExpression1() {
        String exp = "++x";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PreIncrementCommand);
        assertEquals("x", ((PreIncrementCommand) c).getName());
    }

    public void testNormal_PostIncrementExpression1() {
        String exp = "x++";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PostIncrementCommand);
        assertEquals("x", ((PostIncrementCommand) c).getName());
    }

    public void testNormal_PreDecrementExpression1() {
        String exp = "--x";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PreDecrementCommand);
        assertEquals("x", ((PreDecrementCommand) c).getName());
    }

    public void testNormal_PostDecrementExpression1() {
        String exp = "x--";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PostDecrementCommand);
        assertEquals("x", ((PostDecrementCommand) c).getName());
    }

    public void testNormal_UnaryExpressionNotPlusMinus1() {
        String exp = "~1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(2, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof BitReversingCommand);
    }

    public void testNormal_UnaryExpressionNotPlusMinus2() {
        String exp = "~(~1)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof BitReversingCommand);

        c = cl.getCommand(2);
        assertTrue(c instanceof BitReversingCommand);
    }

    public void testNormal_UnaryExpressionNotPlusMinus3() {
        String exp = "!0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(2, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof NotCommand);
    }

    public void testNormal_UnaryExpressionNotPlusMinus4() {
        String exp = "!(!9)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(9, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof NotCommand);

        c = cl.getCommand(2);
        assertTrue(c instanceof NotCommand);
    }

    public void testNormal_UnaryExpression1() {
        String exp = "+0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_UnaryExpression2() {
        String exp = "+(+999)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(1, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(999, ((PushStackCommand) c).getValue(), 0);
    }

    public void testNormal_UnaryExpression3() {
        String exp = "-0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(2, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof SignReversingCommand);
    }

    public void testNormal_UnaryExpression4() {
        String exp = "-(-999)";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(999, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof SignReversingCommand);

        c = cl.getCommand(2);
        assertTrue(c instanceof SignReversingCommand);
    }

    public void testNormal_MultiplicativeExpression1() {
        String exp = "2 * 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof MultiplyCommand);
    }

    public void testNormal_MultiplicativeExpression2() {
        String exp = "4 / 2";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(4, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof DivideCommand);
    }

    public void testNormal_MultiplicativeExpression3() {
        String exp = "10 % 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(10, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof SurplusCommand);
    }

    public void testNormal_MultiplicativeExpression4() {
        String exp = "2 * 6 / 3 % 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(7, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(6, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof MultiplyCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof DivideCommand);

        c = cl.getCommand(5);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(6);
        assertTrue(c instanceof SurplusCommand);
    }

    public void testNormal_AdditiveExpression1() {
        String exp = "1 + 2";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AddCommand);
    }

    public void testNormal_AdditiveExpression2() {
        String exp = "1 - 2";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof SubtractCommand);
    }

    public void testNormal_AdditiveExpression3() {
        String exp = "1 - 2 + 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof SubtractCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof AddCommand);
    }

    public void testNormal_ShiftExpression1() {
        String exp = "0 << 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LeftShiftCommand);
    }

    public void testNormal_ShiftExpression2() {
        String exp = "0 >> 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ArithmeticRightShiftCommand);
    }

    public void testNormal_ShiftExpression3() {
        String exp = "0 >>> 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LogicalRightShiftCommand);
    }

    public void testNormal_ShiftExpression4() {
        String exp = "1 << 2 << 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LeftShiftCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof LeftShiftCommand);
    }

    public void testNormal_RelationalExpression1() {
        String exp = "123 > 122";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(122, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof GreaterThanCommand);
    }

    public void testNormal_RelationalExpression2() {
        String exp = "122 < 123";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(122, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LessThanCommand);
    }

    public void testNormal_RelationalExpression3() {
        String exp = "123 >= 122";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(122, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof GreaterThanEqualCommand);
    }

    public void testNormal_RelationalExpression4() {
        String exp = "122 <= 123";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(122, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LessThanEqualCommand);
    }

    public void testNormal_RelationalExpression5() {
        String exp = "0 > 0 > 0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof GreaterThanCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof GreaterThanCommand);
    }

    public void testNormal_EqualityExpression1() {
        String exp = "123.456 == 123.456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123.456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123.456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof EqualCommand);
    }

    public void testNormal_EqualityExpression2() {
        String exp = "123.456 != 123.456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123.456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123.456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof NotEqualCommand);
    }

    public void testNormal_EqualityExpression3() {
        String exp = "0 == 0 == 0";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof EqualCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof EqualCommand);
    }

    public void testNormal_AndExpression1() {
        String exp = "123 & 456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AndCommand);
    }

    public void testNormal_AndExpression2() {
        String exp = "123 & 456 & 789";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AndCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(789, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof AndCommand);
    }

    public void testNormal_ExclusiveOrExpression1() {
        String exp = "123 ^ 456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ExclusiveOrCommand);
    }

    public void testNormal_ExclusiveOrExpression2() {
        String exp = "123 ^ 456 ^ 789";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ExclusiveOrCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(789, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof ExclusiveOrCommand);
    }

    public void testNormal_InclusiveOrExpression1() {
        String exp = "123 | 456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof InclusiveOrCommand);
    }

    public void testNormal_InclusiveOrExpression2() {
        String exp = "123 | 456 | 789";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(5, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof InclusiveOrCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(789, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof InclusiveOrCommand);
    }

    public void testNormal_ConditionalAndExpression1() {
        String exp = "0 && 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ConditionalAndCommand);
    }

    public void testNormal_ConditionalOrExpression1() {
        String exp = "0 || 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ConditionalOrCommand);
    }

    public void testNormal_ConditionalExpression1() {
        String exp = "0 ? 123 : 456";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(456, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(3);
        assertTrue(c instanceof ConditionCommand);
    }

    public void testNormal_ConditionalExpression2() {
        String exp = "0 ? 1 : 2 ? 3 : 4";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(7, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(0, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(3);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(4);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(4, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(5);
        assertTrue(c instanceof ConditionCommand);

        c = cl.getCommand(6);
        assertTrue(c instanceof ConditionCommand);
    }

    public void testNormal_AssignmentExpression1() {
        String exp = "x = 123";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(2, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(123, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression2() {
        String exp = "x += 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AddCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression3() {
        String exp = "x -= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof SubtractCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression4() {
        String exp = "x *= 2";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(2, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof MultiplyCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression5() {
        String exp = "x /= 3";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(3, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof DivideCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression6() {
        String exp = "x %= 4";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(4, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof SurplusCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression7() {
        String exp = "x &= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof AndCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression8() {
        String exp = "x ^= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ExclusiveOrCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression9() {
        String exp = "x |= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof InclusiveOrCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression10() {
        String exp = "x <<= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LeftShiftCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression11() {
        String exp = "x >>= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof ArithmeticRightShiftCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression12() {
        String exp = "x >>>= 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(4, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof GetVariableCommand);
        assertEquals("x", ((GetVariableCommand) c).getName());

        c = cl.getCommand(1);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(2);
        assertTrue(c instanceof LogicalRightShiftCommand);

        c = cl.getCommand(3);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

    public void testNormal_AssignmentExpression13() {
        String exp = "x = y = 1";

        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        assertEquals(3, cl.size());

        Command c = cl.getCommand(0);
        assertTrue(c instanceof PushStackCommand);
        assertEquals(1, ((PushStackCommand) c).getValue(), 0);

        c = cl.getCommand(1);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("y", ((SetVariableCommand) c).getName());

        c = cl.getCommand(2);
        assertTrue(c instanceof SetVariableCommand);
        assertEquals("x", ((SetVariableCommand) c).getName());
    }

}
