/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.expression_computer.javacc.ASTAdditiveExpression;
import jp.sourceforge.expression_computer.javacc.ASTAdditiveExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTAndExpression;
import jp.sourceforge.expression_computer.javacc.ASTAssignmentExpression;
import jp.sourceforge.expression_computer.javacc.ASTAssignmentExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTConditionalAndExpression;
import jp.sourceforge.expression_computer.javacc.ASTConditionalExpression;
import jp.sourceforge.expression_computer.javacc.ASTConditionalOrExpression;
import jp.sourceforge.expression_computer.javacc.ASTEqualityExpression;
import jp.sourceforge.expression_computer.javacc.ASTEqualityExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTExclusiveOrExpression;
import jp.sourceforge.expression_computer.javacc.ASTExpression;
import jp.sourceforge.expression_computer.javacc.ASTFloatingPointLiteral;
import jp.sourceforge.expression_computer.javacc.ASTFunctionExpression;
import jp.sourceforge.expression_computer.javacc.ASTFunctionName;
import jp.sourceforge.expression_computer.javacc.ASTInclusiveOrExpression;
import jp.sourceforge.expression_computer.javacc.ASTIntegerLiteral;
import jp.sourceforge.expression_computer.javacc.ASTMultiplicativeExpression;
import jp.sourceforge.expression_computer.javacc.ASTMultiplicativeExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTParenthesesExpression;
import jp.sourceforge.expression_computer.javacc.ASTPostDecrementExpression;
import jp.sourceforge.expression_computer.javacc.ASTPostIncrementExpression;
import jp.sourceforge.expression_computer.javacc.ASTPreDecrementExpression;
import jp.sourceforge.expression_computer.javacc.ASTPreIncrementExpression;
import jp.sourceforge.expression_computer.javacc.ASTRelationalExpression;
import jp.sourceforge.expression_computer.javacc.ASTRelationalExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTShiftExpression;
import jp.sourceforge.expression_computer.javacc.ASTShiftExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTUnaryExpression;
import jp.sourceforge.expression_computer.javacc.ASTUnaryExpressionNotPlusMinus;
import jp.sourceforge.expression_computer.javacc.ASTUnaryExpressionNotPlusMinusOperator;
import jp.sourceforge.expression_computer.javacc.ASTUnaryExpressionOperator;
import jp.sourceforge.expression_computer.javacc.ASTVariable;
import jp.sourceforge.expression_computer.javacc.JavaCCParser;
import jp.sourceforge.expression_computer.javacc.SimpleNode;
import jp.sourceforge.expression_computer.node.AdditiveExpressionNode;
import jp.sourceforge.expression_computer.node.AndExpressionNode;
import jp.sourceforge.expression_computer.node.AssignmentExpressionNode;
import jp.sourceforge.expression_computer.node.BracketExpressionNode;
import jp.sourceforge.expression_computer.node.ConditionalAndExpressionNode;
import jp.sourceforge.expression_computer.node.ConditionalExpressionNode;
import jp.sourceforge.expression_computer.node.ConditionalOrExpressionNode;
import jp.sourceforge.expression_computer.node.EqualityExpressionNode;
import jp.sourceforge.expression_computer.node.ExclusiveOrExpressionNode;
import jp.sourceforge.expression_computer.node.ExpressionStatementNode;
import jp.sourceforge.expression_computer.node.FloatingPointLiteralNode;
import jp.sourceforge.expression_computer.node.FunctionExpressionNode;
import jp.sourceforge.expression_computer.node.InclusiveOrExpressionNode;
import jp.sourceforge.expression_computer.node.IntegerLiteralNode;
import jp.sourceforge.expression_computer.node.MultiplicativeExpressionNode;
import jp.sourceforge.expression_computer.node.OperandNode;
import jp.sourceforge.expression_computer.node.PostDecrementExpressionNode;
import jp.sourceforge.expression_computer.node.PostIncrementExpressionNode;
import jp.sourceforge.expression_computer.node.PreDecrementExpressionNode;
import jp.sourceforge.expression_computer.node.PreIncrementExpressionNode;
import jp.sourceforge.expression_computer.node.RelationalExpressionNode;
import jp.sourceforge.expression_computer.node.ShiftExpressionNode;
import jp.sourceforge.expression_computer.node.UnaryExpressionNode;
import jp.sourceforge.expression_computer.node.UnaryExpressionNotPlusMinusNode;
import jp.sourceforge.expression_computer.node.VariableNode;

/**
 * <p>
 * 通常の数式を解析し、抽象構文木を構築します。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class Parser {

    /**
     * <p>
     * 通常の数式を解析し、抽象構文木を構築します。
     * </p>
     * 
     * @param expression
     *            通常の数式。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。<br>
     *            構文解析に失敗した場合、{@link ParseException}例外をスローします。
     * @return 解析した結果の抽象構文木。
     */
    public Node parse(String expression) {
        if (expression == null) {
            throw new NullPointerException("expressionがnullです。");
        }

        JavaCCParser javaCCParser = new JavaCCParser(new StringReader(expression));
        ASTExpression astExpressionNode;
        try {
            astExpressionNode = (ASTExpression) javaCCParser.parse();
        } catch (jp.sourceforge.expression_computer.javacc.ParseException e) {
            throw new ParseException(e);
        }

        Node node = this.buildRpnNode(astExpressionNode);

        return node;
    }

    private Node buildRpnNode(jp.sourceforge.expression_computer.javacc.Node node) {
        if (node instanceof ASTExpression) {
            if (node.jjtGetNumChildren() == 0) {
                return new ExpressionStatementNode();
            } else {
                OperandNode operand = (OperandNode) this.buildRpnNode(node.jjtGetChild(0));

                return new ExpressionStatementNode(operand);
            }
        } else if (node instanceof ASTAssignmentExpression) {
            VariableNode operand1 = (VariableNode) this.buildRpnNode(node.jjtGetChild(0));
            AssignmentExpressionNode.OperatorNode operator = (AssignmentExpressionNode.OperatorNode) this.buildRpnNode(node.jjtGetChild(1));
            OperandNode operand2 = (OperandNode) this.buildRpnNode(node.jjtGetChild(2));

            return new AssignmentExpressionNode(operand1, operator, operand2);
        } else if (node instanceof ASTAssignmentExpressionOperator) {
            ASTAssignmentExpressionOperator ope = (ASTAssignmentExpressionOperator) node;
            if ("=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.EqualNode();
            } else if ("+=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.AddEqualNode();
            } else if ("-=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.SubtractEqualNode();
            } else if ("*=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.MultiplyEqualNode();
            } else if ("/=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.DivideEqualNode();
            } else if ("%=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.SurplusEqualNode();
            } else if ("&=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.AndEqualNode();
            } else if ("^=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.ExclusiveOrEqualNode();
            } else if ("|=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.InclusiveOrEqualNode();
            } else if ("<<=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.LeftShiftEqualNode();
            } else if (">>=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.ArithmeticRightShiftEqualNode();
            } else if (">>>=".equals(ope.getNodeValue())) {
                return new AssignmentExpressionNode.LogicalRightShiftEqualNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTConditionalExpression) {
            OperandNode operand1 = (OperandNode) this.buildRpnNode(node.jjtGetChild(0));
            OperandNode operand2 = (OperandNode) this.buildRpnNode(node.jjtGetChild(1));
            OperandNode operand3 = (OperandNode) this.buildRpnNode(node.jjtGetChild(2));

            return new ConditionalExpressionNode(operand1, operand2, operand3);
        } else if (node instanceof ASTConditionalOrExpression) {
            OperandNode[] operands = new OperandNode[node.jjtGetNumChildren()];
            for (int i = 0; i < operands.length; i++) {
                operands[i] = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
            }

            return new ConditionalOrExpressionNode(operands);
        } else if (node instanceof ASTConditionalAndExpression) {
            OperandNode[] operands = new OperandNode[node.jjtGetNumChildren()];
            for (int i = 0; i < operands.length; i++) {
                operands[i] = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
            }

            return new ConditionalAndExpressionNode(operands);
        } else if (node instanceof ASTInclusiveOrExpression) {
            OperandNode[] operands = new OperandNode[node.jjtGetNumChildren()];
            for (int i = 0; i < operands.length; i++) {
                operands[i] = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
            }

            return new InclusiveOrExpressionNode(operands);
        } else if (node instanceof ASTExclusiveOrExpression) {
            OperandNode[] operands = new OperandNode[node.jjtGetNumChildren()];
            for (int i = 0; i < operands.length; i++) {
                operands[i] = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
            }

            return new ExclusiveOrExpressionNode(operands);
        } else if (node instanceof ASTAndExpression) {
            OperandNode[] operands = new OperandNode[node.jjtGetNumChildren()];
            for (int i = 0; i < operands.length; i++) {
                operands[i] = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
            }

            return new AndExpressionNode(operands);
        } else if (node instanceof ASTEqualityExpression) {
            Node[] nodes = new Node[node.jjtGetNumChildren()];
            for (int i = 0; i < nodes.length; i++) {
                nodes[i] = this.buildRpnNode(node.jjtGetChild(i));
            }

            return new EqualityExpressionNode(nodes);
        } else if (node instanceof ASTEqualityExpressionOperator) {
            ASTEqualityExpressionOperator ope = (ASTEqualityExpressionOperator) node;
            if ("==".equals(ope.getNodeValue())) {
                return new EqualityExpressionNode.EqualNode();
            } else if ("!=".equals(ope.getNodeValue())) {
                return new EqualityExpressionNode.NotEqualNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTRelationalExpression) {
            Node[] nodes = new Node[node.jjtGetNumChildren()];
            for (int i = 0; i < nodes.length; i++) {
                nodes[i] = this.buildRpnNode(node.jjtGetChild(i));
            }

            return new RelationalExpressionNode(nodes);
        } else if (node instanceof ASTRelationalExpressionOperator) {
            ASTRelationalExpressionOperator ope = (ASTRelationalExpressionOperator) node;
            if (">".equals(ope.getNodeValue())) {
                return new RelationalExpressionNode.GreaterThanNode();
            } else if ("<".equals(ope.getNodeValue())) {
                return new RelationalExpressionNode.LessThanNode();
            } else if (">=".equals(ope.getNodeValue())) {
                return new RelationalExpressionNode.GreaterThanEqualNode();
            } else if ("<=".equals(ope.getNodeValue())) {
                return new RelationalExpressionNode.LessThanEqualNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTShiftExpression) {
            Node[] nodes = new Node[node.jjtGetNumChildren()];
            for (int i = 0; i < nodes.length; i++) {
                nodes[i] = this.buildRpnNode(node.jjtGetChild(i));
            }

            return new ShiftExpressionNode(nodes);
        } else if (node instanceof ASTShiftExpressionOperator) {
            ASTShiftExpressionOperator ope = (ASTShiftExpressionOperator) node;
            if ("<<".equals(ope.getNodeValue())) {
                return new ShiftExpressionNode.LeftShiftNode();
            } else if (">>".equals(ope.getNodeValue())) {
                return new ShiftExpressionNode.ArithmeticRightShiftNode();
            } else if (">>>".equals(ope.getNodeValue())) {
                return new ShiftExpressionNode.LogicalRightShiftNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTAdditiveExpression) {
            Node[] nodes = new Node[node.jjtGetNumChildren()];
            for (int i = 0; i < nodes.length; i++) {
                nodes[i] = this.buildRpnNode(node.jjtGetChild(i));
            }

            return new AdditiveExpressionNode(nodes);
        } else if (node instanceof ASTAdditiveExpressionOperator) {
            ASTAdditiveExpressionOperator ope = (ASTAdditiveExpressionOperator) node;
            if ("+".equals(ope.getNodeValue())) {
                return new AdditiveExpressionNode.AddNode();
            } else if ("-".equals(ope.getNodeValue())) {
                return new AdditiveExpressionNode.SubtractNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTMultiplicativeExpression) {
            Node[] nodes = new Node[node.jjtGetNumChildren()];
            for (int i = 0; i < nodes.length; i++) {
                nodes[i] = this.buildRpnNode(node.jjtGetChild(i));
            }

            return new MultiplicativeExpressionNode(nodes);
        } else if (node instanceof ASTMultiplicativeExpressionOperator) {
            ASTMultiplicativeExpressionOperator ope = (ASTMultiplicativeExpressionOperator) node;
            if ("*".equals(ope.getNodeValue())) {
                return new MultiplicativeExpressionNode.MultiplyNode();
            } else if ("/".equals(ope.getNodeValue())) {
                return new MultiplicativeExpressionNode.DivideNode();
            } else if ("%".equals(ope.getNodeValue())) {
                return new MultiplicativeExpressionNode.SurplusNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTUnaryExpression) {
            UnaryExpressionNode.OperatorNode operator = (UnaryExpressionNode.OperatorNode) this.buildRpnNode(node.jjtGetChild(0));
            OperandNode operand = (OperandNode) this.buildRpnNode(node.jjtGetChild(1));

            return new UnaryExpressionNode(operator, operand);
        } else if (node instanceof ASTUnaryExpressionOperator) {
            ASTUnaryExpressionOperator ope = (ASTUnaryExpressionOperator) node;
            if ("+".equals(ope.getNodeValue())) {
                return new UnaryExpressionNode.PlusSignNode();
            } else if ("-".equals(ope.getNodeValue())) {
                return new UnaryExpressionNode.MinusSignNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTPreIncrementExpression) {
            VariableNode operand = (VariableNode) this.buildRpnNode(node.jjtGetChild(0));

            return new PreIncrementExpressionNode(operand);
        } else if (node instanceof ASTPreDecrementExpression) {
            VariableNode operand = (VariableNode) this.buildRpnNode(node.jjtGetChild(0));

            return new PreDecrementExpressionNode(operand);
        } else if (node instanceof ASTUnaryExpressionNotPlusMinus) {
            UnaryExpressionNotPlusMinusNode.OperatorNode operator = (UnaryExpressionNotPlusMinusNode.OperatorNode) this.buildRpnNode(node.jjtGetChild(0));
            OperandNode operand = (OperandNode) this.buildRpnNode(node.jjtGetChild(1));

            return new UnaryExpressionNotPlusMinusNode(operator, operand);
        } else if (node instanceof ASTUnaryExpressionNotPlusMinusOperator) {
            ASTUnaryExpressionNotPlusMinusOperator ope = (ASTUnaryExpressionNotPlusMinusOperator) node;
            if ("~".equals(ope.getNodeValue())) {
                return new UnaryExpressionNotPlusMinusNode.BitReversingNode();
            } else if ("!".equals(ope.getNodeValue())) {
                return new UnaryExpressionNotPlusMinusNode.NotNode();
            } else {
                throw new ParseException(ope.getNodeValue(), ope.getNodeLine(), ope.getNodeColumn());
            }
        } else if (node instanceof ASTPostIncrementExpression) {
            VariableNode operand = (VariableNode) this.buildRpnNode(node.jjtGetChild(0));

            return new PostIncrementExpressionNode(operand);
        } else if (node instanceof ASTPostDecrementExpression) {
            VariableNode operand = (VariableNode) this.buildRpnNode(node.jjtGetChild(0));

            return new PostDecrementExpressionNode(operand);
        } else if (node instanceof ASTIntegerLiteral) {
            ASTIntegerLiteral integerLiteral = (ASTIntegerLiteral) node;

            return new IntegerLiteralNode(integerLiteral.getNodeValue());
        } else if (node instanceof ASTFloatingPointLiteral) {
            ASTFloatingPointLiteral floatingPointLiteral = (ASTFloatingPointLiteral) node;

            return new FloatingPointLiteralNode(floatingPointLiteral.getNodeValue());
        } else if (node instanceof ASTFunctionExpression) {
            FunctionExpressionNode.FunctionNameNode functionName = (FunctionExpressionNode.FunctionNameNode) this.buildRpnNode(node.jjtGetChild(0));
            List argumentList = new ArrayList();
            for (int i = 1; i < node.jjtGetNumChildren(); i++) {
                OperandNode operand = (OperandNode) this.buildRpnNode(node.jjtGetChild(i));
                argumentList.add(operand);
            }
            OperandNode[] arguments = (OperandNode[]) argumentList.toArray(new OperandNode[0]);

            return new FunctionExpressionNode(functionName, arguments);
        } else if (node instanceof ASTFunctionName) {
            ASTFunctionName functionName = (ASTFunctionName) node;

            return new FunctionExpressionNode.FunctionNameNode(functionName.getNodeValue());
        } else if (node instanceof ASTVariable) {
            ASTVariable variable = (ASTVariable) node;

            return new VariableNode(variable.getNodeValue());
        } else if (node instanceof ASTParenthesesExpression) {
            OperandNode expression = (OperandNode) this.buildRpnNode(node.jjtGetChild(0));

            return new BracketExpressionNode(expression);
        } else {
            SimpleNode simpleNode = (SimpleNode) node;
            throw new ParseException(simpleNode.getNodeValue(), simpleNode.getNodeLine(), simpleNode.getNodeColumn());
        }
    }
}
