/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer;

/**
 * <p>
 * 構文解析に失敗したことを現す例外です。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class ParseException extends RuntimeException {

    private String text;

    private int    line;

    private int    column;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param text
     *            出現した不明な文字。
     * @param line
     *            構文解析に失敗した行の番号。
     * @param column
     *            構文解析に失敗した列の番号。
     */
    public ParseException(String text, int line, int column) {
        super("不明な文字{" + text + "}が出現しました。{" + (line + 1) + "行目, " + (column + 1) + "列目}");

        this.text = text;
        this.line = line;
        this.column = column;
    }

    /**
     * <p>
     * JavaCCによる構文解析に失敗したことを表す例外を初期化します。
     * </p>
     * 
     * @param cause
     *            JavaCCの例外。
     */
    public ParseException(jp.sourceforge.expression_computer.javacc.ParseException cause) {
        super("構文解析に失敗しました。{メッセージ=" + cause.getMessage() + "}", cause);

        this.text = null;
        this.line = -1;
        this.column = -1;
    }

    /**
     * <p>
     * 出現した不明な文字を返します。
     * </p>
     * 
     * @return 出現した不明な文字
     */
    public String getText() {
        return this.text;
    }

    /**
     * <p>
     * 構文解析に失敗した行の番号を返します。
     * </p>
     * 
     * @return 構文解析に失敗した行の番号。
     */
    public int getLine() {
        return this.line;
    }

    /**
     * <p>
     * 構文解析に失敗した列の番号を返します。
     * </p>
     * 
     * @return 構文解析に失敗した列の番号。
     */
    public int getColumn() {
        return this.column;
    }

}
