/*
 * Copyright (C) 2006 uguu@users.sourceforge.jp, All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. Neither the name of Clarkware Consulting, Inc. nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without prior written permission. For written
 *       permission, please contact clarkware@clarkware.com.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CLARKWARE CONSULTING OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN  ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.expression_computer.node;

import jp.sourceforge.expression_computer.Command;
import jp.sourceforge.expression_computer.CommandList;
import jp.sourceforge.expression_computer.CompileContext;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.command.AddCommand;
import jp.sourceforge.expression_computer.command.AndCommand;
import jp.sourceforge.expression_computer.command.ArithmeticRightShiftCommand;
import jp.sourceforge.expression_computer.command.DivideCommand;
import jp.sourceforge.expression_computer.command.ExclusiveOrCommand;
import jp.sourceforge.expression_computer.command.InclusiveOrCommand;
import jp.sourceforge.expression_computer.command.LeftShiftCommand;
import jp.sourceforge.expression_computer.command.LogicalRightShiftCommand;
import jp.sourceforge.expression_computer.command.MultiplyCommand;
import jp.sourceforge.expression_computer.command.SetVariableCommand;
import jp.sourceforge.expression_computer.command.SubtractCommand;
import jp.sourceforge.expression_computer.command.SurplusCommand;

/**
 * <p>
 * AssignmentExpressionを表すノードです。
 * </p>
 * 
 * @author uguu@users.sourceforge.jp
 */
public final class AssignmentExpressionNode extends AbstractNode implements OperandNode {

    private VariableNode                          operand1;

    private AssignmentExpressionNode.OperatorNode operator;

    private OperandNode                           operand2;

    /**
     * <p>
     * インスタンスを初期化します。
     * </p>
     * 
     * @param operand1
     *            左項。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param operator
     *            演算子。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     * @param operand2
     *            右項。<br>
     *            nullの場合、{@link NullPointerException}例外をスローします。
     */
    public AssignmentExpressionNode(VariableNode operand1, AssignmentExpressionNode.OperatorNode operator, OperandNode operand2) {
        if (operand1 == null) {
            throw new NullPointerException("operand1がnullです。");
        }
        if (operator == null) {
            throw new NullPointerException("operatorがnullです。");
        }
        if (operand2 == null) {
            throw new NullPointerException("operand2がnullです。");
        }
        this.operand1 = operand1;
        this.operator = operator;
        this.operand2 = operand2;
    }

    /**
     * {@inheritDoc}
     */
    public void compile(CompileContext context) {
        this.operand1.compile(context);
        this.operand2.compile(context);
        this.operator.compile(context);
    }

    /**
     * {@inheritDoc}
     */
    public Node[] getChildren() {
        return new Node[] { this.operand1, this.operator, this.operand2 };
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return this.getClass().getName() + this.toChildrenString();
    }

    /**
     * <p>
     * AssignmentExpressionの演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public abstract static class OperatorNode extends AbstractNode {

        /**
         * {@inheritDoc}
         */
        public final Node[] getChildren() {
            return new Node[0];
        }
    }

    /**
     * <p>
     * AssignmentExpressionの"="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class EqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            context.getCommandList().add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"+="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class AddEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new AddCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"-="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class SubtractEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new SubtractCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"*="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class MultiplyEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new MultiplyCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"/="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class DivideEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new DivideCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"%="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class SurplusEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new SurplusCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"&="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class AndEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new AndCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"^="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class ExclusiveOrEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new ExclusiveOrCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"|="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class InclusiveOrEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new InclusiveOrCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"&lt;&lt;="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class LeftShiftEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new LeftShiftCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"&gt;&gt;="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class ArithmeticRightShiftEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new ArithmeticRightShiftCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

    /**
     * <p>
     * AssignmentExpressionの"&gt;&gt;&gt;="演算子を表すノードです。
     * </p>
     * 
     * @author uguu@users.sourceforge.jp
     */
    public static final class LogicalRightShiftEqualNode extends AssignmentExpressionNode.OperatorNode {

        /**
         * {@inheritDoc}
         */
        public void compile(CompileContext context) {
            CommandList cl = context.getCommandList();

            Command c = cl.getCommand(cl.size() - 2);
            cl.addCommand(cl.size() - 1, c);
            cl.add(new LogicalRightShiftCommand());
            cl.add(new SetVariableCommand());
        }

        /**
         * {@inheritDoc}
         */
        public String toString() {
            return this.getClass().getName();
        }

    }

}
