
package jp.sourceforge.expression_computer.test;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;

import jp.sourceforge.expression_computer.CommandList;
import jp.sourceforge.expression_computer.Compiler;
import jp.sourceforge.expression_computer.Node;
import jp.sourceforge.expression_computer.Parser;
import jp.sourceforge.expression_computer.Serializer;
import junit.framework.TestCase;

public class SerializerTest extends TestCase {

    public SerializerTest(String name) {
        super(name);
    }

    public void testFail_null() throws Exception {
        Serializer serializer = new Serializer();
        try {
            serializer.serialize(null);
            fail();
        } catch (NullPointerException e) {
            assertEquals("commandListがnullです。", e.getMessage());
        }
    }

    public void testNormal_IntegerLiteral1() throws Exception {
        String exp = "1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_IntegerLiteral2() throws Exception {
        String exp = "123456789";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123456789);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_IntegerLiteral3() throws Exception {
        String exp = "0xfFff";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(65535);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_IntegerLiteral4() throws Exception {
        String exp = "07777";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(4095);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_IntegerLiteral5() throws Exception {
        String exp = "0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FloatingPointLiteral6() throws Exception {
        String exp = "123.456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(123.456);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FloatingPointLiteral2() throws Exception {
        String exp = "0.123456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(0.123456);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FloatingPointLiteral3() throws Exception {
        String exp = ".123456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(0.123456);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FloatingPointLiteral4() throws Exception {
        String exp = "0.0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FloatingPointLiteral5() throws Exception {
        String exp = ".0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(0.0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FunctionExpression1() throws Exception {
        String exp = "foo()";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(10);
                objOut.writeUTF("foo");
                objOut.writeInt(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FunctionExpression2() throws Exception {
        String exp = "foo(1, 2)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(10);
                objOut.writeUTF("foo");
                objOut.writeInt(2);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FunctionExpression3() throws Exception {
        String exp = "foo(1 + 2)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(0);
                objOut.writeByte(10);
                objOut.writeUTF("foo");
                objOut.writeInt(1);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_FunctionExpression4() throws Exception {
        String exp = "foo() + bar()";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(10);
                objOut.writeUTF("foo");
                objOut.writeInt(0);
                objOut.writeByte(10);
                objOut.writeUTF("bar");
                objOut.writeInt(0);
                objOut.writeByte(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_Variable1() throws Exception {
        String exp = "x";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_Variable2() throws Exception {
        String exp = "abc123_";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("abc123_");
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_Variable3() throws Exception {
        String exp = "_abc";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("_abc");
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_Variable4() throws Exception {
        String exp = "ほげ";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("ほげ");
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ParenthesesExpression1() throws Exception {
        String exp = "(1 + 2)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ParenthesesExpression2() throws Exception {
        String exp = "(1 + (2 - 3))";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(28);
                objOut.writeByte(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_PreIncrementExpression1() throws Exception {
        String exp = "++x";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(24);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_PreDecrementExpression1() throws Exception {
        String exp = "--x";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(23);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_PostIncrementExpression1() throws Exception {
        String exp = "x++";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(22);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_PostDecrementExpression1() throws Exception {
        String exp = "x--";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(21);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpressionNotPlusMinus1() throws Exception {
        String exp = "~1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(3);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpressionNotPlusMinus2() throws Exception {
        String exp = "~(~1)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(3);
                objOut.writeByte(3);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpressionNotPlusMinus3() throws Exception {
        String exp = "!0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(19);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpressionNotPlusMinus4() throws Exception {
        String exp = "!(!9)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(9);
                objOut.writeByte(19);
                objOut.writeByte(19);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpression1() throws Exception {
        String exp = "+0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNotmal_UnaryExpression2() throws Exception {
        String exp = "+(+999)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(999);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpression3() throws Exception {
        String exp = "-0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(27);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_UnaryExpression4() throws Exception {
        String exp = "-(-999)";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(999);
                objOut.writeByte(27);
                objOut.writeByte(27);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_MultiplicativeExpression1() throws Exception {
        String exp = "2 * 3";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(18);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_MultiplicativeExpression2() throws Exception {
        String exp = "4 / 2";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(4);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(7);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_MultiplicativeExpression3() throws Exception {
        String exp = "10 % 3";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(10);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(29);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_MultiplicativeExpression4() throws Exception {
        String exp = "2 * 6 / 3 % 3";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(6);
                objOut.writeByte(18);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(7);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(29);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AdditiveExpression1() throws Exception {
        String exp = "1 + 2";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AdditiveExpression2() throws Exception {
        String exp = "1 - 2";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(28);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AdditiveExpression3() throws Exception {
        String exp = "1 - 2 + 3";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(28);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(0);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ShiftExpression1() throws Exception {
        String exp = "0 << 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(14);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ShiftExpression2() throws Exception {
        String exp = "0 >> 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(2);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ShiftExpression3() throws Exception {
        String exp = "0 >>> 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(17);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ShiftExpression4() throws Exception {
        String exp = "1 << 2 << 3";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(14);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(14);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_RelationalExpression1() throws Exception {
        String exp = "123 > 122";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(122);
                objOut.writeByte(11);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_RelationalExpression2() throws Exception {
        String exp = "122 < 123";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(122);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(15);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_RelationalExpression3() throws Exception {
        String exp = "123 >= 122";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(122);
                objOut.writeByte(12);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_RelationalExpression4() throws Exception {
        String exp = "122 <= 123";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(122);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(16);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_RelationalExpression5() throws Exception {
        String exp = "0 > 0 > 0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(11);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(11);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_EqualityExpression1() throws Exception {
        String exp = "123.456 == 123.456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(123.456);
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(123.456);
                objOut.writeByte(8);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_EqualityExpression2() throws Exception {
        String exp = "123.456 != 123.456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(123.456);
                objOut.writeByte(25);
                objOut.writeByte(0);
                objOut.writeDouble(123.456);
                objOut.writeByte(20);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_EqualityExpression3() throws Exception {
        String exp = "0 == 0 == 0";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(8);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(8);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AndExpression1() throws Exception {
        String exp = "123 & 456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(1);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AndExpression2() throws Exception {
        String exp = "123 & 456 & 789";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(789);
                objOut.writeByte(1);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ExclusiveOrExpression1() throws Exception {
        String exp = "123 ^ 456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(9);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ExclusiveOrExpression2() throws Exception {
        String exp = "123 ^ 456 ^ 789";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(9);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(789);
                objOut.writeByte(9);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_InclusiveOrExpression1() throws Exception {
        String exp = "123 | 456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(13);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_InclusiveOrExpression2() throws Exception {
        String exp = "123 | 456 | 789";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(13);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(789);
                objOut.writeByte(13);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ConditionalAndExpression1() throws Exception {
        String exp = "0 && 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(4);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ConditionalOrExpression1() throws Exception {
        String exp = "0 || 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(5);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ConditionalExpression1() throws Exception {
        String exp = "0 ? 123 : 456";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(456);
                objOut.writeByte(6);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_ConditionalExpression2() throws Exception {
        String exp = "0 ? 1 : 2 ? 3 : 4";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(0);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(2);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(3);
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(4);
                objOut.writeByte(6);
                objOut.writeByte(6);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression1() throws Exception {
        String exp = "x = 123";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(123);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression2() throws Exception {
        String exp = "x += 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(0);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression3() throws Exception {
        String exp = "x -= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(28);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression4() throws Exception {
        String exp = "x *= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(18);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression5() throws Exception {
        String exp = "x /= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(7);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression6() throws Exception {
        String exp = "x %= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(29);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression7() throws Exception {
        String exp = "x &= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(1);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression8() throws Exception {
        String exp = "x ^= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(9);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression9() throws Exception {
        String exp = "x |= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(13);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression10() throws Exception {
        String exp = "x <<= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(14);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression11() throws Exception {
        String exp = "x >>= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(2);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression12() throws Exception {
        String exp = "x >>>= 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(17);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    public void testNormal_AssignmentExpression13() throws Exception {
        String exp = "x = y = 1";
        byte[] actual = serialize(exp);

        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        try {
            ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
            try {
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("x");
                objOut.writeByte(25);
                objOut.writeByte(2);
                objOut.writeUTF("y");
                objOut.writeByte(25);
                objOut.writeByte(1);
                objOut.writeLong(1);
                objOut.writeByte(26);
                objOut.writeByte(26);
                objOut.writeByte(Byte.MAX_VALUE);
            } finally {
                objOut.close();
            }
        } finally {
            byteOut.close();
        }
        byte[] expected = byteOut.toByteArray();

        assertByteArray(expected, actual);
    }

    private byte[] serialize(String exp) throws IOException {
        Parser parser = new Parser();
        Node node = parser.parse(exp);

        Compiler compiler = new Compiler();
        CommandList cl = compiler.compile(node);

        Serializer serializer = new Serializer();
        byte[] data = serializer.serialize(cl);

        return data;
    }

    private void assertByteArray(byte[] expected, byte[] actual) {
        assertEquals("expected length=" + expected.length + ", actual length=" + actual.length, expected.length, actual.length);

        for (int i = 0; i < expected.length; i++) {
            assertEquals("expected[" + i + "]=" + expected[i] + ", actual[" + i + "]=" + actual[i], expected[i], actual[i]);
        }
    }

}
