/*
 * Frame Buffer Device Driver for VDC5
 *
 * Copyright (C) 2013-2014 Renesas Solutions Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/* For vdc5fb driver */
#include <video/vdc5fb.h>
#include <linux/gpio.h>
/* For the LCD-KIT-B01 */
#include <linux/input/ili210x.h>
/* For the R0P772LE0011RL */
#include <linux/i2c/tsc2007.h>

/*************************************************************************/

/* COMMON MACROS */
#define	P1CLK			((13330000 * 30) / 6)
#define	PIXCLOCK(hz, div)	\
	(u32)(1000000000000 / ((double)(hz) / (double)(div)))

/* KERNEL BOOT OPTIONS */
int disable_ether;
static int disable_sdhi;
static unsigned int vdc5fb0_opts = 1;
static unsigned int vdc5fb1_opts;

int __init early_vdc5fb0(char *str)
{
	get_option(&str, &vdc5fb0_opts);
	return 0;
}
early_param("vdc5fb0", early_vdc5fb0);

int __init early_vdc5fb1(char *str)
{
	get_option(&str, &vdc5fb1_opts);
	if (vdc5fb1_opts != 0) {
		disable_ether = 1;
		disable_sdhi = 1;
	}
	return 0;
}
early_param("vdc5fb1", early_vdc5fb1);

/*************************************************************************/

/* RESOURCES */
static struct resource vdc5fb_resources_ch0[VDC5FB_NUM_RES] = {
	[0] = DEFINE_RES_MEM_NAMED(0xfcff6000, 0x00002000, "vdc5fb.0: reg"),
	[1] = DEFINE_RES_MEM_NAMED(0x60200000, 0x00400000, "vdc5fb.0: fb"),
//	[1] = DEFINE_RES_MEM_NAMED(0x60200000, 0x00200000, "vdc5fb.0: fb"),
	[2] = DEFINE_RES_NAMED(75, 23, "vdc5fb.0: irq", IORESOURCE_IRQ),
};

/*************************************************************************/

static int vdc5fb_pinmux_common(struct platform_device *pdev, int rgb);

/* LCD-KIT-B01 */
static int vdc5fb_pinmux_lcd_kit_b01(struct platform_device *pdev)
{
	return vdc5fb_pinmux_common(pdev, 18);
}

static struct fb_videomode videomode_wvga_lcd_kit_b01 = {
	.name		= "WVGA",
	.refresh	= 60,	/* unsued */
	.xres		= 800,
	.yres		= 480,
	.pixclock	= PIXCLOCK(P1CLK, 2),
	.left_margin	= 0,
	.right_margin	= 64,
	.upper_margin	= 18,
	.lower_margin	= 18,
	.hsync_len	= 128,
	.vsync_len	= 4,
	.sync		= 0,	/* to be fixed */
	.vmode		= 0,	/* to be fixed */
	.flag		= 0,	/* to be fixed */
};

static struct vdc5fb_pdata vdc5fb_pdata_ch0_lcd_kit_b01 = {
	.name			= "LCD-KIT-B01",
	.videomode		= &videomode_wvga_lcd_kit_b01,
	.panel_icksel		= ICKSEL_P1CLK,
	.bpp			= 32,
	.panel_width		= 184,	/* mm, unused */
	.panel_height		= 132,	/* mm, unused */
	.flm_max		= 1,
	.out_format		= OUT_FORMAT_RGB666,
	.use_lvds		= 0,
	.tcon_sel		= {
		[LCD_TCON0]	= TCON_SEL_STVA,	/* VSYNC(NC) */
		[LCD_TCON1]	= TCON_SEL_STH,		/* HSYNC(NC) */
		[LCD_TCON2]	= TCON_SEL_UNUSED,	/* NC */
		[LCD_TCON3]	= TCON_SEL_UNUSED,	/* NC */
		[LCD_TCON4]	= TCON_SEL_UNUSED,	/* NC */
		[LCD_TCON5]	= TCON_SEL_UNUSED,	/* NC */
		[LCD_TCON6]	= TCON_SEL_DE,		/* DE */
	},
	.pinmux			= vdc5fb_pinmux_lcd_kit_b01,
};

/* SETUP */
static struct platform_device vdc5fb_devices[VDC5FB_NUM_CH] = {
	[0] = {
		.name		= "vdc5fb",
		.id		= 0,
		.num_resources	= ARRAY_SIZE(vdc5fb_resources_ch0),
		.resource	= vdc5fb_resources_ch0,
		.dev = {
			.dma_mask		= NULL,
			.coherent_dma_mask	= 0xffffffff,
			.platform_data		= &vdc5fb_pdata_ch0_lcd_kit_b01,
		},
	},
};

static struct platform_device *display_devices[] __initdata = {
	&vdc5fb_devices[0],
};

static int vdc5fb_setup(void)
{
	struct platform_device *pdev;

	pdev = &vdc5fb_devices[0];
	if (pdev->id == 0) {		/* VDC5 CHANNEL 0 */
		pdev->dev.platform_data =
				&vdc5fb_pdata_ch0_lcd_kit_b01;
	}
	platform_add_devices(display_devices, ARRAY_SIZE(display_devices));

	return 0;
}

/*************************************************************************/

struct pfc_pinmux_assign {
	int port;	/* enum */
	int mode;	/* enum */
	int opts;
};

static struct pfc_pinmux_assign lcd0_common[] = {
	{ P3_0,  ALT1, },	/* LCD0_CLK */
	{ P3_8,  ALT1, },	/* LCD0_DATA0 */
	{ P3_9,  ALT1, },	/* LCD0_DATA1 */
	{ P3_10, ALT1, },	/* LCD0_DATA2 */
	{ P3_11, ALT1, },	/* LCD0_DATA3 */
	{ P3_12, ALT1, },	/* LCD0_DATA4 */
	{ P3_13, ALT1, },	/* LCD0_DATA5 */
	{ P3_14, ALT1, },	/* LCD0_DATA6 */
	{ P3_15, ALT1, },	/* LCD0_DATA7 */
	{ P4_0,  ALT1, },	/* LCD0_DATA8 */
	{ P4_1,  ALT1, },	/* LCD0_DATA9 */
	{ P4_2,  ALT1, },	/* LCD0_DATA10 */
	{ P4_3,  ALT1, },	/* LCD0_DATA11 */
	{ P4_4,  ALT1, },	/* LCD0_DATA12 */
	{ P4_5,  ALT1, },	/* LCD0_DATA13 */
	{ P4_6,  ALT1, },	/* LCD0_DATA14 */
	{ P4_7,  ALT1, },	/* LCD0_DATA15 */
	{ P4_8,  ALT1, },	/* LCD0_DATA16 */
	{ P4_9,  ALT1, },	/* LCD0_DATA17 */
};

static void vdc5fb_pinmux(struct pfc_pinmux_assign *pf, size_t num)
{
	size_t n;

	for (n = 0; n < num; pf++, n++)
		rza1_pfc_pin_assign(pf->port, pf->mode, DIIO_PBDC_DIS);
}

static int vdc5fb_pinmux_common(struct platform_device *pdev, int rgb)
{
	int ret = 0;

	switch (pdev->id) {
	case 0:
		vdc5fb_pinmux(lcd0_common, ARRAY_SIZE(lcd0_common));
		break;
	default:
		ret = -ENODEV;
		break;
	}
	return ret;
}

/*************************************************************************/
