/*
 * LCD-KIT-B01 touchscreen
 * Based on tsc2007.c
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 */
#include <linux/input.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/timer.h>
#include <linux/kthread.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/leds.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <linux/i2c/lcdkitb01.h>
#include <asm/uaccess.h>
#include <mach/rza1.h>
#include <linux/cdev.h>
#include <linux/types.h>

#define USE_MOUSE 0

//タッチ情報読み込みコマンド
#define CMD_TOUCH_INFO	0x10

//パネル情報読み込みコマンド
#define CMD_PANEL_INFO	0x20

/* 割り込みステータス */
#define CMD_R8C_INTSTAT	0x01

/* 割り込みマスク */
#define CMD_R8C_INTMASK	0x02

//ファームウェア情報読み込みコマンド
#define CMD_FURM_INFO	0x40

//ファームウェア情報データサイズ
#define FURM_INFO_LEN	4

//パネル情報データサイズ
#define PANEL_INFO_LEN	6

//タッチ情報データサイズ
#define TOUCH_INFO_SIZE	9

//最大読み込みデータサイズ
#define MAX_BLKSIZE		9

//タッチ検出用ビットマスク
#define T1_BITMASK		0x01
#define T2_BITMASK		0x02

#define STATUS_SW3 (1 << 4)
#define STATUS_SW2 (1 << 3)
#define STATUS_SW1 (1 << 2)
#define STATUS_TP1 (1 << 1)
#define STATUS_TP0 (1 << 0)

//#define READ_MSECS	20
#define READ_MSECS	5

//GPIO入力検出
//#define GPIO_INT		GPIO_TO_PIN(6, 12)

//ペンアップ/ダウン検出
#define CHECK_PENUPDOWN	3
#define POS_AVG		5

//タッチフラグ
#define NOTOUCH		0
#define TOUCHED		1

//デバッグメッセージ
//#define AM800480SBTMQWT00_DEBUG
#ifdef AM800480SBTMQWT00_DEBUG
#define dbg(format, arg...) printk(KERN_INFO "%s(%d): "  format "\n", __func__, __LINE__, ## arg);
#else
#define dbg(format, arg...) do{} while(0);
#endif

//スクリーン情報構造体
struct am800480_panel_info{
	int xmax_low;		//X下限値
	int xmax_high;	//X上限値
	int ymax_low;		//Y下限値
	int ymax_high;	//Y上限値
	int x_cnum;		//Xチャンネル数
	int y_cnum;		//Yチャンネル数
};
//タッチ情報構造体
struct am800480_touch_info{
	int p1_x;		//X1
	int p1_y;		//Y1
	int p2_x;		//X2
	int p2_y;		//Y1
	bool p1_enable;	//X1 Y1タッチ状態フラグ
	bool p2_enable;	//X2 Y2タッチ状態フラグ
};

struct touch_info {
	int x;
	int y;
	int x_min;
	int y_min;
	int x_max;
	int y_max;
	int x_total;
	int y_total;
	int count;
	bool down;
	bool enable;
};

//タッチパネル情報構造体
struct lcdkitb01 {
	struct input_dev *input;		//インプットデバイス
	struct i2c_client *client;		//I2Cクライアント
	struct i2c_client *tp_client;		//I2Cクライアント
	struct delayed_work work;		//ワークキュー
	u8 ver[FURM_INFO_LEN];			//ファームウェアバージョン
	struct am800480_panel_info p_info;	//スクリーン情報
	struct touch_info t_info[2];
	int irq;				//IRQ
	u8 status;
	spinlock_t lock;			//スピンロック
	struct led_classdev led;
	struct miscdevice buzzer;
	int pendown;
	int num_report;

	struct timer_list timer;
	int irq_pin_state;

	struct lcdkitb01_platform_data *pdata;
};

#ifdef CONFIG_PM
/*
 * サスペンド
 *
 */
static int lcdkitb01_i2c_suspend(struct i2c_client *client,
					  pm_message_t message)
{
	dbg("**** %s ****\n", __func__);

	return 0;
}

/*
 * レジューム
 *
 */
static int lcdkitb01_i2c_resume(struct i2c_client *client)
{
	dbg("**** %s ****\n", __func__);

	return 0;
}
#endif

/*
 * I2C読み込み(1ワード)
 *
 */
__attribute__((unused))
static int lcdkitb01_i2c_read(struct i2c_client *client, u8 reg)
{
	return 0;
}

/*
 * I2C読み込み(マルチワード)
 *
 */
#define MEASURE_TIME 0
#if MEASURE_TIME
static struct timeval tv0 = {}, tv1 = {};
#endif
static int lcdkitb01_i2c_multi_read(struct i2c_client *client,
				 u8 first_reg, u8 count, u8 *buf)
{
	u8  rcv[MAX_BLKSIZE];	//受信用バッファ
	int ret;			//受信バイト数

	//受信サイズチェック
	if (MAX_BLKSIZE < count) {
		pr_warning("%s overflow size %d\n", __func__, count);
	}

	//受信
#if MEASURE_TIME
	do_gettimeofday(&tv1);
	printk(KERN_DEBUG "MR %8d\n",
	       (tv1.tv_sec - tv0.tv_sec) * 1000000 +
	       (tv1.tv_usec - tv0.tv_usec));
	tv0 = tv1;
#endif
	ret = i2c_smbus_read_i2c_block_data(client, first_reg, count, rcv);
	if (ret < 0 || count > ret) {
		pr_warning("%s i2c_smbus_read_i2c_block_data fail %d\n", __func__, ret);
		return -1;
	}
	memcpy(buf, rcv, count);

	return ret;
}

/*
 * 書き込み
 *
 */
#if 0
static int lcdkitb01_i2c_write(struct i2c_client *client, u8 reg, u8 val)
{
	dbg("**** %s ****\n", __func__);

	return 0;
}
#endif

static void lcdkitb01_penup(struct lcdkitb01 *ts)
{
	//ペンアップ通知

	//spin_lock(&ts->lock);

	if (ts->pendown == 0)
		goto end;

//	printk("Pen Up\n");

	ts->pendown = 0;

	//spin_unlock(&ts->lock);

#if 1
	input_report_abs(ts->input,
			 ABS_MT_TOUCH_MAJOR,
			 0);
#endif
	input_report_abs(ts->input,
			 ABS_PRESSURE,
			 0);
	input_report_key(ts->input,
			 BTN_TOUCH,
			 0);
	input_sync(ts->input);
end:
	;
}
#if 1
static void tp_reset_slot(struct touch_info *ti)
{
	memset(ti, 0, sizeof (*ti));
}
#endif
static int tp_sample(struct touch_info *ti, int x, int y)
{
	ti->x_total += x;
	ti->y_total += y;
		
	if (ti->count == 0) {
		ti->x_min = x;
		ti->y_min = y;
		ti->x_max = x;
		ti->y_max = y;
	} else {
		ti->x_min = min(ti->x_min, x);
		ti->y_min = min(ti->y_min, y);
		ti->x_max = max(ti->x_min, x);
		ti->y_max = max(ti->y_min, y);
	}

	ti->count++;

	ti->enable = ti->count >= POS_AVG;

	if (!ti->enable)
		return 0;

	ti->x_total -= ti->x_min;
	ti->x_total -= ti->x_max;
	ti->y_total -= ti->y_min;
	ti->y_total -= ti->y_max;

	ti->x = ti->x_total / (POS_AVG - 2);
	ti->y = ti->y_total / (POS_AVG - 2);

	return 1;
}

static void tp_process(struct lcdkitb01 *ts)
{
	u8 rcv[MAX_BLKSIZE];		//受信用バッファ
	int tmp_x,tmp_y;		//ポイントテンポラリ
	int pendown = 0;

	dbg("");

	if (0 > lcdkitb01_i2c_multi_read(ts->tp_client,
				      CMD_TOUCH_INFO,
				      TOUCH_INFO_SIZE,
				      rcv)) {
		pr_warning("%s panel info read fail\n", __func__);
		goto end;
	}

	// タッチ情報の読み込み
	ts->t_info[0].down = rcv[0] & T1_BITMASK;
	ts->t_info[1].down = (rcv[0] & T2_BITMASK) >> 1;

	if (!ts->t_info[0].down && !ts->t_info[1].down) {
		lcdkitb01_penup(ts);
		tp_reset_slot(&ts->t_info[0]);
		tp_reset_slot(&ts->t_info[1]);
		goto end;
	}

	if (ts->t_info[0].down) {
		//タッチポイントの取得
		tmp_x = rcv[1] | rcv[2] << 8;
		tmp_y = rcv[3] | rcv[4] << 8;

		tmp_x = clamp(tmp_x,
			      ts->p_info.xmax_low, ts->p_info.xmax_high);
		tmp_y = clamp(tmp_y,
			      ts->p_info.ymax_low, ts->p_info.ymax_high);

		dbg(" x=%d y=%d\n", tmp_x, tmp_y);
		tp_sample(&ts->t_info[0], tmp_x, tmp_y);
	} else
		tp_reset_slot(&ts->t_info[0]);

	if (ts->t_info[1].down) {
		ts->t_info[1].enable = 1;
		//タッチポイントの取得
		tmp_x = rcv[5] | rcv[6] << 8;
		tmp_y = rcv[7] | rcv[8] << 8;

		tmp_x = clamp(tmp_x,
			      ts->p_info.xmax_low, ts->p_info.xmax_high);
		tmp_y = clamp(tmp_y,
			      ts->p_info.ymax_low, ts->p_info.ymax_high);

		tp_sample(&ts->t_info[1], tmp_x, tmp_y);
	} else
		tp_reset_slot(&ts->t_info[1]);


	if (!ts->t_info[0].enable && !ts->t_info[1].enable)
		goto end;

	if (ts->pendown == 0) {
		ts->pendown = 1;
		pendown = 1;
	}

	//printk("pendown=%d\n", pendown);

//	spin_unlock(&ts->lock);

	if (pendown) {
		input_report_key(ts->input, BTN_TOUCH, 1);
//		printk("Pen Down\n");
	}

	//ペンダウン通知
	int major = 70;
	if (ts->t_info[0].enable && ts->t_info[1].enable) {
		printk("report multi touch\n");
		//マルチタッチ通知(p1)
		input_report_abs(ts->input,
				 ABS_MT_POSITION_X,
				 ts->t_info[0].x);
		input_report_abs(ts->input,
				 ABS_MT_POSITION_Y,
				 ts->t_info[0].y);
		input_report_abs(ts->input,
				 ABS_MT_TOUCH_MAJOR,
				 major);
		input_mt_sync(ts->input);
		//マルチタッチ通知(p2)
		input_report_abs(ts->input,
				 ABS_MT_POSITION_X,
				 ts->t_info[1].x);
		input_report_abs(ts->input,
				 ABS_MT_POSITION_Y,
				 ts->t_info[1].y);
		input_report_abs(ts->input,
				 ABS_MT_TOUCH_MAJOR,
				 major);
		input_mt_sync(ts->input);

		tp_reset_slot(&ts->t_info[0]);
		tp_reset_slot(&ts->t_info[1]);
	} else {
		//シングルタッチ通知
		dbg(" x=%d y=%d", ts->t_info[0].x, ts->t_info[0].y);

		input_report_abs(ts->input, ABS_X, ts->t_info[0].x);
		input_report_abs(ts->input, ABS_Y, ts->t_info[0].y);

		input_report_abs(ts->input,
				 ABS_MT_POSITION_X,
				 ts->t_info[0].x);
		input_report_abs(ts->input,
				 ABS_MT_POSITION_Y,
				 ts->t_info[0].y);
		input_report_abs(ts->input,
				 ABS_MT_TOUCH_MAJOR,
				 major);
		input_mt_sync(ts->input);

		//p1通知(シングルタッチ)
		tp_reset_slot(&ts->t_info[0]);
	}

//	spin_lock(&ts->lock);

	input_sync(ts->input);

end:
	;
}

static void lcdkitb01_read(struct work_struct *work)
{
	struct lcdkitb01 *ts;	//タッチパネル情報
	s32 ret = 0;

	ts = container_of(to_delayed_work(work), struct lcdkitb01, work);

	tp_process(ts);

	/* 割り込み情報取得 */
	ret = i2c_smbus_read_byte_data(ts->client, CMD_R8C_INTSTAT);
	
	dbg("ret = 0x%02x", ret);

#if USE_MOUSE
	if ((ts->status & STATUS_SW1) != (ret & STATUS_SW1))
		input_report_key(ts->input, BTN_LEFT, ret & STATUS_SW1);
	if ((ts->status & STATUS_SW2) != (ret & STATUS_SW2))
		input_report_key(ts->input, BTN_MIDDLE, ret & STATUS_SW2);
	if ((ts->status & STATUS_SW3) != (ret & STATUS_SW3))
		input_report_key(ts->input, BTN_RIGHT, ret & STATUS_SW3);
#else
	if ((ts->status & STATUS_SW1) != (ret & STATUS_SW1))
		input_report_key(ts->input, KEY_F1, ret & STATUS_SW1);
	if ((ts->status & STATUS_SW2) != (ret & STATUS_SW2))
		input_report_key(ts->input, KEY_HOME, ret & STATUS_SW2);
	if ((ts->status & STATUS_SW3) != (ret & STATUS_SW3))
		input_report_key(ts->input, KEY_ESC, ret & STATUS_SW3);
#endif
	input_sync(ts->input);

	ts->status = ret;
	if (ts->status) {
		schedule_delayed_work(&ts->work, msecs_to_jiffies(READ_MSECS));
	}
}

/*
 * タッチパネル割り込み
 *
 */
static irqreturn_t lcdkitb01_irq(int irq, void *handle)
{
	struct lcdkitb01 *ts = handle;

	dbg("");

	schedule_delayed_work(&ts->work, 0);

	return IRQ_HANDLED;
}

static void lcdkitb01_poll_irq(unsigned long arg)
{
	struct lcdkitb01 *ts = (struct lcdkitb01 *)arg;
	struct lcdkitb01_platform_data *pdata = ts->pdata;
	int pin_state = pdata->get_irq_pin_state();

	if (ts->irq_pin_state == -1)
		ts->irq_pin_state = pin_state;
	else if (ts->irq_pin_state != pin_state) {
		ts->irq_pin_state = pin_state;
		schedule_delayed_work(&ts->work, 0);
	}

	mod_timer(&ts->timer, jiffies + pdata->poll_period);
}

static struct i2c_board_info tp_i2c_info = {
	I2C_BOARD_INFO("AM800480SBTMQW-T100", 0x41),
};

static void lcdkitb01_led_set(struct led_classdev *led_cdev,
			      enum led_brightness value)
{
	struct lcdkitb01 *ts;

	dbg("");

	ts = container_of(led_cdev, struct lcdkitb01, led);
	value = value * 100 / 255;
	i2c_smbus_write_byte_data(ts->client, 0x03, value);	
}

static struct led_classdev lcdkitb01_led = {
	.name		= "lcd-backlight",
	.brightness_set	= lcdkitb01_led_set,
};

static ssize_t lcdkitb01_buzzer_read(struct file *filp, char __user *buf,
				    size_t count, loff_t *pos)
{
	return -EINVAL;
}

static ssize_t lcdkitb01_buzzer_write(struct file *filp,
				      const char __user *buf,
				      size_t count, loff_t *pos)
{
	struct lcdkitb01 *ts =
		container_of(filp->private_data, struct lcdkitb01, buzzer);
	u8 data;

	get_user(data, buf);
	i2c_smbus_write_byte_data(ts->client, 0x04, data);

	return count;
}

static struct file_operations lcdkitb01_buzzer_fops = {
	.owner = THIS_MODULE,
	.llseek = no_llseek,
	.read = lcdkitb01_buzzer_read,
	.write = lcdkitb01_buzzer_write,
	.open = nonseekable_open,
};

static struct miscdevice lcdkitb01_buzzer_device = {
	MISC_DYNAMIC_MINOR,
	"lcdkitb01_buzzer",
	&lcdkitb01_buzzer_fops,
};

/*
 * ドライバ初期化
 *
 */
//static int __devinit lcdkitb01_probe(struct i2c_client *client,
static int lcdkitb01_probe(struct i2c_client *client,
				   const struct i2c_device_id *id)
{
	struct lcdkitb01_platform_data *pdata = client->dev.platform_data;
	struct lcdkitb01 *ts;		//タッチパネル情報
	struct input_dev *input_dev;	//インプットデバイス情報
	u8 rcv[MAX_BLKSIZE];			//受信用バッファ
	int err = 0;				//戻り値

	printk("LCD-KIT-B01 TouchPanel Driver.\n");

	/* DE設定 */
	rza1_pfc_pin_assign(P3_7, ALT1, 0);

	//I2Cファンクションチェック
	if (!i2c_check_functionality(client->adapter,
				     I2C_FUNC_SMBUS_READ_WORD_DATA)){
		printk("%s i2c_check_functionality fail\n", __func__);
		return -EIO;
	}

	//デバイス用メモリ確保
	ts = kzalloc(sizeof(struct lcdkitb01), GFP_KERNEL);
	input_dev = input_allocate_device();
	if (!ts || !input_dev) {
		printk("%s kzalloc fail\n", __func__);
		err = -ENOMEM;
		goto err_free_mem;
	}

	ts->client = client;
	ts->irq = client->irq;
	ts->pdata = pdata;
	ts->input = input_dev;

	i2c_set_clientdata(client, ts);

	spin_lock_init(&ts->lock);

	INIT_DELAYED_WORK(&ts->work, lcdkitb01_read);

	ts->tp_client = i2c_new_device(client->adapter, &tp_i2c_info);

	if (0 > lcdkitb01_i2c_multi_read(ts->tp_client,
				      CMD_FURM_INFO, FURM_INFO_LEN, rcv)) {
		pr_warning("%s fail get furm version\n", __func__);
		err = -EIO;
		goto err_free_mem;
	}
	memcpy(ts->ver, rcv, FURM_INFO_LEN);

	//パネル情報読み込み
	if (0 > lcdkitb01_i2c_multi_read(ts->tp_client, CMD_PANEL_INFO, PANEL_INFO_LEN, rcv)){
		pr_warning("%s fail get panel info\n", __func__);
		err = -EIO;
		goto err_free_mem;
	}
	ts->status = 0;
	ts->p_info.xmax_low  = 0;
	ts->p_info.ymax_low  = 0;
	ts->p_info.xmax_high =  rcv[0] | rcv[1] << 8;
	ts->p_info.ymax_high =  rcv[2] | rcv[3] << 8;
	ts->p_info.x_cnum    =  rcv[4];
	ts->p_info.y_cnum    =  rcv[5];

	//パネル情報表示
#if 0
	dbg("%s panel info", __func__);
	dbg("furm ver 0x%02X:0x%02X:0x%02X", *(ts->ver+1), *(ts->ver+2), *(ts->ver+3));
	printk("xmax low %d\n", ts->p_info.xmax_low);
	printk("xmax high %d\n", ts->p_info.xmax_high);
	printk("ymax low %d\n", ts->p_info.ymax_low);
	printk("ymax high %d\n", ts->p_info.ymax_high);
	printk("x channel num %d\n", ts->p_info.x_cnum);
	printk("y channel num %d\n", ts->p_info.y_cnum);
#endif
	
	//タッチパネルデバイス初期化
	input_dev->name = "LCD-KIT-B01";
	input_dev->id.bustype = BUS_I2C;
	input_dev->evbit[0] = BIT_MASK(EV_KEY) | BIT_MASK(EV_ABS);

	//IRQ初期化
	dbg("ts->irq = %d", ts->irq);
	if (ts->irq != NO_IRQ) {
		err = request_irq(ts->irq, lcdkitb01_irq,
				  //IRQF_SHARED | IRQ_TYPE_LEVEL_HIGH,
				  IRQF_SHARED | IRQ_TYPE_EDGE_RISING,
				  input_dev->name, ts);
		if (err) {
			pr_warning("lcdkitb01 irq request fail %d\n", err);
			err = -EIO;
			goto err_free_gpio;
		} else {
			i2c_smbus_write_byte_data(client, CMD_R8C_INTMASK, 0x00);
		}
	} else {
		ts->irq_pin_state = -1;
		init_timer(&ts->timer);
		ts->timer.expires = jiffies + ts->pdata->poll_period;
		setup_timer(&ts->timer, lcdkitb01_poll_irq, ts);
		add_timer(&ts->timer);
	}

	//デバイス登録
	input_set_abs_params(input_dev, ABS_X,
			     ts->p_info.xmax_low, ts->p_info.xmax_high, 0, 0);
	input_set_abs_params(input_dev, ABS_Y,
			     ts->p_info.xmax_low, ts->p_info.ymax_high, 0, 0);
	
	input_set_abs_params(input_dev,
			     ABS_MT_TOUCH_MAJOR,
			     0, ts->p_info.xmax_high, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_POSITION_X,
			     ts->p_info.xmax_low, ts->p_info.xmax_high, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_POSITION_Y,
			     ts->p_info.xmax_low, ts->p_info.ymax_high, 0, 0);

#if USE_MOUSE
	input_set_capability(input_dev, EV_KEY, BTN_LEFT);
	input_set_capability(input_dev, EV_KEY, BTN_RIGHT);
	input_set_capability(input_dev, EV_KEY, BTN_MIDDLE);
#else
	input_set_capability(input_dev, EV_KEY, KEY_HOME);
	input_set_capability(input_dev, EV_KEY, KEY_F1);
	input_set_capability(input_dev, EV_KEY, KEY_ESC);
#endif
	input_set_capability(input_dev, EV_KEY, BTN_TOUCH);

	input_set_drvdata(input_dev, ts);	

	err = input_register_device(input_dev);
	if (err)
		goto err_free_irq;

	dbg("LCDKITB01 additional driver: LED, Buzzer");

	// Backlight
	ts->led = lcdkitb01_led;
	led_classdev_register(&client->dev, &ts->led);
	lcdkitb01_led_set(&ts->led, 255);

	// Buzzer
	lcdkitb01_buzzer_device.parent = &input_dev->dev;
	ts->buzzer = lcdkitb01_buzzer_device;
	misc_register(&ts->buzzer);

	return err;

//初期化失敗
err_free_irq:
	free_irq(ts->irq, ts);

err_free_gpio:
//	gpio_free(GPIO_INT);

err_free_mem:
	input_free_device(input_dev);
	kfree(ts);

	return err;
}

/*
 * ドライバ破棄
 *
 */
//static int __devexit lcdkitb01_remove(struct i2c_client *client)
static int lcdkitb01_remove(struct i2c_client *client)
{
	struct lcdkitb01 *ts = i2c_get_clientdata(client);

	led_classdev_unregister(&ts->led);

	if (ts->irq != NO_IRQ)
		free_irq(ts->irq, ts);
	else
		del_timer_sync(&ts->timer);

	input_unregister_device(ts->input);
	kfree(ts);

	return 0;
}
//I2Cデバイステーブル
static const struct i2c_device_id lcdkitb01_idtable[] = {
	{ "LCD-KIT-B01", 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, lcdkitb01_idtable);

//I2Cドライバ情報
static struct i2c_driver lcdkitb01_driver = {
	.driver = {
		.owner	= THIS_MODULE,
		.name	= "LCD-KIT-B01"
	},
	.id_table	= lcdkitb01_idtable,
	.probe		= lcdkitb01_probe,
//	.remove		= __devexit_p(lcdkitb01_remove),
	.remove		= lcdkitb01_remove,
#ifdef CONFIG_PM
	.suspend	= lcdkitb01_i2c_suspend,
	.resume		= lcdkitb01_i2c_resume,
#endif
};

static int __init lcdkitb01_init(void)
{
	/* ドライバ登録 */
	return i2c_add_driver(&lcdkitb01_driver);
}

static void __exit lcdkitb01_exit(void)
{
	i2c_del_driver(&lcdkitb01_driver);
}

module_init(lcdkitb01_init);
module_exit(lcdkitb01_exit);
//module_i2c_driver(lcdkitb01_driver);

MODULE_AUTHOR("Alpha Project Co., LTD");
MODULE_DESCRIPTION("LCD-KIT-B01 TouchScreen Driver");
MODULE_LICENSE("GPL");

