/*
 * max9867.c -- MAX9867 ALSA SoC Audio driver
 *
 * Copyright 2008 Maxim Integrated Products
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
 
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
//#include <sound/driver.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>

#include "max9867.h"

#define AUDIO_NAME "MAX9867"
#define MAX9867_DRIVER_VERSION "0.1" 

/*
 * Debug
 */

#define MAX9867_DEBUG 1

#ifdef MAX9867_DEBUG
#define dbg(format, arg...) \
	printk(KERN_DEBUG AUDIO_NAME ": " format "\n" , ## arg)
#else
#define dbg(format, arg...) do {} while (0)
#endif
#define err(format, arg...) \
	printk(KERN_ERR AUDIO_NAME ": " format "\n" , ## arg)
#define info(format, arg...) \
	printk(KERN_INFO AUDIO_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) \
	printk(KERN_WARNING AUDIO_NAME ": " format "\n" , ## arg)
	
//struct snd_soc_codec_device soc_codec_dev_max9867;

/* codec private data */
struct max9867_priv {
	unsigned int sysclk;
};

/*******************************************************************************
 * read the max9867 register space
 ******************************************************************************/
static unsigned int max9867_read(struct snd_soc_codec *codec, unsigned int reg)
{
	struct i2c_msg msg[2];
	struct i2c_client *client;
	u8 data[2];
	int ret;
	
	client = (struct i2c_client *)codec->control_data;
	data[0] = reg & 0xff;
	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].buf = &data[0];
	msg[0].len = 1;
	
	msg[1].addr = client->addr;
	msg[1].flags = I2C_M_RD;
	msg[1].buf = &data[1];
	msg[1].len = 1;
	
	ret = i2c_transfer(client->adapter, &msg[0], 2);
	return (ret == 2) ? data[1] : -EIO;
	
}

/*******************************************************************************
 * write to the max9867 register space
 ******************************************************************************/
static int max9867_write(struct snd_soc_codec *codec, unsigned int reg,
	unsigned int value)
{
	u8 data[2];

	data[0] = reg & 0x00ff;
	data[1] = value & 0x00ff;
	
	if (codec->hw_write(codec->control_data, data, 2) == 2)
		return 0;
	else
		return -EIO;
}

/*******************************************************************************
 *
 ******************************************************************************/
static const char *max9867_sidetone_mixer[] = {"None", "Left ADC", "Right ADC", "Left+Right ADC"};
static const struct soc_enum max9867_sidetone_mixer_enum[] = {
	SOC_ENUM_SINGLE_EXT(4, max9867_sidetone_mixer),
};

#define MAX9867_SIDETONE_MIX_NONE 0
#define MAX9867_SIDETONE_MIX_LEFT_ADC 1
#define MAX9867_SIDETONE_MIX_RIGHT_ADC 2
#define MAX9867_SIDETONE_MIX_LEFTRIGHT_ADC 3

static int max9867_sidetone_mixer_value;

static int max9867_sidetone_mixer_set(struct snd_kcontrol *kcontrol,
									  struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);

	u8 reg;

	dbg("sidetone_mixer_set()");
	
	/* Return if this call will not change anything */
	if (max9867_sidetone_mixer_value == ucontrol->value.integer.value[0])
		return 0;
	
	/* Save the new setting */
	max9867_sidetone_mixer_value = ucontrol->value.integer.value[0];
	
	/* Clear the bits that may be touched */
	reg = max9867_read(codec, MAX9867_DACGAIN) & 0x3f;

	switch (max9867_sidetone_mixer_value) {
	case MAX9867_SIDETONE_MIX_NONE:
		/* reg |= 0x00; Not required, masking register value above takes care of this */
		break;
	case MAX9867_SIDETONE_MIX_LEFT_ADC:
		reg |= 0x40;
		break;
	case MAX9867_SIDETONE_MIX_RIGHT_ADC:
		reg |= 0x80;
		break;
	case MAX9867_SIDETONE_MIX_LEFTRIGHT_ADC:
		reg |= 0xc0;
		break;
	}
	
	max9867_write(codec, MAX9867_DACGAIN, reg);

	return 1;
}


/*******************************************************************************
 * 
 ******************************************************************************/
static int max9867_sidetone_mixer_get(struct snd_kcontrol *kcontrol,
									  struct snd_ctl_elem_value *ucontrol)
{
	dbg("sidetone_mixer_get()");

	ucontrol->value.integer.value[0] = max9867_sidetone_mixer_value;

	return 0;
}

/*******************************************************************************
 * TODO The variable max9867_input_mixer_value needs to be intialized and
 *      the MAX9867_INPUTCONFIG register should be set to the init value.
 ******************************************************************************/
static const char *max9867_input_mixer[] = {"None", "Analog Mic", "Line", "Analog Mic+Line"};
static const struct soc_enum max9867_input_mixer_enum[] = { 
	SOC_ENUM_SINGLE_EXT(4, max9867_input_mixer),
};
/* These defines correlate to the position of like named value in max9867_input_mixer[] */
#define MAX9867_INPUT_MIX_NONE 0
#define MAX9867_INPUT_MIX_ANALOG_MIC 1
#define MAX9867_INPUT_MIX_LINE 2
#define MAX9867_INPUT_MIX_ANALOG_MIC_PLUS_LINE 3

static int max9867_input_mixer_value;

static int max9867_input_mixer_set(struct snd_kcontrol *kcontrol,
								   struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	u8 reg;

	/* Return if this call will not change anything */
	if (max9867_input_mixer_value == ucontrol->value.integer.value[0])
		return 0;
	
	/* Save the new setting */
	max9867_input_mixer_value = ucontrol->value.integer.value[0];
	
	/* Clear the bits that may be touched */
	reg = max9867_read(codec, MAX9867_INPUTCONFIG) & 0x0f;
	
	switch (max9867_input_mixer_value) {
	case MAX9867_INPUT_MIX_NONE:
		/* reg |= 0x00; Not required, masking above takes care of this */
		break;
	case MAX9867_INPUT_MIX_ANALOG_MIC:
		reg |= 0x50;
		break;
	case MAX9867_INPUT_MIX_LINE:
		reg |= 0xa0;
		break;
	case MAX9867_INPUT_MIX_ANALOG_MIC_PLUS_LINE:
		reg |= 0xf0;
		break;
	}
	
	max9867_write(codec, MAX9867_INPUTCONFIG, reg);

	return 1;
}

/*******************************************************************************
 * 
 ******************************************************************************/
static int max9867_input_mixer_get(struct snd_kcontrol *kcontrol,
								   struct snd_ctl_elem_value *ucontrol)
{
	dbg("maxim_mixer_get()");

	ucontrol->value.integer.value[0] = max9867_input_mixer_value;

	return 0;
}


static const char *max9867_dac_mute[] = {"Disable", "Enable"};
static const char *max9867_spmode[] = {
						"Stereo BTL", "Mono BTL",
						"Stereo OCL", "Mono OCL",
						"Stereo Single-Ended (Clickless)", "Mono Single-Ended (Clickless)",
						"Stereo Single-Ended (Accelerated Turn-ON)", "Mono Single-Ended (Accelerated Turn-ON)"};
static const char *max9867_vsen[] = {"Enable", "Disable"};
static const char *max9867_zden[] = {"Enable", "Disable"};
static const char *max9867_dsp[] = {"IIR", "FIR"};

/*******************************************************************************
 * The mask value for SOC_ENUM_SINGLE is the number of
 * elements in the enumeration. Left and Right controls
 * are used because there is no SOC_ENUM_DOUBLE_R at this time.
 ******************************************************************************/
static const struct soc_enum max9867_enum[] = {
	SOC_ENUM_SINGLE(MAX9867_DACLEVEL, 6, 2, max9867_dac_mute),
	SOC_ENUM_SINGLE(MAX9867_MODECONFIG, 0, 8, max9867_spmode),
	SOC_ENUM_SINGLE(MAX9867_MODECONFIG, 6, 2, max9867_vsen),
	SOC_ENUM_SINGLE(MAX9867_MODECONFIG, 5, 2, max9867_zden),
	SOC_ENUM_SINGLE(MAX9867_CODECFLTR, 7, 2, max9867_dsp),
};

static const struct snd_kcontrol_new max9867_snd_controls[] = {
	/* */
	SOC_DOUBLE_R("Master Playback Volume", MAX9867_LEFTVOL, MAX9867_RIGHTVOL, 0, 63, 1),
	/* */
	SOC_DOUBLE_R("Capture Volume", MAX9867_LEFTLINELVL, MAX9867_RIGTHLINELVL, 0, 15, 1),
	/* */
	SOC_DOUBLE_R("Mic Gain", MAX9867_LEFTMICGAIN, MAX9867_RIGHTMICGAIN, 0, 31, 1),	
	/* */
	SOC_ENUM_EXT("Input Mixer",
							 max9867_input_mixer_enum,
							 max9867_input_mixer_get,
							 max9867_input_mixer_set),
	/* */
	SOC_ENUM_EXT("Digital Sidetone Mixer",
							 max9867_sidetone_mixer_enum,
							 max9867_sidetone_mixer_get,
							 max9867_sidetone_mixer_set),
	/* */
	SOC_SINGLE("Sidetone Gain", MAX9867_DACGAIN, 0, 31, 1),
	/* */
	SOC_SINGLE("DAC Gain", MAX9867_DACLEVEL, 4, 3, 0),
	/* */
	SOC_SINGLE("DAC Attenuation", MAX9867_DACLEVEL, 0, 15, 1),
	/* */
	SOC_DOUBLE("ADC Gain", MAX9867_ADCLEVEL, 4, 0, 15, 1),
	/* */
	SOC_ENUM("DAC Mute", max9867_enum[0]),
	/* */
	SOC_ENUM("Speaker Mode", max9867_enum[1]),
	/* */
	SOC_ENUM("Volume Smoothing", max9867_enum[2]),
	/* */
	SOC_ENUM("Zero-Crossing Detection", max9867_enum[3]),
	/* */
	SOC_ENUM("DSP Filter", max9867_enum[4]),
};


/*******************************************************************************
 * add non dapm controls 
 ******************************************************************************/
static int max9867_add_controls(struct snd_soc_codec *codec)
{
//	int err, i;

//	for (i = 0; i < ARRAY_SIZE(max9867_snd_controls); i++) {
//		if ((err = snd_ctl_add(codec->card,
//				snd_soc_cnew(&max9867_snd_controls[i],codec, NULL))) < 0)
//			return err;
//	}

	return 0;
}

static const struct snd_soc_dapm_widget max9867_dapm_widgets[] = {
	
	SND_SOC_DAPM_MIXER("Output Mixer", SND_SOC_NOPM, 0, 0, NULL, 0),
	
	SND_SOC_DAPM_DAC("LDAC", "HiFi Playback", MAX9867_PWRMAN, 3, 0),
	SND_SOC_DAPM_DAC("RDAC", "HiFi Playback", MAX9867_PWRMAN, 2, 0),
	
	
	SND_SOC_DAPM_OUTPUT("LHPOUT"),
	SND_SOC_DAPM_OUTPUT("LOUT"),
	SND_SOC_DAPM_OUTPUT("RHPOUT"),	
	SND_SOC_DAPM_OUTPUT("ROUT"),
	
	SND_SOC_DAPM_ADC("LADC", "HiFi Capture", MAX9867_PWRMAN, 1, 1),
	SND_SOC_DAPM_ADC("RADC", "HiFi Capture", MAX9867_PWRMAN, 0, 1),

	SND_SOC_DAPM_MIXER("Input Mixer", SND_SOC_NOPM, 0, 0, NULL, 0),
	
	SND_SOC_DAPM_INPUT("LMICIN"),
	SND_SOC_DAPM_INPUT("RMICIN"),
	SND_SOC_DAPM_INPUT("LLINEIN"),
	SND_SOC_DAPM_INPUT("RLINEIN"),
	SND_SOC_DAPM_INPUT("AUXIN"),
};

//static const char *intercon[][3] = {
static const struct snd_soc_dapm_route max9867_intercon[] = {

	/* Output mixer */
	{"Output Mixer", NULL, "Line Input"},
	{"Output Mixer", NULL, "LDAC"},
	{"Output Mixer", NULL, "RDAC"},
	{"Output Mixer", NULL, "Mic Input"},
		
	/* Outputs */
	{"LHPOUT", NULL, "Output Mixer"},
	{"LOUT", NULL, "Output Mixer"},
	{"RHPOUT", NULL, "Output Mixer"},
	{"ROUT", NULL, "Output Mixer"},
		
	/* Input mixer */
	{"Input Mixer", NULL, "Line Input"},
	{"Input Mixer", NULL, "Mic Input"},
	{"Input Mixer", NULL, "Aux Input"},
	{"LADC", NULL, "Input Mixer"},
	{"RADC", NULL, "Input Mixer"},
		
	/* Inputs */
	{"Mic Input", NULL, "LMICIN"},
	{"Mic Input", NULL, "RMICIN"},
	{"Line Input", NULL, "LLINEIN"},
	{"Line Input", NULL, "RLINEIN"},
	{"Aux Input", NULL, "AUXIN"},
		
	/* Terminator */
//	{NULL, NULL, NULL},
};


/*******************************************************************************
 * add widgets
 ******************************************************************************/
static int max9867_add_widgets(struct snd_soc_codec *codec)
{
	struct snd_soc_dapm_context *dapm = &codec->dapm;

	snd_soc_dapm_new_controls(dapm, max9867_dapm_widgets,
				  ARRAY_SIZE(max9867_dapm_widgets));

	snd_soc_dapm_add_routes(dapm, max9867_intercon,
				ARRAY_SIZE(max9867_intercon));

	snd_soc_dapm_new_widgets(dapm);

	return 0;
}

struct _coeff_div {
	u32 mclk;
	u32 rate;
	u8 ni_h;
	u8 ni_l;
};

static const struct _coeff_div coeff_div[] = {
	{12288000, 8000,  0x10, 0x00},
	{12288000, 11025, 0x16, 0x0d},
/* 	{12288000, 12000, 0x18, 0x00}, ALSA does not support this bit rate just yet */
	{12288000, 16000, 0x20, 0x00},
	{12288000, 22050, 0x2c, 0x1a},
/*	{12288000, 24000, 0x30, 0x00},  ALAS does not support this bit rate just yet */
	{12288000, 32000, 0x40, 0x00},
	{12288000, 44100, 0x58, 0x33},
	{12288000, 48000, 0x60, 0x00},
};


/*******************************************************************************
 * Search coefficient table for entry matching
 * master clock and bit rate values.
 ******************************************************************************/
static inline int get_coeff(int mclk, int rate)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(coeff_div); i++) {
		if (coeff_div[i].rate == rate && coeff_div[i].mclk == mclk)
			return i;
	}
	
	err("max9867: could not get coeff for mclk %d @ rate %d",	mclk, rate);
	
	return 0;
}


/*******************************************************************************
 * updated
 ******************************************************************************/
static int max9867_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params,
				 struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct max9867_priv *max9867 = snd_soc_codec_get_drvdata(codec);
	u8 ni_h;
	u8 ni_l;
	int i = get_coeff(max9867->sysclk, params_rate(params));
	
	/* Enable PLL here */
	ni_h = 0x80 | coeff_div[i].ni_h;
	ni_l = coeff_div[i].ni_l << 1;
	
	max9867_write(codec, MAX9867_AUDIOCLKHIGH, ni_h);
	max9867_write(codec, MAX9867_AUDIOCLKLOW, ni_l);
	
	return 0;
}


/*******************************************************************************
 * updated
 * Shutdown the device using the SHDN bit in power control register
 ******************************************************************************/
static int max9867_pcm_prepare(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params,
				struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	
	u16 reg = max9867_read(codec, MAX9867_PWRMAN);
	
	max9867_write(codec, MAX9867_PWRMAN, reg | 0x80);
	
	return 0;
}

/*******************************************************************************
 * updated
 * Enable device using the SHDN bit in power control register
 ******************************************************************************/
static void max9867_shutdown(struct snd_pcm_substream *substream,
				struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	u16 reg = max9867_read(codec, MAX9867_PWRMAN);

	max9867_write(codec, MAX9867_PWRMAN, reg & ~0x80);
}

/*******************************************************************************
 * updated
 * Mute DAC based on value of MUTE
 * The Volume register also has a mute control bit.
 ******************************************************************************/
#if 0
static int max9867_mute(struct snd_soc_codec_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;

	/* 
		Get the current state of the DAC Level register and clear
	  the DAC Mute Enable bit.
	*/
	u8 mute_reg = max9867_read(codec, MAX9867_DACLEVEL) & 0x3f;
	
	/*
		A MUTE value of 1 mutes the DAC
	*/
	if (mute)
		mute_reg |= 0x40;

	max9867_write(codec, MAX9867_DACLEVEL, mute_reg);
	
	return 0;
}
#endif

/*******************************************************************************
 * Validate and save the master clock value. If multiple clocks are to be
 * supported this is the place where the requested clock would be validated
 * against the	a set of supported clocks.
 ******************************************************************************/
static int max9867_set_dai_sysclk(struct snd_soc_dai *codec_dai,
					int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct max9867_priv *max9867 = snd_soc_codec_get_drvdata(codec);
	u8 reg = 0;
	
	/* The max9867 EVKit provides a 12.288MHz clock.	*/
	if (freq == 12288000)	{
		max9867->sysclk = freq;
	}
	else {
		err("max9867_set_dai_sysclk() bad clock frequency %d", freq);
		return -EINVAL;
	}
	
	/* Set the prescaler based on the master clock frequency*/
	reg &= ~0x30;
	if (freq >= 10000000 && freq <= 20000000) {
		reg |= 0x10;
	}
	else if (freq >= 20000000 && freq <= 40000000) {
		reg |= 0x20;
	}
	else if (freq >= 40000000 && freq <= 60000000) {
		reg |= 0x30;
	}
	
	/* Clear the FREQ bits */
	/* TODO
		These may end up being dynamically set when the driver
		is configured for a specific bit rate.
	*/
	reg &= ~0x0f;
	
	max9867_write(codec, MAX9867_SYSCLK, reg);
		
	return 0;
}

/*******************************************************************************
 * updated
 ******************************************************************************/
static int max9867_set_dai_fmt(struct snd_soc_dai *codec_dai,
				   unsigned int fmt)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	
	u8 iface1A = 0x00;
	u8 iface1B = 0x00;
	
	/*
		Gather the bits that make up Interface 1a
	*/

	/* Clear master bit, operate as slave */
	iface1A &= ~0x80;
	
	/* Set DAC Data Delay bit, required for I2S mode */
	iface1A |= 0x10;
	
	/* Disable SDOUT Hi-Z Mode */
	iface1A |= 0x08;
	
	/* Clear PCM mode bit, LRCLK is a 50% duty cycle clock */
	iface1A &= ~0x04;
	
	/* Clock inversion bits, BCI and WCI */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
		/* Normal clock, normal frame */
		case SND_SOC_DAIFMT_NB_NF:
			iface1A &= ~0x60;
			break;
		/* Inverted clock, inverted frame */
		case SND_SOC_DAIFMT_IB_IF:
			iface1A |= 0x60;
			break;
		/* Inverted clock, normal frame */
		case SND_SOC_DAIFMT_IB_NF:
			iface1A &= ~0x40;
			iface1A |= 0x20;
			break;
		/* Normal clock, inverted frame */
		case SND_SOC_DAIFMT_NB_IF:
			iface1A |= 0x40;
			iface1A &= ~0x20;
			break;
		default:
			return -EINVAL;
	}

	max9867_write(codec, MAX9867_IFC1A, iface1A);
	
	/*
		Gather the bits that make up Interface 1b
	*/
	
	/* Enable ADC Data Delay, required for I2S mode */
	iface1B |= 0x10;
	
	/* Disable mono mode */
	iface1B &= ~0x08;
	
	max9867_write(codec, MAX9867_IFC1B, iface1B);
	
	max9867_write(codec, MAX9867_MODECONFIG, 0x04);
	
	return 0;

}

/*******************************************************************************
 * updated
 * Power events
 ******************************************************************************/
static int max9867_dapm_event(struct snd_soc_codec *codec, int event)
{
	switch (event) {
		case SNDRV_CTL_POWER_D0: /* full On */
			max9867_write(codec, MAX9867_PWRMAN, 0x6f);
			break;

		case SNDRV_CTL_POWER_D1: /* partial On */
		break;

		case SNDRV_CTL_POWER_D2: /* partial On */
			break;

		case SNDRV_CTL_POWER_D3hot: /* Off, with power */
			max9867_write(codec, MAX9867_PWRMAN, 0x00);
			break;

		case SNDRV_CTL_POWER_D3cold: /* Off, without power */
			max9867_write(codec, MAX9867_PWRMAN, 0x00);
			break;
	}
//	codec->dapm_state = event;
	
	return 0;
}

#define MAX9867_RATES SNDRV_PCM_RATE_8000_48000
#define MAX9867_FORMATS (SNDRV_PCM_FMTBIT_S16_LE)

static const struct snd_soc_dai_ops max9867_dai_hifi_ops = {
	.shutdown	= max9867_shutdown,
	.hw_params	= max9867_hw_params,
};

//struct snd_soc_codec_dai max9867_dai = {
struct snd_soc_dai_driver max9867_dai = {
//	.name = "MAX9867",
	.name = "max9867-hifi",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 2,
		.rates = MAX9867_RATES,
		.formats = MAX9867_FORMATS,},
	.capture = {
		.stream_name = "Capture",
		.channels_min = 1,
		.channels_max = 2,
		.rates = MAX9867_RATES,
		.formats = MAX9867_FORMATS,},
	.ops = &max9867_dai_hifi_ops,
//	.dai_ops = {
//		.digital_mute = max9867_mute,
//		.set_sysclk = max9867_set_dai_sysclk,
//		.set_fmt = max9867_set_dai_fmt,
//	}
};
//EXPORT_SYMBOL_GPL(max9867_dai);



/*******************************************************************************
 * updated
 * initialise the max9867 driver
 * register the mixer and dsp interfaces with the kernel
 ******************************************************************************/
static int max9867_init(struct snd_soc_device *socdev)
{
//	struct snd_soc_codec *codec = socdev->codec;
	int ret = 0;

//	codec->name = "MAX9867";
//	codec->owner = THIS_MODULE;
//	codec->read = max9867_read;
//	codec->write = max9867_write;
//	codec->dapm_event = max9867_dapm_event;
//	codec->dai = &max9867_dai;
//	codec->num_dai = 1;
	
	/* register pcms */
//	ret = snd_soc_new_pcms(socdev, SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1);
//	if (ret < 0) {
//		err("max9867: failed to create pcms");
//		goto pcm_err;
//	}

	/* power on device */
//	max9867_dapm_event(codec, SNDRV_CTL_POWER_D3hot);
//	max9867_add_widgets(codec);
//	max9867_add_controls(codec);

	ret = snd_soc_register_card(socdev);
	if (ret < 0) {
		err("max9867: failed to register card");
		goto card_err;
	}

	return ret;

card_err:
//	snd_soc_free_pcms(socdev);
//	snd_soc_dapm_free(socdev);
pcm_err:
	
	return ret;
}


static struct snd_soc_device *max9867_socdev;

#if defined (CONFIG_I2C) || defined (CONFIG_I2C_MODULE)

static unsigned short normal_i2c[] = {0, I2C_CLIENT_END};

/* Magic definition of all other variables and things */
I2C_CLIENT_INSMOD;

static struct i2c_driver max9867_i2c_driver;
static struct i2c_client client_template;

/*******************************************************************************
 * If the i2c layer weren't so broken, we could pass this kind of data
 * around 
 ******************************************************************************/
static int max9867_codec_probe(struct i2c_adapter *adap, int addr, int kind)
{
	struct snd_soc_device *socdev = max9867_socdev;
//	struct max9867_setup_data *setup = socdev->codec_data;
//	struct snd_soc_codec *codec = socdev->codec;
//	struct i2c_client *i2c;
	int ret;

//	if (addr != setup->i2c_address)
//		return -ENODEV;

//	client_template.adapter = adap;
//	client_template.addr = addr;

//	i2c = kmemdup(&client_template, sizeof(client_template), GFP_KERNEL);
//	if (i2c == NULL) {
//		kfree(codec);
//		return -ENOMEM;
//	}
//	i2c_set_clientdata(i2c, codec);
//	codec->control_data = i2c;

//	ret = i2c_attach_client(i2c);
//	if (ret < 0) {
//		err("failed to attach codec at addr %x", addr);
//		goto err;
//	}

	ret = max9867_init(socdev);
	if (ret < 0) {
		err("failed to initialise MAX9867");
		goto err;
	}
	return ret;

err:
//	kfree(codec);
//	kfree(i2c);
	return ret;
}

#if 0
/*******************************************************************************
 * 
 ******************************************************************************/
static int max9867_i2c_detach(struct i2c_client *client)
{
	struct snd_soc_codec* codec = i2c_get_clientdata(client);
	
//	i2c_detach_client(client);
//	kfree(codec->reg_cache);
//	kfree(client);
	return 0;
}

/*******************************************************************************
 * 
 ******************************************************************************/
static int max9867_i2c_attach(struct i2c_adapter *adap)
{
//	return i2c_probe(adap, &addr_data, max9867_codec_probe);
	return 0;
}

/* gumstix i2c codec control layer */
static struct i2c_driver max9867_i2c_driver = {
	.driver = {
		.name = "MAX9867 I2C CODEC",
		.owner = THIS_MODULE,
	},
//	.id =             I2C_DRIVERID_MAX9867,
	.attach_adapter = max9867_i2c_attach,
//	.detach_client =  max9867_i2c_detach,
//	.command =        NULL,
};

static struct i2c_client client_template = {
	.name =   "MAX9867",
	.driver = &max9867_i2c_driver,
};
#endif

#endif

/*******************************************************************************
 * Make sure that a max9867 is attached to the I2C bus.
 ******************************************************************************/
//static int max9867_probe(struct platform_device *pdev)
static int max9867_probe(struct snd_soc_codec *codec)
{
#if 0
	return 0;
#else
//	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
//	struct max9867_setup_data *setup;
//	struct snd_soc_codec *codec;
	struct max9867_priv *max9867;
	int ret = 0;
	
	info("MAX9867 Audio CODEC %s", MAX9867_DRIVER_VERSION);
	
	//setup = socdev->codec_data;
	
//	codec = kzalloc(sizeof(struct snd_soc_codec), GFP_KERNEL);
//	if (codec == NULL)
//		return -ENOMEM;

	//max9867 = kzalloc(sizeof(struct max9867_priv), GFP_KERNEL);
	//if (max9867 == NULL) {
	//	kfree(codec);
	//	return -ENOMEM;
	//}

	//codec->private_data = max9867;
//	socdev->codec = codec;
//	mutex_init(&codec->mutex);
//	INIT_LIST_HEAD(&codec->dapm_widgets);
//	INIT_LIST_HEAD(&codec->dapm_paths);

//	max9867_socdev = socdev;
#if defined (CONFIG_I2C) || defined (CONFIG_I2C_MODULE)
//	if (setup->i2c_address) {
//		normal_i2c[0] = setup->i2c_address;
		codec->hw_write = (hw_write_t)i2c_master_send;
//		codec->hw_read = (hw_read_t)i2c_master_recv;
//		ret = i2c_add_driver(&max9867_i2c_driver);
//		if (ret != 0)
//			printk(KERN_ERR "can't add i2c driver");
//	}
#else
	/* Add other interfaces here */
#endif

	return ret;
#endif
}

/*******************************************************************************
 * The driver is being unloaded, power down the codec and
 * free aallocated resources.
 ******************************************************************************/
static int max9867_remove(struct platform_device *pdev)
{
#if 0
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_codec *codec = socdev->codec;
	
	if (codec->control_data)
		max9867_dapm_event(codec, SNDRV_CTL_POWER_D3cold);

	snd_soc_free_pcms(socdev);
	snd_soc_dapm_free(socdev);
	
#if defined (CONFIG_I2C) || defined (CONFIG_I2C_MODULE)
	i2c_del_driver(&max9867_i2c_driver);
#endif
	
//	kfree(codec->private_data);
//	kfree(codec);
#endif	
	return 0;
}

/*******************************************************************************
 * updated
 ******************************************************************************/
static int max9867_suspend(struct snd_soc_codec *codec)
{
	max9867_dapm_event(codec, SNDRV_CTL_POWER_D3cold);

	return 0;
}

/*******************************************************************************
 * updated
 * If power is removed from the codec this function would
 * need to restore the codec registers to their state
 * before the power was removed. The suspend function would

 * be responsible for saving the register state.
 ******************************************************************************/
static int max9867_resume(struct snd_soc_codec *codec)
{
	max9867_dapm_event(codec, SNDRV_CTL_POWER_D3hot);
//	max9867_dapm_event(codec, codec->suspend_dapm_state);

	return 0;	
}

//struct snd_soc_codec_device soc_codec_dev_max9867 = {
struct snd_soc_codec_driver soc_codec_dev_max9867 = {
	.probe   = max9867_probe,
	.remove  = max9867_remove,
	.suspend = max9867_suspend,
	.resume  = max9867_resume,

	.controls = max9867_snd_controls,
	.num_controls = ARRAY_SIZE(max9867_snd_controls),
};

//EXPORT_SYMBOL_GPL(soc_codec_dev_max9867);


static int max9867_i2c_probe(struct i2c_client *i2c,
			     const struct i2c_device_id *id)
{
	struct max9867_priv *max9867;
	int ret;

	max9867 = devm_kzalloc(&i2c->dev, sizeof(struct max9867_priv),
			       GFP_KERNEL);
	if (max9867 == NULL)
		return -ENOMEM;

	i2c_set_clientdata(i2c, max9867);

	ret = snd_soc_register_codec(&i2c->dev,
			&soc_codec_dev_max9867, &max9867_dai, 1);
	return ret;
}

static int max9867_i2c_remove(struct i2c_client *client)
{
	snd_soc_unregister_codec(&client->dev);
	return 0;
}

static const struct i2c_device_id max9867_i2c_id[] = {
	{ "max9867", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, max9867_i2c_id);

static struct i2c_driver max9867_i2c_driver = {
	.driver = {
		.name = "max9867",
		.owner = THIS_MODULE,
	},
	.probe = max9867_i2c_probe,
	.remove = max9867_i2c_remove,
	.id_table = max9867_i2c_id,
};

module_i2c_driver(max9867_i2c_driver);

MODULE_DESCRIPTION("ALSA SoC MAX9867 driver");
MODULE_AUTHOR("Maxim Integrated Products");
MODULE_LICENSE("GPL");
