
package jp.riken.brain.ni.samuraigraph.figure;

import java.awt.Font;
import java.util.ArrayList;

import jp.riken.brain.ni.samuraigraph.base.SGDrawingElement;
import jp.riken.brain.ni.samuraigraph.base.SGIDrawingElementConstants;
import jp.riken.brain.ni.samuraigraph.base.SGProperties;
import jp.riken.brain.ni.samuraigraph.base.SGTuple2f;


/**
 * A super class for the drawing element with text.
 */
public abstract class SGDrawingElementString extends SGDrawingElement
	implements SGIStringConstants, SGIDrawingElementConstants
{

	
	/**
	 * Text for this string element.
	 */
	protected String mString = new String();


	/**
	 * Font used to draw string actually.
	 * This object is updated with the primary information source:
	 * font size, font style, font name and the magnification.
	 */
	protected Font mFont;


	/**
	 * Font name.
	 */
	protected String mFontName;


	/**
	 * Font size in the default zoom.
	 */
	protected float mFontSize;


	/**
	 * Font style.
	 */
	protected int mFontStyle;


	/**
	 * X-coordinate of the string.
	 */
	protected float mX;


	/**
	 * Y-coordinate of the string.
	 */
	protected float mY;


	/**
	 * Angled of this string.
	 */
	protected float mAngle;


	/**
	 * Default constructor.
	 */
	public SGDrawingElementString()
	{
		super();
	}


	/**
	 * Construct a string element with given text.
	 */
	public SGDrawingElementString( final String str )
	{
		super();
		mString = new String( str );
	}


	/**
	 * Construct a string element with given string element.
	 */
	public SGDrawingElementString(
		final SGDrawingElementString element )
	{
		super();
		this.mMagnification = element.mMagnification;
		this.mString = new String( element.mString );
		this.setFont( element.getFontName(), element.getFontStyle(), element.getFontSize() );
		this.mX = element.mX;
		this.mY = element.mY;
		this.mAngle = element.mAngle;
		this.mColorList = new ArrayList( element.mColorList );
	}


	/**
	 * Construct a string element with given text and font information.
	 */
	public SGDrawingElementString(
		final String str,
		final String fontName,
		final int fontStyle,
		final float fontSize )
	{
		super();
		this.mString = new String( str );
		this.setFont( fontName, fontStyle, fontSize );
	}
	

	/**
	 * Construct a string element with given text, font information and the location.
	 */
	public SGDrawingElementString(
		final String str,
		final String fontName,
		final int fontStyle,
		final float fontSize,
		final float x,
		final float y )
	{
		super();
		this.mString = new String( str );
		this.setFont( fontName, fontStyle, fontSize );
		this.mX = x;
		this.mY = y;
	}



	/**
	 * Returns the text of this string element.
	 */
	public String toString()
	{
		return this.getString();
	}



	/**
	 * Set the text.
	 * @param str - text
	 */
	public boolean setString( final String str )
	{
		mString = new String( str );
		return true;
	}


	/**
	 * Set the magnification.
	 * @param mag - magnification
	 */
	public boolean setMagnification( final float mag )
	{
		super.setMagnification( mag );

		// update the attribute
		this.updateFont();

		return true;
	}


	/**
	 * Set the font information.
	 * @param f - font to be set
	 */
	public void setFont( final Font f )
	{
		this.setFont( f.getFontName(), f.getStyle(), f.getSize() );
	}


	/**
	 * Set the font name.
	 * @param name - font name
	 */
	public void setFontName( final String name )
	{
		this.setFont( name, this.mFontStyle, this.mFontSize );
	}


	/**
	 * Set the font style.
	 * @param style - font style
	 */
	public void setFontStyle( final int style )
	{
		this.setFont( this.mFontName, style, this.mFontSize );
	}


	/**
	 * Set the font size in the default zoom
	 * @param size - font size
	 */
	public void setFontSize( final float size )
	{
		this.setFont( this.mFontName, this.mFontStyle, size );
	}


	/**
	 * Set the font information.
	 * @param name - font name
	 * @param style - font style
	 * @param size - font size in the default zoom
	 * @return
	 */
	public boolean setFont( final String name, final int style, final float size )
	{
		this.mFontName = name;
		this.mFontStyle = style;
		this.mFontSize = size;

		// update the attribute
		this.updateFont();

		return true;
	}



	/**
	 * Update the font object in attributes.
	 */
	private boolean updateFont()
	{
		this.mFont = new Font(
			this.mFontName,
			this.mFontStyle,
			(int)(this.mMagnification*this.mFontSize)
		);
		return true;
	}




	/**
	 * Set the location of this string.
	 * @param pos - location
	 * @return
	 */
	public boolean setLocation( final SGTuple2f pos )
	{
		if( pos==null )
		{
			throw new IllegalArgumentException("pos==null");
		}
		this.setLocation( pos.x, pos.y );
		return true;
	}



	/**
	 * Set the location of this string.
	 * @param x - X-coordinate
	 * @param y - Y-coordinate
	 * @return
	 */
	public boolean setLocation( final float x, final float y )
	{
		this.mX = x;
		this.mY = y;
		return true;
	}



	/**
	 * Set the x-coordinate of this string.
	 * @param x - X-coordinate
	 * @return
	 */
	public boolean setX( final float x )
	{
		this.setLocation( x, this.getY() );
		return true;
	}


	/**
	 * Set the y-coordinate of this string.
	 * @param y - Y-coordinate
	 * @return
	 */
	public boolean setY( final float y )
	{
		this.setLocation( this.getX(), y );
		return true;
	}


	/**
	 * Set the angle of this string.
	 * @param angle - angle to be set
	 * @return
	 */
	public boolean setAngle( final float angle )
	{
		mAngle = angle;
		return true;
	}



	/**
	 * Returns a string object.
	 * @return sring object
	 */
	public String getString()
	{
		return new String( mString );
	}



	/**
	 * Returns the X-coordinate.
	 * @return X-coordinate
	 */
	public float getX()
	{
		return this.getLocation().x;
	}


	/**
	 * Returns the Y-coordinate.
	 * @return Y-coordinate
	 */
	public float getY()
	{
		return this.getLocation().y;
	}



	/**
	 * Returns the location.
	 * @return location
	 */
	public SGTuple2f getLocation()
	{
		return new SGTuple2f( this.mX, this.mY );
	}



	/**
	 * Returns the font name.
	 * @return font name
	 */
	public String getFontName()
	{
		return this.mFontName;
	}


	/**
	 * Returns the font size in the default zoom.
	 * @return font size in the default zoom
	 */
	public float getFontSize()
	{
		return this.mFontSize;
	}


	/**
	 * Returns the font style.
	 * @return font style
	 */
	public int getFontStyle()
	{
		return this.mFontStyle;
	}


	/**
	 * Returns the angle of this string.
	 * @return angle of this string
	 */
	public float getAngle()
	{
		return this.mAngle;
	}



	/**
	 * 
	 */
	public boolean setProperty( final SGDrawingElement element )
	{
		if( !(element instanceof SGDrawingElementString) )
		{
			return false;
		}

		if( super.setProperty( element ) == false )
		{
			return false;
		}

		SGDrawingElementString el = (SGDrawingElementString)element;

		this.setX( el.getX() );
		this.setY( el.getY() );
		this.setString( el.getString() );
		this.setFont( el.getFontName(), el.getFontStyle(), el.getFontSize() );
		this.setAngle( el.getAngle() );

		return true;
	}



	/**
	 * 
	 */
	public SGProperties getProperties()
	{
		StringProperties p = new StringProperties();
		this.getProperties(p);
		return p;
	}


	/**
	 * 
	 */
	public boolean getProperties( SGProperties p )
	{
		if( p==null ) return false;
		if( ( p instanceof StringProperties ) == false ) return false;

		super.getProperties(p);

		StringProperties sp = (StringProperties)p;
		sp.setFontName( this.getFontName() );
		sp.setFontSize( this.getFontSize() );
		sp.setFontStyle( this.getFontStyle() );
		sp.setAngle( this.getAngle() );

		return true;
	}



	/**
	 * 
	 */
	public boolean setProperties( SGProperties p )
	{
		if( ( p instanceof StringProperties ) == false ) return false;

		if( super.setProperties(p) == false ) return false;

		StringProperties ep = (StringProperties)p;

		this.setFont(
			ep.getFontName(),
			ep.getFontStyle().intValue(),
			ep.getFontSize().floatValue() );
		this.setAngle( ep.getAngle().floatValue() );

		return true;

	}


	/**
	 * Translate this string element.
	 * @param dx - Displacement to the X-axis direction
	 * @param dy - Displacement to the Y-axis direction
	 */
	public void translate( final float dx, final float dy )
	{
		this.setLocation( this.getX() + dx, this.getY() + dy );
	}





	/**
	 * 
	 */
	public static class StringProperties extends DrawingElementProperties
	{

		private float mFontSize = 0.0f;
		private int mFontStyle = -1;
		private String mFontName = null;
		private float mAngle = 0.0f;


		/**
		 * 
		 *
		 */
		public StringProperties()
		{
			super();
		}


		/**
		 * 
		 */
		public boolean equals( final Object obj )
		{
			if( ( obj instanceof StringProperties ) == false )
			{
				return false;
			}

			if( super.equals(obj) == false ) return false;

			StringProperties p = (StringProperties)obj;

			if( this.mFontSize!=p.mFontSize ) return false;
			if( this.mFontStyle!=p.mFontStyle ) return false;
			if( this.mFontName.equals(p.mFontName) == false ) return false;
			if( this.mAngle!=p.mAngle ) return false;

			return true;
		}


		public String getFontName()
		{
			return this.mFontName;
		}
		
		public Float getFontSize()
		{
			return new Float( this.mFontSize );
		}

		public Integer getFontStyle()
		{
			return new Integer( this.mFontStyle );
		}

		public Float getAngle()
		{
			return new Float( this.mAngle );
		}


		public boolean setFontName( final String name )
		{
			this.mFontName = name;
			return true;
		}

		public boolean setFontSize( final float size )
		{
			this.mFontSize = size;
			return true;
		}

		public boolean setFontStyle( final int style )
		{
			this.mFontStyle = style;
			return true;
		}

		public boolean setAngle( final float angle )
		{
			this.mAngle = angle;
			return true;
		}

	
	}


}

